/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static herr_t H5O__cache_get_initial_load_size(void *udata, size_t *image_len);
static herr_t H5O__cache_get_final_load_size(const void *image_ptr, size_t image_len, void *udata,
                                             size_t *actual_len);
static htri_t H5O__cache_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5O__cache_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5O__cache_image_len(const void *thing, size_t *image_len);
static herr_t H5O__cache_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5O__cache_notify(H5AC_notify_action_t action, void *_thing);
static herr_t H5O__cache_free_icr(void *thing);

static herr_t H5O__cache_chk_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5O__cache_chk_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5O__cache_chk_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5O__cache_chk_image_len(const void *thing, size_t *image_len);
static herr_t H5O__cache_chk_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5O__cache_chk_notify(H5AC_notify_action_t action, void *_thing);
static herr_t H5O__cache_chk_free_icr(void *thing);

static herr_t H5O__prefix_deserialize(const uint8_t *image, size_t len, H5O_cache_ud_t *udata);

static herr_t H5O__chunk_deserialize(H5O_t *oh, haddr_t addr, size_t chunk_size, const uint8_t *image,
                                     size_t len, H5O_common_cache_ud_t *udata, bool *dirty);
static herr_t H5O__chunk_serialize(const H5F_t *f, H5O_t *oh, unsigned chunkno);

static herr_t H5O__add_cont_msg(H5O_cont_msgs_t *cont_msg_info, const H5O_cont_t *cont);

const H5AC_class_t H5AC_OHDR[1] = {{
    H5AC_OHDR_ID,                      
    "object header",                   
    H5FD_MEM_OHDR,                     
    H5AC__CLASS_SPECULATIVE_LOAD_FLAG, 
    H5O__cache_get_initial_load_size,  
    H5O__cache_get_final_load_size,    
    H5O__cache_verify_chksum,          
    H5O__cache_deserialize,            
    H5O__cache_image_len,              
    NULL,                              
    H5O__cache_serialize,              
    H5O__cache_notify,                 
    H5O__cache_free_icr,               
    NULL,                              
}};

const H5AC_class_t H5AC_OHDR_CHK[1] = {{
    H5AC_OHDR_CHK_ID,                     
    "object header continuation chunk",   
    H5FD_MEM_OHDR,                        
    H5AC__CLASS_NO_FLAGS_SET,             
    H5O__cache_chk_get_initial_load_size, 
    NULL,                                 
    H5O__cache_chk_verify_chksum,         
    H5O__cache_chk_deserialize,           
    H5O__cache_chk_image_len,             
    NULL,                                 
    H5O__cache_chk_serialize,             
    H5O__cache_chk_notify,                
    H5O__cache_chk_free_icr,              
    NULL,                                 
}};

H5FL_EXTERN(H5O_unknown_t);

H5FL_EXTERN(H5O_chunk_proxy_t);

H5FL_SEQ_DEFINE(H5O_cont_t);

static herr_t
H5O__cache_get_initial_load_size(void H5_ATTR_UNUSED *_udata, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(image_len);

    
    *image_len = H5O_SPEC_READ_SIZE;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__cache_get_final_load_size(const void *image, size_t image_len, void *_udata, size_t *actual_len)
{
    H5O_cache_ud_t *udata     = (H5O_cache_ud_t *)_udata; 
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);
    assert(actual_len);
    assert(*actual_len == image_len);

    
    if (H5O__prefix_deserialize((const uint8_t *)image, image_len, udata) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't deserialize object header prefix");

    
    assert(udata->oh);

    
    *actual_len = udata->chunk0_size + (size_t)H5O_SIZEOF_HDR(udata->oh);

    
    udata->oh_version = udata->oh->version;

    
    if (H5O__free(udata->oh, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't destroy object header");
    udata->oh = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__cache_verify_chksum(const void *_image, size_t len, void *_udata)
{
    const uint8_t  *image     = (const uint8_t *)_image;  
    H5O_cache_ud_t *udata     = (H5O_cache_ud_t *)_udata; 
    htri_t          ret_value = true;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);

    
    if (udata->oh_version != H5O_VERSION_1) {
        uint32_t stored_chksum;   
        uint32_t computed_chksum; 

        
        if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get checksums");

        if (stored_chksum != computed_chksum)
            ret_value = false;
    }
    else
        assert(!(udata->common.file_intent & H5F_ACC_SWMR_WRITE));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__cache_deserialize(const void *image, size_t len, void *_udata, bool *dirty)
{
    H5O_t          *oh        = NULL;                     
    H5O_cache_ud_t *udata     = (H5O_cache_ud_t *)_udata; 
    void           *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(len > 0);
    assert(udata);
    assert(udata->common.f);
    assert(udata->common.cont_msg_info);
    assert(dirty);
    assert(udata->oh == NULL);

    if (H5O__prefix_deserialize((const uint8_t *)image, len, udata) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, NULL, "can't deserialize object header prefix");
    assert(udata->oh);

    oh = udata->oh;

    
    oh->swmr_write = !!(H5F_INTENT(udata->common.f) & H5F_ACC_SWMR_WRITE);

    
    if (oh->swmr_write) {
        
        if (NULL == (oh->proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCREATE, NULL, "can't create object header proxy");
    }
    else
        oh->proxy = NULL;

    
    if (H5O__chunk_deserialize(oh, udata->common.addr, udata->chunk0_size, (const uint8_t *)image, len,
                               &(udata->common), dirty) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "can't deserialize first object header chunk");

    
    if (oh->version == H5O_VERSION_1 && udata->v1_pfx_nmesgs < oh->nmesgs)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "bad object header message count");

    
    udata->made_attempt = true;

    
    ret_value = oh;

done:
    
    if (!ret_value && oh)
        if (H5O__free(oh, false) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, NULL, "unable to destroy object header data");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_image_len(const void *_thing, size_t *image_len)
{
    const H5O_t *oh = (const H5O_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(oh);
    assert(oh->cache_info.type == H5AC_OHDR);
    assert(image_len);

    
    *image_len = oh->chunk[0].size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__cache_serialize(const H5F_t *f, void *image, size_t len, void *_thing)
{
    H5O_t   *oh = (H5O_t *)_thing; 
    uint8_t *chunk_image;          
    herr_t   ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(image);
    assert(oh);
    assert(oh->cache_info.type == H5AC_OHDR);
    assert(oh->chunk[0].size == len);
#ifdef H5O_DEBUG
    H5O__assert(oh);
#endif 

    
    chunk_image = oh->chunk[0].image;

    
    if (oh->version > H5O_VERSION_1) {
        uint64_t chunk0_size; 

        assert(oh->chunk[0].size >= (size_t)H5O_SIZEOF_HDR(oh));
        chunk0_size = oh->chunk[0].size - (size_t)H5O_SIZEOF_HDR(oh);

        
        assert(!memcmp(chunk_image, H5O_HDR_MAGIC, H5_SIZEOF_MAGIC));
        chunk_image += H5_SIZEOF_MAGIC;

        
        *chunk_image++ = oh->version;

        
        *chunk_image++ = oh->flags;

        
        if (oh->flags & H5O_HDR_STORE_TIMES) {
            UINT32ENCODE(chunk_image, oh->atime);
            UINT32ENCODE(chunk_image, oh->mtime);
            UINT32ENCODE(chunk_image, oh->ctime);
            UINT32ENCODE(chunk_image, oh->btime);
        }

        
        if (oh->flags & H5O_HDR_ATTR_STORE_PHASE_CHANGE) {
            UINT16ENCODE(chunk_image, oh->max_compact);
            UINT16ENCODE(chunk_image, oh->min_dense);
        }

        
        switch (oh->flags & H5O_HDR_CHUNK0_SIZE) {
            case 0: 
                assert(chunk0_size < 256);
                *chunk_image++ = (uint8_t)chunk0_size;
                break;

            case 1: 
                assert(chunk0_size < 65536);
                UINT16ENCODE(chunk_image, chunk0_size);
                break;

            case 2: 
                
                assert(chunk0_size <= 4294967295UL);
                UINT32ENCODE(chunk_image, chunk0_size);
                break;

            case 3: 
                UINT64ENCODE(chunk_image, chunk0_size);
                break;

            default:
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad size for chunk 0");
        }
    }
    else {
        
        *chunk_image++ = oh->version;

        
        *chunk_image++ = 0;

        
#ifdef H5O_ENABLE_BAD_MESG_COUNT
        if (oh->store_bad_mesg_count)
            UINT16ENCODE(chunk_image, (oh->nmesgs - 1));
        else
#endif 
            UINT16ENCODE(chunk_image, oh->nmesgs);

        
        UINT32ENCODE(chunk_image, oh->nlink);

        
        UINT32ENCODE(chunk_image, (oh->chunk[0].size - (size_t)H5O_SIZEOF_HDR(oh)));

        
        memset(chunk_image, 0, (size_t)(H5O_SIZEOF_HDR(oh) - 12));
        chunk_image += (size_t)(H5O_SIZEOF_HDR(oh) - 12);
    }

    assert((size_t)(chunk_image - oh->chunk[0].image) ==
           (size_t)(H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh)));

    
    if (H5O__chunk_serialize(f, oh, (unsigned)0) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSERIALIZE, FAIL, "unable to serialize first object header chunk");

    
    H5MM_memcpy(image, oh->chunk[0].image, len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_notify(H5AC_notify_action_t action, void *_thing)
{
    H5O_t *oh        = (H5O_t *)_thing;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(oh);

    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            if (oh->swmr_write) {
                
                assert(oh->proxy);

                
                if (H5AC_proxy_entry_add_parent(oh->proxy, oh) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't add object header as parent of proxy");
            }
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED: {
            unsigned u; 

            
            for (u = 0; u < oh->nmesgs; u++)
                if (oh->mesg[u].chunkno == 0)
                    oh->mesg[u].dirty = false;
#ifndef NDEBUG
            
            oh->ndecode_dirtied = 0;
#endif
        } break;

        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            if (oh->swmr_write) {
                
                if (H5AC_proxy_entry_remove_parent(oh->proxy, oh) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't remove object header as parent of proxy");
            }
            break;

        default:
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_free_icr(void *_thing)
{
    H5O_t *oh        = (H5O_t *)_thing; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(oh);
    assert(oh->cache_info.type == H5AC_OHDR);

    
    if (H5O__free(oh, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't destroy object header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_chk_get_initial_load_size(void *_udata, size_t *image_len)
{
    const H5O_chk_cache_ud_t *udata = (const H5O_chk_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(udata);
    assert(udata->oh);
    assert(image_len);
    assert(udata->size);

    
    *image_len = udata->size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5O__cache_chk_verify_chksum(const void *_image, size_t len, void *_udata)
{
    const uint8_t      *image     = (const uint8_t *)_image;      
    H5O_chk_cache_ud_t *udata     = (H5O_chk_cache_ud_t *)_udata; 
    htri_t              ret_value = true;

    FUNC_ENTER_PACKAGE

    assert(image);

    
    if (udata->oh->version != H5O_VERSION_1) {
        uint32_t stored_chksum;   
        uint32_t computed_chksum; 

        
        if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get checksums");

        if (stored_chksum != computed_chksum)
            ret_value = false;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__cache_chk_deserialize(const void *image, size_t len, void *_udata, bool *dirty)
{
    H5O_chunk_proxy_t  *chk_proxy = NULL;                         
    H5O_chk_cache_ud_t *udata     = (H5O_chk_cache_ud_t *)_udata; 
    void               *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(len > 0);
    assert(udata);
    assert(udata->oh);
    assert(dirty);

    
    if (NULL == (chk_proxy = H5FL_CALLOC(H5O_chunk_proxy_t)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    
    if (udata->decoding) {
        assert(udata->common.f);
        assert(udata->common.cont_msg_info);

        
        if (H5O__chunk_deserialize(udata->oh, udata->common.addr, udata->size, (const uint8_t *)image, len,
                                   &(udata->common), dirty) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, NULL, "can't deserialize object header chunk");

        
        H5_CHECKED_ASSIGN(chk_proxy->chunkno, unsigned, udata->oh->nchunks - 1, size_t);
    }
    else {
        
        assert(udata->chunkno < udata->oh->nchunks);

        
        chk_proxy->chunkno = udata->chunkno;

        
        assert(0 == memcmp(image, udata->oh->chunk[chk_proxy->chunkno].image,
                           udata->oh->chunk[chk_proxy->chunkno].size));
    }

    
    if (H5O__inc_rc(udata->oh) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINC, NULL, "can't increment reference count on object header");
    chk_proxy->oh = udata->oh;

    
    ret_value = chk_proxy;

done:
    if (NULL == ret_value)
        if (chk_proxy && H5O__chunk_dest(chk_proxy) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, NULL, "unable to destroy object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_chk_image_len(const void *_thing, size_t *image_len)
{
    const H5O_chunk_proxy_t *chk_proxy = (const H5O_chunk_proxy_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(chk_proxy);
    assert(chk_proxy->cache_info.type == H5AC_OHDR_CHK);
    assert(chk_proxy->oh);
    assert(image_len);

    *image_len = chk_proxy->oh->chunk[chk_proxy->chunkno].size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__cache_chk_serialize(const H5F_t *f, void *image, size_t len, void *_thing)
{
    H5O_chunk_proxy_t *chk_proxy = (H5O_chunk_proxy_t *)_thing; 
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(image);
    assert(chk_proxy);
    assert(chk_proxy->cache_info.type == H5AC_OHDR_CHK);
    assert(chk_proxy->oh);
    assert(chk_proxy->oh->chunk[chk_proxy->chunkno].size == len);

    
    if (H5O__chunk_serialize(f, chk_proxy->oh, chk_proxy->chunkno) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSERIALIZE, FAIL,
                    "unable to serialize object header continuation chunk");

    
    
    if (len > chk_proxy->oh->chunk[chk_proxy->chunkno].size)
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "buffer overflow detected");

    H5MM_memcpy(image, chk_proxy->oh->chunk[chk_proxy->chunkno].image, len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_chk_notify(H5AC_notify_action_t action, void *_thing)
{
    H5O_chunk_proxy_t *chk_proxy = (H5O_chunk_proxy_t *)_thing;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(chk_proxy);
    assert(chk_proxy->oh);

    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
            if (chk_proxy->oh->swmr_write) {
                
                if (chk_proxy->fd_parent) {
                    
                    assert(((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type);
                    assert((((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type->id == H5AC_OHDR_ID) ||
                           (((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type->id == H5AC_OHDR_CHK_ID));

                    
                    if (H5AC_create_flush_dependency(chk_proxy->fd_parent, chk_proxy) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
                }

                
                {
                    if (H5AC_create_flush_dependency(chk_proxy->oh, chk_proxy) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTDEPEND, FAIL, "unable to create flush dependency");
                }

                
                {
                    
                    assert(chk_proxy->oh->proxy);

                    
                    if (H5AC_proxy_entry_add_parent(chk_proxy->oh->proxy, chk_proxy) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL,
                                    "can't add object header chunk as parent of proxy");
                }
            }
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED: {
            unsigned u; 

            
            for (u = 0; u < chk_proxy->oh->nmesgs; u++)
                if (chk_proxy->oh->mesg[u].chunkno == chk_proxy->chunkno)
                    chk_proxy->oh->mesg[u].dirty = false;
        } break;

        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            if (chk_proxy->oh->swmr_write) {
                
                if (chk_proxy->fd_parent) {
                    
                    assert(((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type);
                    assert((((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type->id == H5AC_OHDR_ID) ||
                           (((H5C_cache_entry_t *)(chk_proxy->fd_parent))->type->id == H5AC_OHDR_CHK_ID));

                    if (H5AC_destroy_flush_dependency(chk_proxy->fd_parent, chk_proxy) < 0)
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
                    chk_proxy->fd_parent = NULL;
                }

                
                if (H5AC_destroy_flush_dependency(chk_proxy->oh, chk_proxy) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");

                
                if (H5AC_proxy_entry_remove_parent(chk_proxy->oh->proxy, chk_proxy) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL,
                                "can't remove object header chunk as parent of proxy");
            }
            break;

        default:
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__cache_chk_free_icr(void *_thing)
{
    H5O_chunk_proxy_t *chk_proxy = (H5O_chunk_proxy_t *)_thing; 
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(chk_proxy);
    assert(chk_proxy->cache_info.type == H5AC_OHDR_CHK);

    
    if (H5O__chunk_dest(chk_proxy) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "unable to destroy object header chunk proxy");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__add_cont_msg(H5O_cont_msgs_t *cont_msg_info, const H5O_cont_t *cont)
{
    size_t contno; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(cont_msg_info);
    assert(cont);

    
    if (cont_msg_info->nmsgs >= cont_msg_info->alloc_nmsgs) {
        size_t na = MAX(H5O_NCHUNKS, cont_msg_info->alloc_nmsgs * 2); 
        H5O_cont_t *x;

        if (NULL == (x = H5FL_SEQ_REALLOC(H5O_cont_t, cont_msg_info->msgs, na)))
            HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, FAIL, "memory allocation failed");
        cont_msg_info->alloc_nmsgs = na;
        cont_msg_info->msgs        = x;
    }

    
    contno                              = cont_msg_info->nmsgs++;
    cont_msg_info->msgs[contno].addr    = cont->addr;
    cont_msg_info->msgs[contno].size    = cont->size;
    cont_msg_info->msgs[contno].chunkno = cont->chunkno;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__prefix_deserialize(const uint8_t *_image, size_t len, H5O_cache_ud_t *udata)
{
    const uint8_t *image     = (const uint8_t *)_image; 
    const uint8_t *p_end     = image + len - 1;         
    H5O_t         *oh        = NULL;                    
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(udata);

    
    if (NULL == (oh = H5FL_CALLOC(H5O_t)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "memory allocation failed");

    
    oh->sizeof_size = H5F_SIZEOF_SIZE(udata->common.f);
    oh->sizeof_addr = H5F_SIZEOF_ADDR(udata->common.f);

    
    
    if (H5_IS_BUFFER_OVERFLOW(image, H5_SIZEOF_MAGIC, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    if (!memcmp(image, H5O_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC)) {

        
        image += H5_SIZEOF_MAGIC;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        oh->version = *image++;
        if (H5O_VERSION_2 != oh->version)
            HGOTO_ERROR(H5E_OHDR, H5E_VERSION, FAIL, "bad object header version number");

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        oh->flags = *image++;
        if (oh->flags & ~H5O_HDR_ALL_FLAGS)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "unknown object header status flag(s)");

        
        oh->nlink = 1;

        
        if (oh->flags & H5O_HDR_STORE_TIMES) {
            uint32_t tmp;

            if (H5_IS_BUFFER_OVERFLOW(image, 4 + 4 + 4 + 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");

            UINT32DECODE(image, tmp);
            oh->atime = (time_t)tmp;
            UINT32DECODE(image, tmp);
            oh->mtime = (time_t)tmp;
            UINT32DECODE(image, tmp);
            oh->ctime = (time_t)tmp;
            UINT32DECODE(image, tmp);
            oh->btime = (time_t)tmp;
        }
        else
            oh->atime = oh->mtime = oh->ctime = oh->btime = 0;

        
        if (oh->flags & H5O_HDR_ATTR_STORE_PHASE_CHANGE) {
            if (H5_IS_BUFFER_OVERFLOW(image, 2 + 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");

            UINT16DECODE(image, oh->max_compact);
            UINT16DECODE(image, oh->min_dense);
            if (oh->max_compact < oh->min_dense)
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad object header attribute phase change values");
        }
        else {
            oh->max_compact = H5O_CRT_ATTR_MAX_COMPACT_DEF;
            oh->min_dense   = H5O_CRT_ATTR_MIN_DENSE_DEF;
        }

        
        switch (oh->flags & H5O_HDR_CHUNK0_SIZE) {
            case 0: 
                if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                udata->chunk0_size = *image++;
                break;

            case 1: 
                if (H5_IS_BUFFER_OVERFLOW(image, 2, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                UINT16DECODE(image, udata->chunk0_size);
                break;

            case 2: 
                if (H5_IS_BUFFER_OVERFLOW(image, 4, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                UINT32DECODE(image, udata->chunk0_size);
                break;

            case 3: 
                if (H5_IS_BUFFER_OVERFLOW(image, 8, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                UINT64DECODE(image, udata->chunk0_size);
                break;

            default:
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad size for chunk 0");
        }
        if (udata->chunk0_size > 0 && udata->chunk0_size < H5O_SIZEOF_MSGHDR_OH(oh))
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad object header chunk size");
    }
    else {
        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        oh->version = *image++;
        if (H5O_VERSION_1 != oh->version)
            HGOTO_ERROR(H5E_OHDR, H5E_VERSION, FAIL, "bad object header version number");

        
        oh->flags = H5O_CRT_OHDR_FLAGS_DEF;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        image++;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT16DECODE(image, udata->v1_pfx_nmesgs);

        
        if (H5_IS_BUFFER_OVERFLOW(image, 4, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT32DECODE(image, oh->nlink);

        
        oh->atime = oh->mtime = oh->ctime = oh->btime = 0;

        
        oh->max_compact = 0;
        oh->min_dense   = 0;

        
        if (H5_IS_BUFFER_OVERFLOW(image, 4, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT32DECODE(image, udata->chunk0_size);
        if ((udata->v1_pfx_nmesgs > 0 && udata->chunk0_size < H5O_SIZEOF_MSGHDR_OH(oh)) ||
            (udata->v1_pfx_nmesgs == 0 && udata->chunk0_size > 0))
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad object header chunk size");

        
        if (H5_IS_BUFFER_OVERFLOW(image, 4, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        image += 4;
    }

    
    if ((size_t)(image - _image) != (size_t)(H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh)))
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "bad object header prefix length");

    
    udata->oh = oh;
    oh        = NULL;

done:
    
    if (ret_value < 0 && oh)
        if (H5O__free(oh, false) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "unable to destroy object header data");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__chunk_deserialize(H5O_t *oh, haddr_t addr, size_t chunk_size, const uint8_t *image, size_t len,
                       H5O_common_cache_ud_t *udata, bool *dirty)
{
    const uint8_t *chunk_image = NULL;   
    const uint8_t *p_end       = NULL;   
    uint8_t       *eom_ptr;              
    unsigned       merged_null_msgs = 0; 
    unsigned       chunkno;              
    unsigned       nullcnt;              
    bool           mesgs_modified =
        false; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(oh);
    assert(H5_addr_defined(addr));
    assert(image);
    assert(len);
    assert(udata->f);
    assert(udata->cont_msg_info);

    
    if (oh->nchunks >= oh->alloc_nchunks) {
        size_t       na = MAX(H5O_NCHUNKS, oh->alloc_nchunks * 2); 
        H5O_chunk_t *x;

        if (NULL == (x = H5FL_SEQ_REALLOC(H5O_chunk_t, oh->chunk, na)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "memory allocation failed");
        oh->alloc_nchunks = na;
        oh->chunk         = x;
    }

    
    chunkno                 = (unsigned)oh->nchunks++;
    oh->chunk[chunkno].gap  = 0;
    oh->chunk[chunkno].addr = addr;
    if (chunkno == 0)
        
        oh->chunk[0].size = chunk_size + (size_t)H5O_SIZEOF_HDR(oh);
    else
        oh->chunk[chunkno].size = chunk_size;
    if (NULL == (oh->chunk[chunkno].image = H5FL_BLK_MALLOC(chunk_image, oh->chunk[chunkno].size)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "memory allocation failed");
    oh->chunk[chunkno].chunk_proxy = NULL;

    
    if (len < oh->chunk[chunkno].size)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "attempted to copy too many disk image bytes into buffer");
    H5MM_memcpy(oh->chunk[chunkno].image, image, oh->chunk[chunkno].size);

    
    chunk_image = oh->chunk[chunkno].image;
    p_end       = chunk_image + oh->chunk[chunkno].size - 1;

    
    if (chunkno == 0) {
        size_t skip = (size_t)(H5O_SIZEOF_HDR(oh) - H5O_SIZEOF_CHKSUM_OH(oh));

        if (H5_IS_BUFFER_OVERFLOW(chunk_image, skip, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        chunk_image += skip;
    }

    
    else if (chunkno > 0 && oh->version > H5O_VERSION_1) {
        
        if (H5_IS_BUFFER_OVERFLOW(chunk_image, H5_SIZEOF_MAGIC, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        if (memcmp(chunk_image, H5O_CHK_MAGIC, H5_SIZEOF_MAGIC) != 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "wrong object header chunk signature");
        chunk_image += H5_SIZEOF_MAGIC;
    }

    
    eom_ptr = oh->chunk[chunkno].image + (oh->chunk[chunkno].size - H5O_SIZEOF_CHKSUM_OH(oh));
    nullcnt = 0;

    while (chunk_image < eom_ptr) {
        size_t            mesg_size;   
        unsigned          id;          
        uint8_t           flags;       
        H5O_msg_crt_idx_t crt_idx = 0; 

        

        
        if (oh->version == H5O_VERSION_1) {
            if (H5_IS_BUFFER_OVERFLOW(chunk_image, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT16DECODE(chunk_image, id);
        }
        else {
            if (H5_IS_BUFFER_OVERFLOW(chunk_image, 1, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            id = *chunk_image++;
        }

        
        if (H5_IS_BUFFER_OVERFLOW(chunk_image, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT16DECODE(chunk_image, mesg_size);
        if (mesg_size != H5O_ALIGN_OH(oh, mesg_size))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "message not aligned");

        if (H5_IS_BUFFER_OVERFLOW(chunk_image, mesg_size, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "message size exceeds buffer end");

        
        if (H5_IS_BUFFER_OVERFLOW(chunk_image, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        flags = *chunk_image++;
        if (flags & ~H5O_MSG_FLAG_BITS)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unknown flag for message");
        if ((flags & H5O_MSG_FLAG_SHARED) && (flags & H5O_MSG_FLAG_DONTSHARE))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "bad flag combination for message");
        if ((flags & H5O_MSG_FLAG_WAS_UNKNOWN) && (flags & H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "bad flag combination for message");
        if ((flags & H5O_MSG_FLAG_WAS_UNKNOWN) && !(flags & H5O_MSG_FLAG_MARK_IF_UNKNOWN))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "bad flag combination for message");
        

        
        if (oh->version == H5O_VERSION_1) {
            
            if (H5_IS_BUFFER_OVERFLOW(chunk_image, 3, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            chunk_image += 3;
        }
        else {
            
            if (oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED) {
                if (H5_IS_BUFFER_OVERFLOW(chunk_image, 2, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                UINT16DECODE(chunk_image, crt_idx);
            }
        }

        
        if (H5O_NULL_ID == id)
            nullcnt++;

        
        if ((udata->file_intent & H5F_ACC_RDWR) && H5O_NULL_ID == id && oh->nmesgs > 0 &&
            H5O_NULL_ID == oh->mesg[oh->nmesgs - 1].type->id && oh->mesg[oh->nmesgs - 1].chunkno == chunkno) {

            size_t mesgno; 

            
            mesgno = oh->nmesgs - 1;
            oh->mesg[mesgno].raw_size += (size_t)H5O_SIZEOF_MSGHDR_OH(oh) + mesg_size;
            oh->mesg[mesgno].dirty = true;
            merged_null_msgs++;
        }
        else {
            H5O_mesg_t *mesg;        
            unsigned    ioflags = 0; 

            
            if (oh->nmesgs >= oh->alloc_nmesgs)
                if (H5O__alloc_msgs(oh, (size_t)1) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "can't allocate more space for messages");

            
            mesg = &oh->mesg[oh->nmesgs];

            
            oh->nmesgs++;

            
            mesg->dirty   = false;
            mesg->flags   = flags;
            mesg->crt_idx = crt_idx;
            mesg->native  = NULL;
            H5_WARN_CAST_AWAY_CONST_OFF
            mesg->raw = (uint8_t *)chunk_image;
            H5_WARN_CAST_AWAY_CONST_ON
            mesg->raw_size = mesg_size;
            mesg->chunkno  = chunkno;

            
            
            if (id >= H5O_UNKNOWN_ID ||
#ifdef H5O_ENABLE_BOGUS
                id == H5O_BOGUS_VALID_ID ||
#endif
                NULL == H5O_msg_class_g[id]) {

                H5O_unknown_t *unknown; 

                
                if (NULL == (unknown = H5FL_MALLOC(H5O_unknown_t)))
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, FAIL, "memory allocation failed");

                
                *unknown = id;

                
                mesg->native = unknown;

                
                mesg->type = H5O_msg_class_g[H5O_UNKNOWN_ID];

                
                if (((udata->file_intent & H5F_ACC_RDWR) &&
                     (flags & H5O_MSG_FLAG_FAIL_IF_UNKNOWN_AND_OPEN_FOR_WRITE)) ||
                    (flags & H5O_MSG_FLAG_FAIL_IF_UNKNOWN_ALWAYS))
                    HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL,
                                "unknown message with 'fail if unknown' flag found");
                
                else if ((flags & H5O_MSG_FLAG_MARK_IF_UNKNOWN) && !(flags & H5O_MSG_FLAG_WAS_UNKNOWN) &&
                         (udata->file_intent & H5F_ACC_RDWR)) {

                    
                    
                    
                    mesg->flags |= H5O_MSG_FLAG_WAS_UNKNOWN;

                    
                    mesg->dirty    = true;
                    mesgs_modified = true;
                }
            }
            else {
                
                if (((flags & H5O_MSG_FLAG_SHARED) || (flags & H5O_MSG_FLAG_SHAREABLE)) &&
                    H5O_msg_class_g[id] && !(H5O_msg_class_g[id]->share_flags & H5O_SHARE_IS_SHARABLE))
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL,
                                "message of unshareable class flagged as shareable");

                
                mesg->type = H5O_msg_class_g[id];
            }

            
            

            
            if (H5O_CONT_ID == id) {
                H5O_cont_t *cont;

                
                if (NULL == (cont = (H5O_cont_t *)(H5O_MSG_CONT->decode)(udata->f, NULL, 0, &ioflags,
                                                                         mesg->raw_size, mesg->raw)))
                    HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "bad continuation message found");
                H5_CHECKED_ASSIGN(cont->chunkno, unsigned, udata->cont_msg_info->nmsgs + 1,
                                  size_t); 

                
                mesg->native = cont;

                
                if (H5O__add_cont_msg(udata->cont_msg_info, cont) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't add continuation message");
            }
            
            else if (H5O_REFCOUNT_ID == id) {
                H5O_refcount_t *refcount;

                
                if (oh->version <= H5O_VERSION_1)
                    HGOTO_ERROR(H5E_OHDR, H5E_VERSION, FAIL,
                                "object header version does not support reference count message");
                refcount = (H5O_refcount_t *)(H5O_MSG_REFCOUNT->decode)(udata->f, NULL, 0, &ioflags,
                                                                        mesg->raw_size, mesg->raw);

                
                mesg->native = refcount;

                
                oh->has_refcount_msg = true;
                if (!refcount)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't decode refcount");
                oh->nlink = *refcount;
            }
            
            else if (H5O_MTIME_ID == id) {
                time_t *mtime = NULL;

                
                mtime =
                    (time_t *)(H5O_MSG_MTIME->decode)(udata->f, NULL, 0, &ioflags, mesg->raw_size, mesg->raw);

                
                if (!mtime)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't decode old format mtime");

                
                mesg->native = mtime;
                oh->ctime    = *mtime;
            }
            
            else if (H5O_MTIME_NEW_ID == id) {
                time_t *mtime = NULL;

                
                mtime = (time_t *)(H5O_MSG_MTIME_NEW->decode)(udata->f, NULL, 0, &ioflags, mesg->raw_size,
                                                              mesg->raw);

                
                if (!mtime)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, FAIL, "can't decode new format mtime");

                
                mesg->native = mtime;
                oh->ctime    = *mtime;
            }
            
            else if (H5O_LINK_ID == id) {
                
                oh->link_msgs_seen++;
            }
            
            else if (H5O_ATTR_ID == id) {
                
                oh->attr_msgs_seen++;
            }

            
            if ((ioflags & H5O_DECODEIO_DIRTY) && (udata->file_intent & H5F_ACC_RDWR)) {
                mesg->dirty    = true;
                mesgs_modified = true;
            }
        }

        
        chunk_image += mesg_size;

        
        if ((eom_ptr - chunk_image) > 0 && (eom_ptr - chunk_image) < H5O_SIZEOF_MSGHDR_OH(oh)) {
            
            if (oh->version == H5O_VERSION_1)
                HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "gap found in early version of file format");

            
            if (nullcnt != 0)
                HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "gap in chunk with no null messages");

            
            oh->chunk[chunkno].gap = (size_t)(eom_ptr - chunk_image);

            
            chunk_image += oh->chunk[chunkno].gap;
        }
    }

    
    if (oh->version > H5O_VERSION_1) {
        uint32_t stored_chksum; 

        

        
        if (H5_IS_BUFFER_OVERFLOW(chunk_image, 4, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
        UINT32DECODE(chunk_image, stored_chksum);
    }

    
    if (chunk_image != oh->chunk[chunkno].image + oh->chunk[chunkno].size)
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "object header image size mismatch");

    
    if (mesgs_modified)
        *dirty = true;

    
    if (merged_null_msgs > 0) {
        udata->merged_null_msgs += merged_null_msgs;
        *dirty = true;
    }

done:
    if (ret_value < 0 && udata->cont_msg_info->msgs) {
        udata->cont_msg_info->msgs        = H5FL_SEQ_FREE(H5O_cont_t, udata->cont_msg_info->msgs);
        udata->cont_msg_info->alloc_nmsgs = 0;
    }
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__chunk_serialize(const H5F_t *f, H5O_t *oh, unsigned chunkno)
{
    H5O_mesg_t *curr_msg; 
    unsigned    u;        
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(oh);

    
    for (u = 0, curr_msg = &oh->mesg[0]; u < oh->nmesgs; u++, curr_msg++)
        if (curr_msg->dirty && curr_msg->chunkno == chunkno) {
            H5_WARN_CAST_AWAY_CONST_OFF
            if (H5O_msg_flush((H5F_t *)f, oh, curr_msg) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, FAIL, "unable to encode object header message");
            H5_WARN_CAST_AWAY_CONST_ON
        }

    
    if (oh->version > H5O_VERSION_1)
        
        assert(!memcmp(oh->chunk[chunkno].image, (chunkno == 0 ? H5O_HDR_MAGIC : H5O_CHK_MAGIC),
                       H5_SIZEOF_MAGIC));
    else
        
        assert(oh->chunk[chunkno].gap == 0);

    
    if (oh->version > H5O_VERSION_1) {
        uint32_t metadata_chksum; 
        uint8_t *chunk_image;     

        
        if (oh->chunk[chunkno].gap)
            memset((oh->chunk[chunkno].image + oh->chunk[chunkno].size) -
                       (H5O_SIZEOF_CHKSUM + oh->chunk[chunkno].gap),
                   0, oh->chunk[chunkno].gap);

        
        metadata_chksum =
            H5_checksum_metadata(oh->chunk[chunkno].image, (oh->chunk[chunkno].size - H5O_SIZEOF_CHKSUM), 0);

        
        chunk_image = oh->chunk[chunkno].image + (oh->chunk[chunkno].size - H5O_SIZEOF_CHKSUM);
        UINT32ENCODE(chunk_image, metadata_chksum);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
