/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5B2module.h" 

#include "H5private.h"   
#include "H5B2pkg.h"     
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

#define H5B2_NUM_LEAF_REC(n, r) (((n)-H5B2_LEAF_PREFIX_SIZE) / (r))

H5FL_DEFINE_STATIC(H5B2_hdr_t);

H5FL_BLK_DEFINE_STATIC(node_page);

H5FL_SEQ_DEFINE_STATIC(size_t);

H5FL_SEQ_DEFINE(H5B2_node_info_t);

herr_t
H5B2__hdr_init(H5B2_hdr_t *hdr, const H5B2_create_t *cparam, void *ctx_udata, uint16_t depth)
{
    size_t   sz_max_nrec;         
    unsigned u_max_nrec_size;     
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(cparam);
    assert(cparam->cls);
    assert((cparam->cls->crt_context && cparam->cls->dst_context) ||
           (NULL == cparam->cls->crt_context && NULL == cparam->cls->dst_context));
    assert(cparam->node_size > 0);
    assert(cparam->rrec_size > 0);
    assert(cparam->merge_percent > 0 && cparam->merge_percent <= 100);
    assert(cparam->split_percent > 0 && cparam->split_percent <= 100);
    assert(cparam->merge_percent < (cparam->split_percent / 2));

    
    hdr->depth = depth;

    
    hdr->split_percent = cparam->split_percent;
    hdr->merge_percent = cparam->merge_percent;
    hdr->node_size     = cparam->node_size;
    hdr->rrec_size     = cparam->rrec_size;

    
    hdr->cls = cparam->cls;

    
    if (NULL == (hdr->page = H5FL_BLK_MALLOC(node_page, hdr->node_size)))
        HGOTO_ERROR(H5E_BTREE, H5E_NOSPACE, FAIL, "memory allocation failed");
    memset(hdr->page, 0, hdr->node_size);

    
    if (NULL == (hdr->node_info = H5FL_SEQ_MALLOC(H5B2_node_info_t, (size_t)(hdr->depth + 1))))
        HGOTO_ERROR(H5E_BTREE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    sz_max_nrec = H5B2_NUM_LEAF_REC(hdr->node_size, hdr->rrec_size);
    H5_CHECKED_ASSIGN(hdr->node_info[0].max_nrec, unsigned, sz_max_nrec, size_t);
    hdr->node_info[0].split_nrec        = (hdr->node_info[0].max_nrec * hdr->split_percent) / 100;
    hdr->node_info[0].merge_nrec        = (hdr->node_info[0].max_nrec * hdr->merge_percent) / 100;
    hdr->node_info[0].cum_max_nrec      = hdr->node_info[0].max_nrec;
    hdr->node_info[0].cum_max_nrec_size = 0;
    if (NULL ==
        (hdr->node_info[0].nat_rec_fac = H5FL_fac_init(hdr->cls->nrec_size * hdr->node_info[0].max_nrec)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "can't create node native key block factory");
    hdr->node_info[0].node_ptr_fac = NULL;

    
    
    if (NULL == (hdr->nat_off = H5FL_SEQ_MALLOC(size_t, (size_t)hdr->node_info[0].max_nrec)))
        HGOTO_ERROR(H5E_BTREE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    
    for (u = 0; u < hdr->node_info[0].max_nrec; u++)
        hdr->nat_off[u] = hdr->cls->nrec_size * u;

    
    
    u_max_nrec_size = H5VM_limit_enc_size((uint64_t)hdr->node_info[0].max_nrec);
    H5_CHECKED_ASSIGN(hdr->max_nrec_size, uint8_t, u_max_nrec_size, unsigned);
    assert(hdr->max_nrec_size <= H5B2_SIZEOF_RECORDS_PER_NODE);

    
    if (depth > 0) {
        for (u = 1; u < (unsigned)(depth + 1); u++) {
            sz_max_nrec = H5B2_NUM_INT_REC(hdr, u);
            H5_CHECKED_ASSIGN(hdr->node_info[u].max_nrec, unsigned, sz_max_nrec, size_t);
            assert(hdr->node_info[u].max_nrec <= hdr->node_info[u - 1].max_nrec);

            hdr->node_info[u].split_nrec = (hdr->node_info[u].max_nrec * hdr->split_percent) / 100;
            hdr->node_info[u].merge_nrec = (hdr->node_info[u].max_nrec * hdr->merge_percent) / 100;

            hdr->node_info[u].cum_max_nrec =
                ((hdr->node_info[u].max_nrec + 1) * hdr->node_info[u - 1].cum_max_nrec) +
                hdr->node_info[u].max_nrec;
            u_max_nrec_size = H5VM_limit_enc_size((uint64_t)hdr->node_info[u].cum_max_nrec);
            H5_CHECKED_ASSIGN(hdr->node_info[u].cum_max_nrec_size, uint8_t, u_max_nrec_size, unsigned);

            if (NULL == (hdr->node_info[u].nat_rec_fac =
                             H5FL_fac_init(hdr->cls->nrec_size * hdr->node_info[u].max_nrec)))
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL, "can't create node native key block factory");
            if (NULL == (hdr->node_info[u].node_ptr_fac =
                             H5FL_fac_init(sizeof(H5B2_node_ptr_t) * (hdr->node_info[u].max_nrec + 1))))
                HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, FAIL,
                            "can't create internal 'branch' node node pointer block factory");
        } 
    }     

    
    hdr->swmr_write = (H5F_INTENT(hdr->f) & H5F_ACC_SWMR_WRITE) > 0 &&
                      (hdr->cls->id == H5B2_CDSET_ID || hdr->cls->id == H5B2_CDSET_FILT_ID);

    
    hdr->shadow_epoch = 0;

    
    if (hdr->cls->crt_context)
        if (NULL == (hdr->cb_ctx = (*hdr->cls->crt_context)(ctx_udata)))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCREATE, FAIL,
                        "unable to create v2 B-tree client callback context");

done:
    if (ret_value < 0)
        if (H5B2__hdr_free(hdr) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to free shared v2 B-tree info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B2_hdr_t *
H5B2__hdr_alloc(H5F_t *f)
{
    H5B2_hdr_t *hdr       = NULL; 
    H5B2_hdr_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (hdr = H5FL_CALLOC(H5B2_hdr_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "memory allocation failed for B-tree header");

    
    hdr->f           = f;
    hdr->sizeof_addr = H5F_SIZEOF_ADDR(f);
    hdr->sizeof_size = H5F_SIZEOF_SIZE(f);
    hdr->hdr_size    = H5B2_HEADER_SIZE_HDR(hdr);
    hdr->root.addr   = HADDR_UNDEF;

    
    ret_value = hdr;

done:

    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5B2__hdr_create(H5F_t *f, const H5B2_create_t *cparam, void *ctx_udata)
{
    H5B2_hdr_t *hdr       = NULL;        
    bool        inserted  = false;       
    haddr_t     ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cparam);

    
    if (NULL == (hdr = H5B2__hdr_alloc(f)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, HADDR_UNDEF, "allocation failed for B-tree header");

    
    if (H5B2__hdr_init(hdr, cparam, ctx_udata, (uint16_t)0) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINIT, HADDR_UNDEF, "can't create shared B-tree info");

    
    if (HADDR_UNDEF == (hdr->addr = H5MF_alloc(f, H5FD_MEM_BTREE, (hsize_t)hdr->hdr_size)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, HADDR_UNDEF, "file allocation failed for B-tree header");

    
    if (hdr->swmr_write)
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCREATE, HADDR_UNDEF, "can't create v2 B-tree proxy");

    
    if (H5AC_insert_entry(f, H5AC_BT2_HDR, hdr->addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTINSERT, HADDR_UNDEF, "can't add B-tree header to cache");
    inserted = true;

    
    if (hdr->top_proxy)
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add v2 B-tree header as child of array proxy");

    
    ret_value = hdr->addr;

done:
    if (!H5_addr_defined(ret_value))
        if (hdr) {
            
            if (inserted)
                if (H5AC_remove_entry(hdr) < 0)
                    HDONE_ERROR(H5E_BTREE, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove v2 B-tree header from cache");

            
            if (H5_addr_defined(hdr->addr) &&
                H5MF_xfree(f, H5FD_MEM_BTREE, hdr->addr, (hsize_t)hdr->hdr_size) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, HADDR_UNDEF, "unable to free v2 B-tree header");

            
            if (H5B2__hdr_free(hdr) < 0)
                HDONE_ERROR(H5E_BTREE, H5E_CANTRELEASE, HADDR_UNDEF, "unable to release v2 B-tree header");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_incr(H5B2_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->rc == 0)
        if (H5AC_pin_protected_entry(hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTPIN, FAIL, "unable to pin v2 B-tree header");

    
    hdr->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_decr(H5B2_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc > 0);

    
    hdr->rc--;

    
    if (hdr->rc == 0)
        if (H5AC_unpin_entry(hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPIN, FAIL, "unable to unpin v2 B-tree header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_fuse_incr(H5B2_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    
    hdr->file_rc++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

size_t
H5B2__hdr_fuse_decr(H5B2_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(hdr->file_rc);

    
    hdr->file_rc--;

    FUNC_LEAVE_NOAPI(hdr->file_rc)
} 

herr_t
H5B2__hdr_dirty(H5B2_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5AC_mark_entry_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTMARKDIRTY, FAIL, "unable to mark v2 B-tree header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5B2_hdr_t *
H5B2__hdr_protect(H5F_t *f, haddr_t hdr_addr, void *ctx_udata, unsigned flags)
{
    H5B2_hdr_cache_ud_t udata;            
    H5B2_hdr_t         *hdr       = NULL; 
    H5B2_hdr_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(hdr_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.f         = f;
    udata.addr      = hdr_addr;
    udata.ctx_udata = ctx_udata;

    
    if (NULL == (hdr = (H5B2_hdr_t *)H5AC_protect(f, H5AC_BT2_HDR, hdr_addr, &udata, flags)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTPROTECT, NULL, "unable to load v2 B-tree header, address = %llu",
                    (unsigned long long)hdr_addr);
    hdr->f = f; 

    
    if (hdr->swmr_write && NULL == hdr->top_proxy) {
        
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_BTREE, H5E_CANTCREATE, NULL, "can't create v2 B-tree proxy");

        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTSET, NULL, "unable to add v2 B-tree header as child of proxy");
    } 

    
    ret_value = hdr;

done:
    
    if (!ret_value) {
        
        if (hdr && H5AC_unprotect(hdr->f, H5AC_BT2_HDR, hdr_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect v2 B-tree header, address = %llu", (unsigned long long)hdr_addr);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_unprotect(H5B2_hdr_t *hdr, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5AC_unprotect(hdr->f, H5AC_BT2_HDR, hdr->addr, hdr, cache_flags) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect v2 B-tree header, address = %llu", (unsigned long long)hdr->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_free(H5B2_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->cb_ctx) {
        if ((*hdr->cls->dst_context)(hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL, "can't destroy v2 B-tree client callback context");
        hdr->cb_ctx = NULL;
    } 

    
    if (hdr->page)
        hdr->page = H5FL_BLK_FREE(node_page, hdr->page);

    
    if (hdr->nat_off)
        hdr->nat_off = H5FL_SEQ_FREE(size_t, hdr->nat_off);

    
    if (hdr->node_info) {
        unsigned u; 

        
        for (u = 0; u < (unsigned)(hdr->depth + 1); u++) {
            if (hdr->node_info[u].nat_rec_fac)
                if (H5FL_fac_term(hdr->node_info[u].nat_rec_fac) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's native record block factory");
            if (hdr->node_info[u].node_ptr_fac)
                if (H5FL_fac_term(hdr->node_info[u].node_ptr_fac) < 0)
                    HGOTO_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL,
                                "can't destroy node's node pointer block factory");
        } 

        
        hdr->node_info = H5FL_SEQ_FREE(H5B2_node_info_t, hdr->node_info);
    } 

    
    if (hdr->min_native_rec)
        hdr->min_native_rec = H5MM_xfree(hdr->min_native_rec);
    if (hdr->max_native_rec)
        hdr->max_native_rec = H5MM_xfree(hdr->max_native_rec);

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_dest(hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTRELEASE, FAIL, "unable to destroy v2 B-tree 'top' proxy");
        hdr->top_proxy = NULL;
    } 

    
    hdr = H5FL_FREE(H5B2_hdr_t, hdr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5B2__hdr_delete(H5B2_hdr_t *hdr)
{
    unsigned cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t   ret_value   = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

#ifndef NDEBUG
    {
        unsigned hdr_status = 0; 

        
        if (H5AC_get_entry_status(hdr->f, hdr->addr, &hdr_status) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for v2 B-tree header");

        
        assert(hdr_status & H5AC_ES__IN_CACHE);
        assert(hdr_status & H5AC_ES__IS_PROTECTED);
    }  
#endif 

    
    if (H5_addr_defined(hdr->root.addr))
        if (H5B2__delete_node(hdr, hdr->depth, &hdr->root, hdr, hdr->remove_op, hdr->remove_op_data) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDELETE, FAIL, "unable to delete B-tree nodes");

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (H5B2__hdr_unprotect(hdr, cache_flags) < 0)
        HDONE_ERROR(H5E_BTREE, H5E_CANTUNPROTECT, FAIL, "unable to release v2 B-tree header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
