/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FSmodule.h" 
#define H5HF_DEBUGGING  
#define H5MF_DEBUGGING  

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FSpkg.h"     
#include "H5HFprivate.h" 
#include "H5MFprivate.h" 

herr_t
H5FS_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth)
{
    H5FS_t             *fspace = NULL;       
    H5FS_hdr_cache_ud_t cache_udata;         
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    cache_udata.f              = f;
    cache_udata.nclasses       = 0;
    cache_udata.classes        = NULL;
    cache_udata.cls_init_udata = NULL;
    cache_udata.addr           = addr;

    
    if (NULL ==
        (fspace = (H5FS_t *)H5AC_protect(f, H5AC_FSPACE_HDR, addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, FAIL, "unable to load free space header");

    
    Rfprintf(stream, "%*sFree Space Header...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Free space client:",
            (fspace->client == H5FS_CLIENT_FHEAP_ID
                 ? "Fractal heap"
                 : (fspace->client == H5FS_CLIENT_FILE_ID ? "File" : "Unknown")));
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Total free space tracked:", fspace->tot_space);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Total number of free space sections tracked:", fspace->tot_sect_count);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of serializable free space sections tracked:", fspace->serial_sect_count);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Number of ghost free space sections tracked:", fspace->ghost_sect_count);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Number of free space section classes:", (unsigned)fspace->nclasses);
    Rfprintf(stream, "%*s%-*s %u%%\n", indent, "", fwidth, "Shrink percent:", fspace->shrink_percent);
    Rfprintf(stream, "%*s%-*s %u%%\n", indent, "", fwidth, "Expand percent:", fspace->expand_percent);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "# of bits for section address space:", fspace->max_sect_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Maximum section size:", fspace->max_sect_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Serialized sections address:", fspace->sect_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Serialized sections size used:", fspace->sect_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Serialized sections size allocated:", fspace->alloc_sect_size);

done:
    if (fspace && H5AC_unprotect(f, H5AC_FSPACE_HDR, addr, fspace, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_PROTECT, FAIL, "unable to release free space header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sect_debug(const H5FS_t *fspace, const H5FS_section_info_t *sect, FILE *stream, int indent, int fwidth)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fspace);
    assert(sect);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if (fspace->sect_cls[sect->type].debug)
        if ((fspace->sect_cls[sect->type].debug)(sect, stream, indent, fwidth) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't dump section's debugging info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_sects_debug(H5F_t *f, haddr_t H5_ATTR_UNUSED addr, FILE *stream, int indent, int fwidth, haddr_t fs_addr,
                 haddr_t client_addr)
{
    H5FS_t             *fspace = NULL;       
    H5FS_client_t       client;              
    H5FS_hdr_cache_ud_t cache_udata;         
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(H5_addr_defined(fs_addr));
    assert(H5_addr_defined(client_addr));

    
    cache_udata.f              = f;
    cache_udata.nclasses       = 0;
    cache_udata.classes        = NULL;
    cache_udata.cls_init_udata = NULL;
    cache_udata.addr           = fs_addr;

    
    if (NULL ==
        (fspace = (H5FS_t *)H5AC_protect(f, H5AC_FSPACE_HDR, fs_addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, FAIL, "unable to load free space header");

    
    client = fspace->client;

    
    
    if (H5AC_unprotect(f, H5AC_FSPACE_HDR, fs_addr, fspace, H5AC__DELETED_FLAG) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_PROTECT, FAIL, "unable to release free space header");
    fspace = NULL;

    
    Rfprintf(stream, "%*sFree Space Sections...\n", indent, "");

    
    switch (client) {
        case H5FS_CLIENT_FHEAP_ID:
            if (H5HF_sects_debug(f, client_addr, stream, indent + 3, MAX(0, fwidth - 3)) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_SYSTEM, FAIL, "unable to dump fractal heap free space sections");
            break;

        case H5FS_CLIENT_FILE_ID:
            if (H5MF_sects_debug(f, fs_addr, stream, indent + 3, MAX(0, fwidth - 3)) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_SYSTEM, FAIL, "unable to dump file free space sections");
            break;

        case H5FS_NUM_CLIENT_ID:
        default:
            Rfprintf(stream, "Unknown client!\n");
            break;
    } 

done:
    if (fspace && H5AC_unprotect(f, H5AC_FSPACE_HDR, fs_addr, fspace, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_PROTECT, FAIL, "unable to release free space header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
