% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bmult_inequalities_mult.R
\name{mult_bf_inequality}
\alias{mult_bf_inequality}
\title{Computes Bayes Factors For Inequality Constrained Multinomial Parameters}
\usage{
mult_bf_inequality(
  samples = NULL,
  restrictions = NULL,
  x = NULL,
  Hr = NULL,
  a = rep(1, ncol(samples)),
  factor_levels = NULL,
  prior = FALSE,
  index = 1,
  maxiter = 1000,
  seed = NULL,
  niter = 5000,
  nburnin = niter * 0.05
)
}
\arguments{
\item{samples}{matrix of dimension \code{nsamples x nparams} with samples from truncated Dirichlet density}

\item{restrictions}{\code{list} of class \code{bmult_rl} or of class \code{bmult_rl_ineq} as returned from \code{\link{generate_restriction_list}} that encodes
inequality constraints for each independent restriction}

\item{x}{numeric. Vector with data}

\item{Hr}{string or character. Encodes the user specified informed hypothesis. Use either specified \code{factor_levels}
or indices to refer to parameters. See ``Note'' section for details on how to formulate informed hypotheses}

\item{a}{numeric. Vector with concentration parameters of Dirichlet distribution. Must be the same length as \code{x}. Default sets all concentration parameters to 1}

\item{factor_levels}{character. Vector with category names. Must be the same length as \code{x}}

\item{prior}{logical. If \code{TRUE} the function will ignore the data and evaluate only the prior distribution}

\item{index}{numeric. Index of current restriction. Default is 1}

\item{maxiter}{numeric. Maximum number of iterations for the iterative updating scheme used in the bridge sampling routine.
Default is 1,000 to avoid infinite loops}

\item{seed}{numeric. Sets the seed for reproducible pseudo-random number generation}

\item{niter}{numeric. Vector with number of samples to be drawn from truncated distribution}

\item{nburnin}{numeric. A single value specifying the number of burn-in samples when drawing from the truncated distribution.
Minimum number of burn-in samples is 10. Default is 5\% of the number of samples. Burn-in samples are removed automatically after the sampling.}
}
\value{
List consisting of the following elements:
\describe{
\item{\code{$eval}}{
\itemize{
\item \code{q11}: log prior or posterior evaluations for prior or posterior samples
\item \code{q12}: log proposal evaluations for prior or posterior samples
\item \code{q21}: log prior or posterior evaluations for samples from proposal
\item \code{q22}: log proposal evaluations for samples from proposal
}}
\item{\code{$niter}}{number of iterations of the iterative updating scheme}
\item{\code{$logml}}{estimate of log marginal likelihood}
\item{\code{$hyp}}{evaluated inequality constrained hypothesis}
\item{\code{$error_measures}}{
\itemize{
\item \code{re2}: the approximate
relative mean-squared error for the marginal likelihood estimate
\item \code{cv}: the approximate coefficient of variation for the marginal
likelihood estimate (assumes that bridge estimate is unbiased)
\item \code{percentage}: the approximate percentage error of the marginal likelihood estimate
}}
}
}
\description{
Computes Bayes factor for inequality constrained multinomial parameters using a bridge sampling routine.
Restricted hypothesis \eqn{H_r} states that category proportions follow a particular trend.
Alternative hypothesis \eqn{H_e} states that category proportions are free to vary.
}
\details{
The model assumes that data follow a multinomial distribution and assigns a Dirichlet distribution as prior for the model parameters
(i.e., underlying category proportions). That is:
\deqn{x ~ Multinomial(N, \theta)}
\deqn{\theta ~ Dirichlet(\alpha)}
}
\note{
The following signs can be used to encode restricted hypotheses: \code{"<"} and \code{">"} for inequality constraints, \code{"="} for equality constraints,
\code{","} for free parameters, and \code{"&"} for independent hypotheses. The restricted hypothesis can either be a string or a character vector.
For instance, the hypothesis \code{c("theta1 < theta2, theta3")} means
\itemize{
\item \code{theta1} is smaller than both \code{theta2} and \code{theta3}
\item The parameters \code{theta2} and \code{theta3} both have \code{theta1} as lower bound, but are not influenced by each other.
}
The hypothesis \code{c("theta1 < theta2 = theta3 & theta4 > theta5")} means that
\itemize{
\item Two independent hypotheses are stipulated: \code{"theta1 < theta2 = theta3"} and \code{"theta4 > theta5"}
\item The restrictions on the parameters \code{theta1}, \code{theta2}, and \code{theta3} do
not influence the restrictions on the parameters \code{theta4} and \code{theta5}.
\item \code{theta1} is smaller than \code{theta2} and \code{theta3}
\item \code{theta2} and \code{theta3} are assumed to be equal
\item \code{theta4} is larger than \code{theta5}
}
}
\examples{
# priors
a <- c(1, 1, 1, 1)

# informed hypothesis
factor_levels <- c('theta1', 'theta2', 'theta3', 'theta4')
Hr            <- c('theta1', '<',  'theta2', '<', 'theta3', '<', 'theta4')

results_prior  <- mult_bf_inequality(Hr=Hr, a=a, factor_levels=factor_levels, 
prior=TRUE, seed = 2020)
# corresponds to
cbind(exp(results_prior$logml), 1/factorial(4))

# alternative - if you have samples and a restriction list
inequalities  <- generate_restriction_list(Hr=Hr, a=a,
factor_levels=factor_levels)$inequality_constraints
prior_samples <- mult_tsampling(inequalities, niter = 2e3, 
prior=TRUE, seed = 2020)
results_prior <- mult_bf_inequality(prior_samples, inequalities, seed=2020)
cbind(exp(results_prior$logml), 1/factorial(4))
}
\references{
\insertRef{damien2001sampling}{multibridge}

\insertRef{gronau2017tutorial}{multibridge}

\insertRef{fruhwirth2004estimating}{multibridge}

\insertRef{sarafoglou2020evaluatingPreprint}{multibridge}
}
\seealso{
\code{\link{generate_restriction_list}}

Other functions to evaluate informed hypotheses: 
\code{\link{binom_bf_equality}()},
\code{\link{binom_bf_inequality}()},
\code{\link{binom_bf_informed}()},
\code{\link{mult_bf_equality}()},
\code{\link{mult_bf_informed}()}
}
\concept{functions to evaluate informed hypotheses}
