#' Optimal full matching
#' 
#' Given two groups, such as a treatment and a control group, and a
#' treatment-by-control discrepancy matrix indicating desirability and
#' permissibility of potential matches, create optimal full matches of members
#' of the groups.  Optionally, incorporate restrictions on matched sets' ratios
#' of treatment to control units.
#'
#' Finite entries in matrix slots of \code{distance} indicate permissible
#' matches, with smaller discrepancies indicating more desirable matches.  Matrix
#' \code{distance} must have row and column names.
#'   
#' Consider using \code{\link{match_on}} to generate the distances. This generic
#' function as several useful methods for handling propensity score models,
#' computing Mahalanobis distances (and other arbitrary distances), and using
#' user supplied functions. These distances can also be combined with those
#' generated by \code{\link{exactMatch}} and \code{\link{caliper}} to create
#' very nuanced matching specifications.
#'   
#' The value of \code{tol} can have a substantial effect on computation time;
#' with smaller values, computation takes longer.  Not every tolerance can be
#' met, and how small a tolerance is too small varies with the machine and with
#' the details of the problem.  If \code{fullmatch} can't guarantee that the
#' tolerance is as small as the given value of argument \code{tol}, then
#' matching proceeds but a warning is issued.  
#'
#' @param distance A matrix of non-negative discrepancies, each indicating the
#' permissibility and desirability of matching the unit corresponding to its row
#' (a 'treatment') to the unit corresponding to its column (a 'control'); or,
#' better, a distance specification as produced by \code{\link{match_on}}.
#'
#' @param min.controls The minimum ratio of controls to treatments that is to
#' be permitted within a matched set: should be non-negative and finite.  If
#' \code{min.controls} is not a whole number, the reciprocal of a whole number,
#' or zero, then it is rounded \emph{down} to the nearest whole number or
#' reciprocal of a whole number.
#'
#' When matching within subclasses (such as those created by
#' \code{\link{exactMatch}}), \code{min.controls} may be a named numeric vector
#' separately specifying the minimum permissible ratio of controls to treatments
#' for each subclass.  The names of this vector should include names of all
#' subproblems \code{distance}.
#' 
#' @param max.controls The maximum ratio of controls to treatments that is
#' to be permitted within a matched set: should be positive and numeric.
#' If \code{max.controls} is not a whole number, the reciprocal of a
#' whole number, or \code{Inf}, then it is rounded \emph{up} to the
#' nearest whole number or reciprocal of a whole number.
#'
#' When matching within subclasses (such as those created by
#' \code{\link{exactMatch}}), \code{max.controls} may be a named numeric vector
#' separately specifying the maximum permissible ratio of controls to treatments
#' in each subclass.
#'
#' @param omit.fraction Optionally, specify what fraction of
#' controls or treated subjects are to be rejected.  If
#' \code{omit.fraction} is a positive fraction less than one, then
#' \code{fullmatch} leaves up to that fraction of the control reservoir
#' unmatched.  If \code{omit.fraction} is a negative number greater than
#' -1, then \code{fullmatch} leaves up to |\code{omit.fraction}| of the
#' treated group unmatched.  Positive values are only accepted if
#' \code{max.controls} >= 1; negative values, only if \code{min.controls}
#' <= 1.  If \code{omit.fraction} is not specified, then only those
#' treated and control subjects without permissible matches among the
#' control and treated subjects, respectively, are omitted.
#' 
#' When matching within subclasses (such as those created by
#' \code{\link{exactMatch}}), \code{omit.fraction} specifies the fraction of
#' controls to be rejected in each subproblem, a parameter that can be made to
#' differ by subclass by setting \code{omit.fraction} equal to a named numeric
#' vector of fractions.
#'
#' @param tol Because of internal rounding, \code{fullmatch} may
#' solve a slightly different matching problem than the one
#' specified, in which the match generated by
#' \code{fullmatch} may not coincide with an optimal solution of
#' the specified problem.  \code{tol} times the number of subjects
#' to be matched specifies the extent to
#' which \code{fullmatch}'s output is permitted to differ from an
#' optimal solution to the original problem, as measured by the
#' sum of discrepancies for all treatments and controls placed
#' into the same matched sets.
#'
#' @param data Optional \code{data.frame} or \code{vector} to use to get order
#' of the final matching factor. If a \code{data.frame}, the \code{rownames}
#' are used. If a vector, the \code{names} are first tried, otherwise the contents
#' is considered to be a character vector of names. Useful to pass if you want to
#' combine a match (using, e.g., \code{cbind}) with the data that were used to
#' generate it (for example, in a propensity score matching).
#'
#' @return A \code{\link{optmatch}} object (\code{factor}) indicating matched groups. 
#'
#' @references
#'  Hansen, B.B. and Klopfer, S.O. (2006), \sQuote{ Optimal full matching and related designs via network flows}, 
#'  \emph{Journal of Computational and Graphical Statistics}, \bold{15}, 609--627.
#'  
#'  Hansen, B.B. (2004), \sQuote{Full Matching in an Observational Study
#'  of Coaching for the SAT}, \emph{Journal of the American
#'  Statistical Association}, \bold{99}, 609--618.  
#'
#'  Rosenbaum, P. (1991), \sQuote{A Characterization of Optimal Designs for Observational
#'  Studies}, \emph{Journal of the Royal Statistical Society, Series B},
#'  \bold{53}, 597--610.
#' 
#' @example inst/examples/fullmatch.R
#' @keywords nonparametric optimize
#' @export
fullmatch <- function(distance,  
    min.controls = 0, 
    max.controls = Inf, 
    omit.fraction = NULL, 
    tol = .001,
    data = NULL) {

  ### Checking Input ###
  
  # this will throw an error if not valid
  validDistanceSpecification(distance)

  if (is.null(data)) {
    warning("Without 'data' argument the order of the match is not guaranteed
    to be the same as your original data.")  
  }

  # note: we might want to move these checks to validDistSpec
  dnms <- dimnames(distance)
  if (is.null(dnms) | is.null(dnms[[1]]) | is.null(dnms[[2]])) {
    stop("argument \'distance\' must have dimnames") 
  }
  
  if (any(duplicated(unlist(dnms)))){ 
    stop("dimnames of argument \'distance\' contain duplicates") 
  }

  nmtrt <- dnms[[1]]
  nmctl <- dnms[[2]]

  # note: this next _should_ be unnecessary, the objects should do this
  # but better safe than sorry
  if (!isTRUE(all.equal(dim(distance), c(length(nmtrt), length(nmctl))))) {
    stop("argument \'distance\' dimensions do not match row and column names")  
  }

  if (!is.numeric(min.controls)) {
    stop("argument \'min.controls\' must be numeric")  
  }
  if (!is.numeric(max.controls)) {
    stop("argument \'max.controls\' must be numeric")  
  }
  if (!is.null(omit.fraction)) {
    if (any(abs(omit.fraction) > 1, na.rm = TRUE) | !is.numeric(omit.fraction)) { 
      stop("omit.fraction must be NULL or numeric between -1 and 1") 
    }
  }


  # problems is guaranteed to be a list of DistanceSpecifictions
  # it may only have 1 entry
  problems <- findSubproblems(distance)

  # the number of problems should match the argument lengths for
  # min, max, and omit

  np <- length(problems)
  if (length(min.controls) > 1 & np != length(min.controls)) {
    stop(paste("Length of \'min.controls\' arg must be same ",
              "as number of subproblems [", np, "]", sep = ""))  
  }
  if (length(max.controls) > 1 & np != length(max.controls)) {
    stop(paste("Length of \'max.controls\' arg must be same ",
              "as number of subproblems [", np, "]", sep = ""))  
  }
  if (!is.null(omit.fraction) & length(omit.fraction) > 1 & np !=
    length(omit.fraction)) {
    stop(paste("Length of \'omit.fraction\' arg must be same ",
              "as number of subproblems [", np, "]", sep = ""))  
  }
  
  # reset the arguments to be the right length if they are not
  if (length(min.controls) == 1) {
    min.controls <- rep(min.controls, np)   
  }
  if (length(max.controls) == 1) {
    max.controls <- rep(max.controls, np)   
  }

  if (is.null(omit.fraction)) {
    omit.fraction <- NA  
  }
  if (length(omit.fraction) == 1) {
    omit.fraction <- rep(omit.fraction, np)   
  }

  if (any(omit.fraction > 0 & max.controls <= .5, na.rm=TRUE)) {
      stop("positive \'omit.fraction\' with \'max.controls\' <= 1/2 not permitted")    
  }

  if (any(omit.fraction < 0 & min.controls >= 2, na.rm=TRUE)) {
      stop("negative \'omit.fraction\' with \'min.controls\' >= 2 not permitted")
  }
  
  total.n <- sum(dim(distance))

  TOL <- tol * total.n

  # a helper to handle a single matching problem. all args required. 
  # input error checking happens in the public fullmatch function.
  .fullmatch <- function(d, mnctl, mxctl, omf) {
    
    # if the subproblem is completely empty, short circuit
    if (length(d) == 0 || all(is.infinite(d))) {
      x <- dim(d)
      cells.a <- rep(NA, x[1])
      cells.b <- rep(NA, x[2])
      names(cells.a) <- rownames(d)
      names(cells.b) <- colnames(d)
      tmp <- list(cells = c(cells.a, cells.b), maxerr = -1)
      return(tmp) 
    }

    ncol <- dim(d)[2]
    nrow <- dim(d)[1]

    tol.frac <- (nrow + ncol - 2)/(total.n - 2 * np)
  
    # if omf is specified (i.e. not NA), see if is greater than 0
    # if omf is not specified, check to see if mxctl is > .5
    if (switch(1 + is.na(omf), omf > 0,  mxctl > .5)) {
      maxc <- min(mxctl, ncol)
      minc <- max(mnctl, 1/nrow)
      omf.calc <- omf
  
    } else {
      maxc <- min(1/mnctl, ncol)
      minc <- max(1/mxctl, 1/nrow)
      omf.calc <- -1 * omf
    }

    temp <- SubDivStrat(rownames = rownames(d),
                        colnames = colnames(d),
                        distspec = d, 
                        max.cpt = maxc,
                        min.cpt = minc,
                        tolerance = TOL * tol.frac,
                        omit.fraction = if(!is.na(omf)) { omf.calc }) # passes NULL for NA

    return(temp)
  }
  
  solutions <- mapply(.fullmatch, problems, min.controls, max.controls, omit.fraction, SIMPLIFY = FALSE)

  return(makeOptmatch(distance, solutions, match.call(), data))
}


#' @aliases fullmatch
#' @rdname fullmatch
#' @export
full <- fullmatch
