% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/twoLocusIdentity.R
\name{twoLocusIdentity}
\alias{twoLocusIdentity}
\title{Two-locus identity coefficients}
\usage{
twoLocusIdentity(x, ids, rho, coefs = NULL, detailed = FALSE, verbose = FALSE)
}
\arguments{
\item{x}{A pedigree in the form of a \code{\link[pedtools:ped]{pedtools::ped}} object.}

\item{ids}{A character (or coercible to character) containing ID labels of
two pedigree members.}

\item{rho}{A number in the interval \eqn{[0, 0.5]}; the recombination rate
between the two loci.}

\item{coefs}{A character indicating which coefficient(s) to compute. A subset
of \code{c('D00', 'D01', ..., 'D99')}. By default, all coefficients are
computed.}

\item{detailed}{A logical, indicating whether the condensed (default) or
detailed coefficients should be returned.}

\item{verbose}{A logical.}
}
\value{
By default, a symmetric 9*9 matrix containing the two-locus condensed
identity coefficients \eqn{\Delta_{i,j}}{\Delta_ij}.

If either \code{coefs} is explicitly given (i.e., not NULL), or \code{detailed = TRUE}, the computed coefficients are returned as a named vector.
}
\description{
Computes the 9*9 matrix of two-locus condensed identity coefficients of a
pair of pedigree members, for a given recombination rate.
}
\details{
Let A, B be two pedigree members, and L1, L2 two loci with a given
recombination rate \eqn{\rho}. The two-locus identity coefficient
\eqn{\Delta_{i,j}(\rho)}{\Delta_ij(\rho)}, for \eqn{1 \le i,j \le 9} is
defined as the probability that the identity state of the alleles of A and B
are \eqn{\Sigma_i} at L1 and \eqn{\Sigma_j} at L2 simultaneously. (The
ordering of the 9 states follows Jacquard (1974).)

For details about the algorithm, see Vigeland (2022).
}
\examples{
### Full sibs ###
x = nuclearPed(2)
kapp = twoLocusIBD(x, ids = 3:4, rho = 0.25)
jacq = twoLocusIdentity(x, ids = 3:4, rho = 0.25)

stopifnot(all.equal(jacq[9:7,9:7], kapp, check.attributes = FALSE))

### Parent-child ###
x = nuclearPed(1)
jacq = twoLocusIdentity(x, ids = c(1,3), rho = 0.25)

stopifnot(jacq[8,8] == 1)

### Full sib mating ###
x = fullSibMating(1)
j = condensedIdentity(x, ids = 5:6)
j2 = twoLocusIdentity(x, ids = 5:6, rho = 0.25)

stopifnot(identical(unname(rowSums(j2)), j))


}
\references{
\itemize{
\item Jacquard (1974). The Genetic Structure of Populations. Springer.
\item Vigeland (2022) \emph{Two-locus identity coefficients in pedigrees} (In progress)
}
}
\seealso{
\code{\link[=twoLocusIBD]{twoLocusIBD()}}, \code{\link[=identityCoefs]{identityCoefs()}}
}
