% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binSegR6.R
\docType{class}
\name{binSeg}
\alias{binSeg}
\title{Binary Segmentation (\code{binSeg})}
\description{
An \code{R6} class implementing binary segmentation for offline change-point detection.
}
\details{
Binary segmentation is a classic algorithm for change-point detection that recursively
splits the data at locations that minimise the cost function.

\code{binSeg} requires  a \code{R6} object of class \code{costFunc}, which can be created via \code{costFunc$new()}. Currently, the following cost functions are supported:
\itemize{
\item \code{"L1"} and \code{"L2"} for (independent) piecewise Gaussian process with \strong{constant variance}
\item \code{"SIGMA"}: for (independent) piecewise Gaussian process with \strong{varying variance}
\item \code{"VAR"}: for piecewise Gaussian vector-regressive process with \strong{constant noise variance}
\item \code{"LinearL2"}: for piecewise linear regression process with \strong{constant noise variance}
}

See \verb{$eval()} method for more details on computation of cost.

Some examples are provided below. See the \href{https://github.com/edelweiss611428/rupturesRcpp/blob/main/README.md}{GitHub README}
for detailed basic usage!
}
\section{Methods}{

\describe{
\item{\code{$new()}}{Initialises a \code{binSeg} object.}
\item{\code{$describe()}}{Describes the \code{binSeg} object.}
\item{\code{$fit()}}{Constructs a \code{binSeg} module in \verb{C++}.}
\item{\code{$eval()}}{Evaluates the cost of a segment.}
\item{\code{$predict()}}{Performs \code{binSeg} given a linear penalty value.}
\item{\code{$plot()}}{Plots change-point segmentation in \code{ggplot} style.}
\item{\code{$clone()}}{Clones the \code{R6} object.}
}
}

\examples{

## L2 example
set.seed(1121)
signals = as.matrix(c(rnorm(100,0,1),
                     rnorm(100,5,1)))
# Default L2 cost function
binSegObj = binSeg$new(minSize = 1L, jump = 1L)
binSegObj$fit(signals)
binSegObj$predict(pen = 100)
binSegObj$plot()

## SIGMA example
set.seed(111)
signals = as.matrix(c(rnorm(100,-5,1),
                      rnorm(100,-5,10),
                      rnorm(100,-5,1)))
# L2 cost function
binSegObj = binSeg$new(minSize = 1L, jump = 1L)
binSegObj$fit(signals)
# We choose pen = 50.
binSegObj$predict(pen = 50)
binSegObj$plot()

# The standard L2 cost function is not suitable.
# Use the SIGMA cost function.
binSegObj$costFunc = costFunc$new(costFunc = "SIGMA")
binSegObj$predict(pen = 50)
binSegObj$plot()

}
\references{
Truong, C., Oudre, L., & Vayatis, N. (2020). Selective review of offline change point detection methods.
Signal Processing, 167, 107299.

Hocking, T. D. (2024). Finite Sample Complexity Analysis of Binary Segmentation. arXiv preprint arXiv:2410.08654.
}
\author{
Minh Long Nguyen \email{edelweiss611428@gmail.com} \cr
Toby Dylan Hocking \email{toby.hocking@r-project.org} \cr
Charles Truong \email{ctruong@ens-paris-saclay.fr}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{minSize}}{Integer. Minimum allowed segment length. Can be accessed or modified via \verb{$minSize}.
Modifying \code{minSize} will automatically trigger \verb{$fit()}.}

\item{\code{jump}}{Integer. Search grid step size. Can be accessed or modified via \verb{$jump}.
Modifying \code{jump} will automatically trigger \verb{$fit()}.}

\item{\code{costFunc}}{\code{R6} object of class \code{costFunc}. Search grid step size. Can be accessed or modified via \verb{$costFunc}.
Modifying \code{costFunc} will automatically trigger \verb{$fit()}.}

\item{\code{tsMat}}{Numeric matrix. Input time series matrix of size \eqn{n \times p}. Can be accessed or modified via \verb{$tsMat}.
Modifying \code{tsMat} will automatically trigger \verb{$fit()}.}

\item{\code{covariates}}{Numeric matrix. Input time series matrix having a similar number of observations as \code{tsMat}. Can be accessed or modified via \verb{$covariates}.
Modifying \code{covariates} will automatically trigger \verb{$fit()}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-binSeg-new}{\code{binSeg$new()}}
\item \href{#method-binSeg-describe}{\code{binSeg$describe()}}
\item \href{#method-binSeg-fit}{\code{binSeg$fit()}}
\item \href{#method-binSeg-eval}{\code{binSeg$eval()}}
\item \href{#method-binSeg-predict}{\code{binSeg$predict()}}
\item \href{#method-binSeg-plot}{\code{binSeg$plot()}}
\item \href{#method-binSeg-clone}{\code{binSeg$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-new"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-new}{}}}
\subsection{Method \code{new()}}{
Initialises a \code{binSeg} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$new(minSize, jump, costFunc)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{minSize}}{Integer. Minimum allowed segment length. Default: \code{1L}.}

\item{\code{jump}}{Integer. Search grid step size: only positions in \{k, 2k, ...\} are considered. Default: \code{1L}.}

\item{\code{costFunc}}{A \code{R6} object of class \code{costFunc}. Should be created via \code{costFunc$new()} to avoid error.
Default: \code{costFunc$new("L2")}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly returns \code{NULL}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-describe"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-describe}{}}}
\subsection{Method \code{describe()}}{
Describes a \code{binSeg} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$describe(printConfig = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{printConfig}}{Logical. Whether to print object configurations. Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly returns a list storing at least the following fields:

\describe{
\item{\code{minSize}}{Minimum allowed segment length.}
\item{\code{jump}}{Search grid step size.}
\item{\code{costFunc}}{The \code{costFun} object.}
\item{\code{fitted}}{Whether or not \verb{$fit()} has been run.}
\item{\code{tsMat}}{Time series matrix.}
\item{\code{covariates}}{Covariate matrix (if exists).}
\item{\code{n}}{Number of observations.}
\item{\code{p}}{Number of features.}
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-fit"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-fit}{}}}
\subsection{Method \code{fit()}}{
Constructs a \verb{C++} module for binary segmentation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$fit(tsMat = NULL, covariates = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{tsMat}}{Numeric matrix. A time series matrix of size \eqn{n \times p} whose rows are observations ordered in time.
If \code{tsMat = NULL}, the method will use the previously assigned \code{tsMat} (e.g., set via the active binding \verb{$tsMat}
or from a prior \verb{$fit(tsMat)}). Default: \code{NULL}.}

\item{\code{covariates}}{Numeric matrix. A time series matrix having a similar number of observations as \code{tsMat}.
Required for models involving both dependent and independent variables.
If \code{covariates = NULL} and no prior covariates were set (i.e., \verb{$covariates} is still \code{NULL}),
the model is force-fitted with only an intercept. Default: \code{NULL}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
This method constructs a \verb{C++} \code{binSeg} module and sets \code{private$.fitted} to \code{TRUE},
enabling the use of \verb{$predict()} and \verb{$eval()}. Some precomputations are performed to allow
\verb{$predict()} to run in linear time with respect to the number of data points
(see \verb{$predict()} for more details).
}

\subsection{Returns}{
Invisibly returns \code{NULL}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-eval"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-eval}{}}}
\subsection{Method \code{eval()}}{
Evaluate the cost of the segment (a,b]
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$eval(a, b)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{a}}{Integer. Start index of the segment (exclusive). Must satisfy \code{start < end}.}

\item{\code{b}}{Integer. End index of the segment (inclusive).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The segment cost is evaluated as follows:
\itemize{
\item \strong{L1 cost function}:
\deqn{c_{L_1}(y_{(a+1)...b}) := \sum_{t = a+1}^{b} \| y_t - \tilde{y}_{(a+1)...b} \|_1}
where \eqn{\tilde{y}_{(a+1)...b}} is the coordinate-wise median of the segment. If \eqn{a \ge b - 1}, return 0.
\item \strong{L2 cost function}:
\deqn{c_{L_2}(y_{(a+1)...b}) := \sum_{t = a+1}^{b} \| y_t - \bar{y}_{(a+1)...b} \|_2^2}
where \eqn{\bar{y}_{(a+1)...b}} is the empirical mean of the segment. If \eqn{a \ge b - 1}, return 0.
\item \strong{SIGMA cost function}:
\deqn{c_{\sum}(y_{(a+1)...b}) := (b - a)\log \det \hat{\Sigma}_{(a+1)...b}} where \eqn{\hat{\Sigma}_{(a+1)...b}} is
the empirical covariance matrix of the segment without Bessel's correction. Here, if \code{addSmallDiag = TRUE}, a small
bias \code{epsilon} is added to the diagonal of estimated covariance matrices to improve numerical stability. \cr
\cr
By default, \code{addSmallDiag = TRUE} and \code{epsilon = 1e-6}. In case \code{addSmallDiag = TRUE}, if the computed determinant of covariance matrix is either 0 (singular)
or smaller than \code{p*log(epsilon)} - the lower bound, return \code{(b - a)*p*log(epsilon)}, otherwise, output an error message.
\item \strong{VAR(r) cost function}:
\deqn{c_{\mathrm{VAR}}(y_{(a+1)...b}) := \sum_{t = a+r+1}^{b} \left\| y_t - \sum_{j=1}^r \hat A_j y_{t-j} \right\|_2^2}
where \eqn{\hat A_j} are the estimated VAR coefficients, commonly estimated via the OLS criterion. If system is singular,
\eqn{a-b < p*r+1} (i.e., not enough observations), or \eqn{a \ge n-p} (where \code{n} is the time series length), return 0.
\item \strong{"LinearL2"} for piecewise linear regression process with \strong{constant noise variance}
\deqn{c_{\text{LinearL2}}(y_{(a+1):b}) := \sum_{t=a+1}^b \| y_t - X_t \hat{\beta} \|_2^2} where \eqn{\hat{\beta}} are OLS estimates on segment \eqn{(a+1):b}. If segment is shorter than the minimum number of
points needed for OLS, return 0.
}
}

\subsection{Returns}{
The segment cost.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-predict"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-predict}{}}}
\subsection{Method \code{predict()}}{
Performs \code{binSeg} given a linear penalty value.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$predict(pen = 0)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{pen}}{Numeric. Penalty per change-point. Default: \code{0}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The algorithm recursively partitions a time series to detect multiple change-points.
At each step, the algorithm identifies the segment that, if split, would result in the greatest reduction in total cost.
This process continues until no further splits are possible (e.g., each segment is of minimal length or each breakpoint
corresponds to a single data point).

Then, the algorithm selects the "optimal" set of break-points given the linear penalty threshold. Let \eqn{[c_1, \dots, c_k, c_{k+1}]} denote the set of segment end-points with \eqn{c_1 < c_2 < \dots < c_k < c_{k+1} = n},
where \eqn{k} is the number of detected change-points and \eqn{n} is the total number of data points.
and \eqn{k} is the number of change-points. Let \eqn{c_{(c_i, c_{i+1}]}} be the cost of segment \eqn{(c_i, c_{i+1}]}.
The total penalised cost is then
\deqn{
  \text{TotalCost} = \sum_{i=1}^{k+1} c_{(c_i, c_{i+1}]} + \lambda \cdot k,
}
where \eqn{\lambda} is a linear penalty applied per change-point. We then optimise over
\eqn{k} to minimise the penalised cost function.

This approach allows detecting multiple change-points in a time series while controlling
model complexity through the linear penalty threshold.

In our implementation, the recursive step is carried out during \verb{$fit()}.
Therefore, \verb{$predict()} runs in linear time with respect to the number of data points.

Temporary segment end-points are saved to \code{private$.tmpEndPoints} after \verb{$predict()}, enabling users to call \verb{$plot()} without
specifying endpoints manually.
}

\subsection{Returns}{
An integer vector of regime end-points. By design, the last element is the
number of observations.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-plot"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-plot}{}}}
\subsection{Method \code{plot()}}{
Plots change-point segmentation
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$plot(
  d = 1L,
  endPts,
  dimNames,
  main,
  xlab,
  tsWidth = 0.25,
  tsCol = "#5B9BD5",
  bgCol = c("#A3C4F3", "#FBB1BD"),
  bgAlpha = 0.5,
  ncol = 1L
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{d}}{Integer vector. Dimensions to plot. Default: \code{1L}.}

\item{\code{endPts}}{Integer vector. End points. Default: latest temporary changepoints obtained via \verb{$predict()}.}

\item{\code{dimNames}}{Character vector. Feature names matching length of \code{d}. Defaults to \verb{"X1", "X2", ...}.}

\item{\code{main}}{Character. Main title. Defaults to \code{"binSeg: d = ..."}.}

\item{\code{xlab}}{Character. X-axis label. Default: \code{"Time"}.}

\item{\code{tsWidth}}{Numeric. Line width for time series and segments. Default: \code{0.25}.}

\item{\code{tsCol}}{Character. Time series color. Default: \code{"#5B9BD5"}.}

\item{\code{bgCol}}{Character vector. Segment colors, recycled to length of \code{endPts}. Default: \code{c("#A3C4F3", "#FBB1BD")}.}

\item{\code{bgAlpha}}{Numeric. Background transparency. Default: \code{0.5}.}

\item{\code{ncol}}{Integer. Number of columns in facet layout. Default: \code{1L}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Plots change-point segmentation results. Based on \code{ggplot2}. Multiple plots can easily be
horizontally and vertically stacked using \code{patchwork}'s operators \code{/} and \code{|}, respectively.
}

\subsection{Returns}{
An object of classes \code{gg} and \code{ggplot}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-binSeg-clone"></a>}}
\if{latex}{\out{\hypertarget{method-binSeg-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{binSeg$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
