% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_cube_local.R
\name{sits_cube.vector_cube}
\alias{sits_cube.vector_cube}
\title{Create a vector cube from local files}
\usage{
\method{sits_cube}{vector_cube}(
  source,
  collection,
  ...,
  raster_cube,
  vector_dir,
  vector_band,
  parse_info = c("X1", "X2", "tile", "start_date", "end_date", "band", "version"),
  version = "v1",
  delim = "_",
  multicores = 2L,
  progress = TRUE
)
}
\arguments{
\item{source}{Data source: one of \code{"AWS"}, \code{"BDC"},
\code{"CDSE"}, \code{"DEAFRICA"}, \code{"DEAUSTRALIA"},
\code{"HLS"}, \code{"PLANETSCOPE"}, \code{"MPC"},
\code{"SDC"} or \code{"USGS"}. This is the source
from which the data has been downloaded.}

\item{collection}{Image collection in data source.
To find out the supported collections,
use \code{\link{sits_list_collections}()}).}

\item{...}{Other parameters to be passed for specific types.}

\item{raster_cube}{Raster cube to be merged with vector data}

\item{vector_dir}{Local directory where vector files are stored}

\item{vector_band}{Band for vector cube ("segments", "probs", "class")}

\item{parse_info}{Parsing information for local image files}

\item{version}{Version of the classified and/or labelled files.}

\item{delim}{Delimiter for parsing local files
(default = "_")}

\item{multicores}{Number of workers for parallel processing
(integer, min = 1, max = 2048).}

\item{progress}{Logical: show a progress bar?}
}
\value{
A \code{tibble} describing the contents of a data cube.
}
\description{
Creates a data cube from local files which include a vector file
produced by a segmentation algorithm.
}
\note{
This function creates vector cubes from local files produced by
\code{\link[sits]{sits_segment}}, \code{\link[sits]{sits_classify}}
or \code{\link[sits]{sits_label_classification}} when the output
is a vector cube. In this case,
\code{parse_info} is specified differently as \code{c("X1", "X2", "tile",
"start_date", "end_date", "band")}.
The parameter \code{vector_dir} is the directory where the vector file is
stored.
Parameter \code{vector_band} is band name of the type of vector cube:
\itemize{
\item{\code{"segments"}, for vector cubes produced by
   \code{\link{sits_segment}}.}
 \item{\code{"probs"}, for probability cubes produced by
   \code{\link{sits_classify.vector_cube}}.}
\item{\code{"entropy"} when using
   \code{\link{sits_uncertainty.probs_vector_cube}}.}
\item{\code{"class"} for cubes produced by
   \code{\link{sits_label_classification}}.}
}
}
\examples{
if (sits_run_examples()) {
    # --- Create a cube based on a local MODIS data
    # MODIS local files have names such as
    # "TERRA_MODIS_012010_NDVI_2013-09-14.jp2"
    # see the parse info parameter as an example on how to
    # decode local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    modis_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        data_dir = data_dir,
        parse_info = c("satellite", "sensor", "tile", "band", "date")
    )
    # segment the vector cube
    segs_cube <- sits_segment(
        cube = modis_cube,
        seg_fn = sits_slic(
            step = 10,
            compactness = 1,
            dist_fun = "euclidean",
            avg_fun = "median",
            iter = 30,
            minarea = 10
        ),
        output_dir = tempdir()
    )
    plot(segs_cube)

    # recover the local segmented cube
    local_segs_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        raster_cube = modis_cube,
        vector_dir = tempdir(),
        vector_band = "segments"
    )
    # plot the recover model and compare
    plot(local_segs_cube)

    # classify the segments
    # create a random forest model
    rfor_model <- sits_train(samples_modis_ndvi, sits_rfor())
    probs_vector_cube <- sits_classify(
        data = segs_cube,
        ml_model = rfor_model,
        output_dir = tempdir(),
        n_sam_pol = 10
    )
    plot(probs_vector_cube)

    # recover vector cube
    local_probs_vector_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        raster_cube = modis_cube,
        vector_dir = tempdir(),
        vector_band = "probs"
    )
    plot(local_probs_vector_cube)

    # label the segments
    class_vector_cube <- sits_label_classification(
        cube = probs_vector_cube,
        output_dir = tempdir(),
    )
    plot(class_vector_cube)

    # recover vector cube
    local_class_vector_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        raster_cube = modis_cube,
        vector_dir = tempdir(),
        vector_band = "class"
    )
    plot(local_class_vector_cube)
}

}
