#' @title creates a hierarchical structure
#' @name flow_chart
#' @author Luciane Ferreira Alcoforado
#'
#' @description Function to build the Diagram of hierarchies
#'
#' @param names is a vector with names for goal, criteria and choices in this exact sequence
#' @param c is a integer number of criteria, c>=2
#' @param a is a integer number of choices, a>=2
#' @return Returns Diagram of hierarchies
#' @examples
#' p=flow_chart(names=NULL, a=2, c=2)
#' p
#'
#' p=flow_chart(names=NULL, a=2, c=3)
#' p+ggplot2::theme_void()
#'
#' flow_chart(names=c("G", "cost", "time", "hour", "home", "beach"),c=3, a=2)
#'
#' @importFrom igraph graph_from_data_frame layout_as_tree vertex_attr
#' @importFrom rlang .data
#' @import ggplot2
#' @import dplyr
#' @importFrom tidyr pivot_longer
#' @importFrom tibble as_tibble
#' @export

flow_chart = function(names, c, a){
  if (any(c < 2))  stop("need positive value c >=2")
  if (any(a < 2)) stop("need positive value a >=2")
  
  if(is.null(names)) names=c("Goal", paste0("c",1:c), paste0("a",1:a))
  
  dt = data.frame(
    from = c(rep(names[1],c), rep(names[2:(c+1)],rep(a,c))),
    to = c(names[2:(c+1)], rep(names[(c+2):(c+a+1)],c))
  )
  
  g = igraph::graph_from_data_frame(dt, directed = TRUE)
  coords = igraph::layout_as_tree(g)
  colnames(coords) = c("x", "y")
  
  coords[(c+2): (c+a+1),1] = coords[(c+2): (c+a+1),1] + 1
  step_attr = igraph::vertex_attr(g, "name")
  tp = factor(c("Goal", rep("criteria",c), rep("choices",a)))
  
  output_df = tibble::as_tibble(coords)
  
  # Aqui usamos .data para o CRAN entender que são colunas
  output_df = output_df %>% dplyr::mutate(
    step = step_attr,
    label = step_attr,
    x = .data$x * 1,
    type = tp
  )
  
  plot_nodes = output_df %>%
    dplyr::mutate(
      xmin = .data$x - 0.25,
      xmax = .data$x + 0.25,
      ymin = .data$y - 0.25,
      ymax = .data$y + 0.25
    )
  
  plot_edges = dt %>%
    dplyr::mutate(id = dplyr::row_number()) %>%
    tidyr::pivot_longer(cols = c("from", "to"),
                        names_to = "s_e",
                        values_to = "step") %>%
    dplyr::left_join(plot_nodes, by = "step") %>%
    dplyr::select(-c(.data$label, .data$type, .data$y, .data$xmin, .data$xmax)) %>%
    dplyr::mutate(y = ifelse(.data$s_e == "from", .data$ymin, .data$ymax)) %>%
    dplyr::select(-c(.data$ymin, .data$ymax))
  
  p = ggplot2::ggplot() + 
    ggplot2::coord_fixed() +
    ggplot2::geom_rect(data = plot_nodes,
                       mapping = ggplot2::aes(xmin = .data$xmin, ymin = .data$ymin,
                                              xmax = .data$xmax, ymax = .data$ymax,
                                              fill = .data$type, colour = .data$type),
                       alpha = 0.5) +
    ggplot2::geom_text(data = plot_nodes,
                       mapping = ggplot2::aes(x = .data$x, y = .data$y, label = .data$label),
                       color = "#685c50") +
    ggplot2::geom_path(data = plot_edges,
                       mapping = ggplot2::aes(x = .data$x, y = .data$y, group = .data$id),
                       colour = "#685c50",
                       arrow = ggplot2::arrow(length = ggplot2::unit(0.2, "cm"), type = "closed")) +
    ggplot2::labs(title = "Hierarchical Tree of Decision",
                  caption = "R-package AHPWR, 2026") +
    ggplot2::theme(axis.text.x = ggplot2::element_blank(),
                   axis.ticks.x = ggplot2::element_blank(),
                   axis.text.y = ggplot2::element_blank(),
                   axis.ticks.y = ggplot2::element_blank())
  
  return(p)
}