% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameter_BKP.R, R/parameter_DKP.R
\name{parameter}
\alias{parameter}
\alias{parameter.BKP}
\alias{parameter.DKP}
\title{Extract Model Parameters from a Fitted BKP or DKP Model}
\usage{
parameter(object, ...)

\method{parameter}{BKP}(object, ...)

\method{parameter}{DKP}(object, ...)
}
\arguments{
\item{object}{An object of class \code{BKP} or \code{DKP}, typically the
result of a call to \code{\link{fit_BKP}} or \code{\link{fit_DKP}}.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A named list containing:
\itemize{
\item \code{theta}: Estimated kernel hyperparameters.
\item \code{alpha_n}: Posterior Dirichlet/Beta \eqn{\alpha} parameters.
\item \code{beta_n}: (BKP only) Posterior Beta \eqn{\beta} parameters.
}
}
\description{
Retrieve the key model parameters from a fitted \code{BKP} or
\code{DKP} object. For a \code{BKP} model, this typically includes the
optimized kernel hyperparameters and posterior Beta parameters. For a
\code{DKP} model, this includes the kernel hyperparameters and posterior
Dirichlet parameters.
}
\examples{
# -------------------------- BKP ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model <- fit_BKP(X, y, m, Xbounds = Xbounds)

# Extract posterior and kernel parameters
parameter(model)

# -------------------------- DKP ---------------------------
#' set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p1 <- 1/(1+exp(-3*X))
  p2 <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model <- fit_DKP(X, Y, Xbounds = Xbounds)

# Extract posterior quantiles
parameter(model)

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\seealso{
\code{\link{fit_BKP}} for fitting BKP models, \code{\link{fit_DKP}}
for fitting DKP models.
}
\keyword{BKP}
\keyword{DKP}
