% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/process-plate.R
\name{process_plate}
\alias{process_plate}
\title{Process a Plate and Save Normalised Output to CSV}
\usage{
process_plate(
  plate,
  filename = NULL,
  output_dir = "normalised_data",
  write_output = TRUE,
  normalisation_type = "RAU",
  data_type = "Median",
  blank_adjustment = FALSE,
  verbose = TRUE,
  reference_dilution = 1/400,
  ...
)
}
\arguments{
\item{plate}{(\code{Plate}) A plate object containing fluorescence intensity data.}

\item{filename}{(\code{character(1)}, optional) Output CSV filename. Defaults to \verb{\{plate_name\}_\{normalisation_type\}.csv}.
\itemize{
\item If omitted (\code{NULL}), the filename is auto-generated.
\item If the filename lacks \code{.csv}, the extension is automatically added.
\item If \code{output_dir} is specified, it is combined with \code{filename} unless \code{filename} is an absolute path.
\item If a file already exists at the path, it will be overwritten.
}}

\item{output_dir}{(\code{character(1)}, default = \code{'normalised_data'}) Directory for saving the output file.
\itemize{
\item If the directory does not exist, it will be created.
\item If \code{NULL}, the current working directory is used.
}}

\item{write_output}{(\code{logical(1)}, default = \code{TRUE}) Whether to save the output to a CSV file.}

\item{normalisation_type}{(\code{character(1)}, default = \code{'RAU'}) The normalisation method to apply.
\itemize{
\item Allowed values: \code{c(MFI, RAU, nMFI)}.
}}

\item{data_type}{(\code{character(1)}, default = \code{'Median'}) The data type to use for calculations.}

\item{blank_adjustment}{(\code{logical(1)}, default = \code{FALSE}) Whether to perform blank adjustment before computing values.}

\item{verbose}{(\code{logical(1)}, default = \code{TRUE}) Whether to print additional information during execution.}

\item{reference_dilution}{(\code{numeric(1)} or \code{character(1)}, default = \code{1/400})
\itemize{
\item The target dilution used as a reference for nMFI normalisation.
\item Ignored for RAU normalisation.
\item Can be numeric (e.g., \code{0.0025}) or a string (\code{'1/400'}).
}}

\item{...}{Additional arguments passed to the model fitting function (\code{create_standard_curve_model_analyte}).}
}
\value{
A data frame containing the computed normalised values.
}
\description{
This function processes a plate and computes normalised values based on the specified \code{normalisation_type}.
The function supports three types of normalisation:
\itemize{
\item \strong{RAU} (Relative Antibody Units) - Default normalisation type.
\item \strong{nMFI} (Normalised Median Fluorescence Intensity).
\item \strong{MFI} (Median Fluorescence Intensity, blank-adjusted raw MFI values).
}

Depending on the chosen normalisation type, the function:
\itemize{
\item Adjusts for blanks (if \code{blank_adjustment = TRUE}).
\item Computes the relevant normalised values for each analyte in the plate.
\item Aggregates computed values into a single data frame.
\item Optionally writes the results to a CSV file.
}

\strong{RAU Normalisation Workflow:}
\enumerate{
\item Blank adjustment (if enabled).
\item Fit standard curve models for each analyte.
\item Compute RAU values based on the fitted models.
\item Aggregate RAU values into a data frame.
\item Save the results to a CSV file.
}

\strong{nMFI Normalisation Workflow:}
\enumerate{
\item Blank adjustment (if enabled).
\item Compute nMFI values using the target dilution.
\item Aggregate nMFI values into a data frame.
\item Save the results to a CSV file.
}

\strong{MFI Normalisation Workflow:}
\enumerate{
\item Blank adjustment (if enabled).
\item Save the adjusted MFI values to a CSV file.
}

If the plate is already blank-adjusted, setting \code{blank_adjustment = TRUE} has no effect.

More details on the normalisation methods can be found in:
\itemize{
\item RAU: \link[SerolyzeR]{create_standard_curve_model_analyte}.
\item nMFI: \link[SerolyzeR]{get_nmfi}.
}
}
\examples{
plate_file <- system.file("extdata", "CovidOISExPONTENT_CO_reduced.csv", package = "SerolyzeR")
layout_file <- system.file("extdata", "CovidOISExPONTENT_CO_layout.xlsx", package = "SerolyzeR")
plate <- read_luminex_data(plate_file, layout_file, verbose = FALSE)
example_dir <- tempdir(check = TRUE)

# Process plate with default settings (RAU normalisation)
process_plate(plate, output_dir = example_dir)

# Process plate without blank adjustment, custom filename
process_plate(plate,
  filename = "plate_no_blank_adjustment.csv",
  output_dir = example_dir, blank_adjustment = FALSE
)

# Process plate with nMFI normalisation
process_plate(plate,
  output_dir = example_dir, normalisation_type = "nMFI",
  reference_dilution = 1 / 400
)

}
