/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 
#include "H5PBprivate.h" 

herr_t
H5F_shared_block_read(H5F_shared_t *f_sh, H5FD_mem_t type, haddr_t addr, size_t size, void *buf )
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert(buf);
    assert(H5_addr_defined(addr));

    
    if (H5_addr_le(f_sh->tmp_addr, (addr + size)))
        HGOTO_ERROR(H5E_IO, H5E_BADRANGE, FAIL, "attempting I/O in temporary file space");

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5PB_read(f_sh, map_type, addr, size, buf) < 0)
        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "read through page buffer failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_block_read(H5F_t *f, H5FD_mem_t type, haddr_t addr, size_t size, void *buf )
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(buf);
    assert(H5_addr_defined(addr));

    
    if (H5_addr_le(f->shared->tmp_addr, (addr + size)))
        HGOTO_ERROR(H5E_IO, H5E_BADRANGE, FAIL, "attempting I/O in temporary file space");

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5PB_read(f->shared, map_type, addr, size, buf) < 0)
        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "read through page buffer failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_shared_block_write(H5F_shared_t *f_sh, H5FD_mem_t type, haddr_t addr, size_t size, const void *buf)
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert(H5F_SHARED_INTENT(f_sh) & H5F_ACC_RDWR);
    assert(buf);
    assert(H5_addr_defined(addr));

    
    if (H5_addr_le(f_sh->tmp_addr, (addr + size)))
        HGOTO_ERROR(H5E_IO, H5E_BADRANGE, FAIL, "attempting I/O in temporary file space");

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5PB_write(f_sh, map_type, addr, size, buf) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "write through page buffer failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_block_write(H5F_t *f, H5FD_mem_t type, haddr_t addr, size_t size, const void *buf)
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(H5F_INTENT(f) & H5F_ACC_RDWR);
    assert(buf);
    assert(H5_addr_defined(addr));

    
    if (H5_addr_le(f->shared->tmp_addr, (addr + size)))
        HGOTO_ERROR(H5E_IO, H5E_BADRANGE, FAIL, "attempting I/O in temporary file space");

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5PB_write(f->shared, map_type, addr, size, buf) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "write through page buffer failed");
done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_shared_select_read(H5F_shared_t *f_sh, H5FD_mem_t type, uint32_t count, H5S_t **mem_spaces,
                       H5S_t **file_spaces, haddr_t offsets[], size_t element_sizes[], void *bufs[] )
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5FD_read_selection(f_sh->lf, map_type, count, mem_spaces, file_spaces, offsets, element_sizes,
                            bufs) < 0)
        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "selection read through file driver failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_shared_select_write(H5F_shared_t *f_sh, H5FD_mem_t type, uint32_t count, H5S_t **mem_spaces,
                        H5S_t **file_spaces, haddr_t offsets[], size_t element_sizes[], const void *bufs[])
{
    H5FD_mem_t map_type;            
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    map_type = (type == H5FD_MEM_GHEAP) ? H5FD_MEM_DRAW : type;

    
    if (H5FD_write_selection(f_sh->lf, map_type, count, mem_spaces, file_spaces, offsets, element_sizes,
                             bufs) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "selection write through file driver failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_shared_vector_read(H5F_shared_t *f_sh, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                       size_t sizes[], void *bufs[])
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert((types) || (count == 0));
    assert((addrs) || (count == 0));
    assert((sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
#ifndef NDEBUG
    for (uint32_t i = 0; i < count; i++) {
        
        if (i > 0 && types[i] == H5FD_MEM_NOLIST)
            break;

        assert(types[i] != H5FD_MEM_GHEAP);
    }
#endif

    
    if (H5FD_read_vector(f_sh->lf, count, types, addrs, sizes, bufs) < 0)
        HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL, "vector read through file driver failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5F_shared_vector_write(H5F_shared_t *f_sh, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                        size_t sizes[], const void *bufs[])
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert((types) || (count == 0));
    assert((addrs) || (count == 0));
    assert((sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
#ifndef NDEBUG
    for (uint32_t i = 0; i < count; i++) {
        
        if (i > 0 && types[i] == H5FD_MEM_NOLIST)
            break;

        assert(types[i] != H5FD_MEM_GHEAP);
    }
#endif

    
    if (H5FD_write_vector(f_sh->lf, count, types, addrs, sizes, bufs) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vector write through file driver failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5F_flush_tagged_metadata(H5F_t *f, haddr_t tag)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5AC_flush_tagged_metadata(f, tag) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush tagged metadata");

    
    if (H5F__accum_reset(f->shared, true, false) < 0)
        HGOTO_ERROR(H5E_IO, H5E_CANTRESET, FAIL, "can't reset accumulator");

    
    if (H5FD_flush(f->shared->lf, false) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "low level flush failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__evict_cache_entries(H5F_t *f)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(f->shared);

    
    if (H5AC_evict(f) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "unable to evict all except pinned entries");

#ifndef NDEBUG
    {
        unsigned status = 0;
        uint32_t cur_num_entries;

        
        if (H5AC_get_entry_status(f, (haddr_t)0, &status) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to get entry status");

        
        if (!(status & H5AC_ES__IN_CACHE) || !(status & H5AC_ES__IS_PINNED))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to get entry status");

        
        if (H5AC_get_cache_size(f->shared->cache, NULL, NULL, NULL, &cur_num_entries) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5AC_get_cache_size() failed.");

        
        if (cur_num_entries != 1)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "number of cache entries is not correct");
    }
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_get_checksums(const uint8_t *buf, size_t buf_size, uint32_t *s_chksum , uint32_t *c_chksum )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(buf);
    assert(buf_size);

    
    if (buf_size < H5_SIZEOF_CHKSUM)
        HGOTO_ERROR(H5E_IO, H5E_BADVALUE, FAIL, "checksum buffer is smaller than expected");

    
    if (s_chksum) {
        const uint8_t *chk_p; 

        
        chk_p = buf + buf_size - H5_SIZEOF_CHKSUM;

        
        UINT32DECODE(chk_p, *s_chksum);
    } 

    
    if (c_chksum)
        *c_chksum = H5_checksum_metadata(buf, buf_size - H5_SIZEOF_CHKSUM, 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
