/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Imodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Gprivate.h"  
#include "H5Ipkg.h"      
#include "H5RSprivate.h" 
#include "H5Tprivate.h"  
#include "H5VLprivate.h" 

static int H5I__id_dump_cb(void *_item, void *_key, void *_udata);

static int
H5I__id_dump_cb(void *_item, void H5_ATTR_UNUSED *_key, void *_udata)
{
    H5I_id_info_t    *info   = (H5I_id_info_t *)_item; 
    H5I_type_t        type   = *(H5I_type_t *)_udata;  
    const H5G_name_t *path   = NULL;                   
    void             *object = NULL;                   
    bool              is_native;                       

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(Rstderr, "         id = %" PRIdHID "\n", info->id);
    Rfprintf(Rstderr, "         count = %u\n", info->count);
    Rfprintf(Rstderr, "         obj   = 0x%8p\n", info->u.c_object);
    Rfprintf(Rstderr, "         marked = %d\n", info->marked);

    
    switch (type) {
        case H5I_GROUP: {
            const H5VL_object_t *vol_obj = (const H5VL_object_t *)info->u.c_object;

            is_native = false;
            H5VL_object_is_native(vol_obj, &is_native);
            if (is_native)
                path = H5G_nameof(object);
            break;
        }

        case H5I_DATASET: {
            const H5VL_object_t *vol_obj = (const H5VL_object_t *)info->u.c_object;

            is_native = false;
            H5VL_object_is_native(vol_obj, &is_native);
            if (is_native)
                path = H5D_nameof(object);
            break;
        }

        case H5I_DATATYPE: {
            H5T_t *dt = info->u.object;

            object = H5T_get_actual_type((H5T_t *)dt);
            path   = H5T_nameof(object);
            break;
        }

        
        case H5I_MAP:

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            break; 
    }

    if (path) {
        if (path->user_path_r)
            Rfprintf(Rstderr, "                user_path = %s\n", H5RS_get_str(path->user_path_r));
        if (path->full_path_r)
            Rfprintf(Rstderr, "                full_path = %s\n", H5RS_get_str(path->full_path_r));
    }

    FUNC_LEAVE_NOAPI(H5_ITER_CONT)
} 

herr_t
H5I_dump_ids_for_type(H5I_type_t type)
{
    H5I_type_info_t *type_info = NULL;

    FUNC_ENTER_NOAPI_NOERR

    Rfprintf(Rstderr, "Dumping ID type %d\n", (int)type);
    type_info = H5I_type_info_array_g[type];

    if (type_info) {

        H5I_id_info_t *item = NULL;
        H5I_id_info_t *tmp  = NULL;

        
        Rfprintf(Rstderr, "     init_count = %u\n", type_info->init_count);
        Rfprintf(Rstderr, "     reserved   = %u\n", type_info->cls->reserved);
        Rfprintf(Rstderr, "     id_count   = %llu\n", (unsigned long long)type_info->id_count);
        Rfprintf(Rstderr, "     nextid        = %llu\n", (unsigned long long)type_info->nextid);

        
        if (type_info->id_count > 0) {
            Rfprintf(Rstderr, "     List:\n");
            
            Rfprintf(Rstderr, "     (HASH TABLE)\n");
            HASH_ITER(hh, type_info->hash_table, item, tmp)
            {
                H5I__id_dump_cb((void *)item, NULL, (void *)&type);
            }
        }
    }
    else
        Rfprintf(Rstderr, "Global type info/tracking pointer for that type is NULL\n");

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
