/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__ginfo_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static herr_t H5O__ginfo_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                const void *_mesg);
static void  *H5O__ginfo_copy(const void *_mesg, void *_dest);
static size_t H5O__ginfo_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__ginfo_free(void *_mesg);
static herr_t H5O__ginfo_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_GINFO[1] = {{
    H5O_GINFO_ID,        
    "ginfo",             
    sizeof(H5O_ginfo_t), 
    0,                   
    H5O__ginfo_decode,   
    H5O__ginfo_encode,   
    H5O__ginfo_copy,     
    H5O__ginfo_size,     
    NULL,                
    H5O__ginfo_free,     
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    NULL,                
    H5O__ginfo_debug     
}};

#define H5O_GINFO_VERSION 0

#define H5O_GINFO_STORE_PHASE_CHANGE   0x01
#define H5O_GINFO_STORE_EST_ENTRY_INFO 0x02
#define H5O_GINFO_ALL_FLAGS            (H5O_GINFO_STORE_PHASE_CHANGE | H5O_GINFO_STORE_EST_ENTRY_INFO)

H5FL_DEFINE_STATIC(H5O_ginfo_t);

static void *
H5O__ginfo_decode(H5F_t H5_ATTR_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                  unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_ginfo_t   *ginfo = NULL;               
    unsigned char  flags;                      
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_GINFO_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (ginfo = H5FL_CALLOC(H5O_ginfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

    flags = *p++;
    if (flags & ~H5O_GINFO_ALL_FLAGS)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad flag value for message");
    ginfo->store_link_phase_change = (flags & H5O_GINFO_STORE_PHASE_CHANGE) ? true : false;
    ginfo->store_est_entry_info    = (flags & H5O_GINFO_STORE_EST_ENTRY_INFO) ? true : false;

    
    if (ginfo->store_link_phase_change) {
        if (H5_IS_BUFFER_OVERFLOW(p, 2 * 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, ginfo->max_compact);
        UINT16DECODE(p, ginfo->min_dense);
    }
    else {
        ginfo->max_compact = H5G_CRT_GINFO_MAX_COMPACT;
        ginfo->min_dense   = H5G_CRT_GINFO_MIN_DENSE;
    }

    
    if (ginfo->store_est_entry_info) {
        if (H5_IS_BUFFER_OVERFLOW(p, 2 * 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, ginfo->est_num_entries);
        UINT16DECODE(p, ginfo->est_name_len);
    }
    else {
        ginfo->est_num_entries = H5G_CRT_GINFO_EST_NUM_ENTRIES;
        ginfo->est_name_len    = H5G_CRT_GINFO_EST_NAME_LEN;
    }

    
    ret_value = ginfo;

done:
    if (!ret_value && ginfo)
        H5FL_FREE(H5O_ginfo_t, ginfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ginfo_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size,
                  uint8_t *p, const void *_mesg)
{
    const H5O_ginfo_t *ginfo = (const H5O_ginfo_t *)_mesg;
    unsigned char      flags = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(p);
    assert(ginfo);

    
    *p++ = H5O_GINFO_VERSION;

    
    flags = (unsigned char)(ginfo->store_link_phase_change ? H5O_GINFO_STORE_PHASE_CHANGE : 0);
    flags = (unsigned char)(flags | (ginfo->store_est_entry_info ? H5O_GINFO_STORE_EST_ENTRY_INFO : 0));
    *p++  = flags;

    
    if (ginfo->store_link_phase_change) {
        UINT16ENCODE(p, ginfo->max_compact);
        UINT16ENCODE(p, ginfo->min_dense);
    } 

    
    if (ginfo->store_est_entry_info) {
        UINT16ENCODE(p, ginfo->est_num_entries);
        UINT16ENCODE(p, ginfo->est_name_len);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__ginfo_copy(const void *_mesg, void *_dest)
{
    const H5O_ginfo_t *ginfo     = (const H5O_ginfo_t *)_mesg;
    H5O_ginfo_t       *dest      = (H5O_ginfo_t *)_dest;
    void              *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(ginfo);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_ginfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *ginfo;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__ginfo_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_ginfo_t *ginfo     = (const H5O_ginfo_t *)_mesg;
    size_t             ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = 1 +                                             
                1 +                                             
                (ginfo->store_link_phase_change ? ((size_t)(2 + 
                                                            2)  
                                                   )
                                                : 0) +       
                (ginfo->store_est_entry_info ? ((size_t)(2 + 
                                                         2)  
                                                )
                                             : 0);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ginfo_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_ginfo_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__ginfo_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_ginfo_t *ginfo = (const H5O_ginfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(ginfo);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Max. compact links:", ginfo->max_compact);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Min. dense links:", ginfo->min_dense);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Estimated # of objects in group:", ginfo->est_num_entries);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Estimated length of object in group's name:", ginfo->est_name_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
