/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Tpkg.h"     

herr_t
H5T__visit(H5T_t *dt, unsigned visit_flags, H5T_operator_t op, void *op_value)
{
    bool   is_composite;        
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(op);

    
    is_composite = H5T_IS_COMPOSITE(dt->shared->type);

    
    if (is_composite && (visit_flags & H5T_VISIT_COMPOSITE_FIRST))
        if (op(dt, op_value) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "operator callback failed");

    
    switch (dt->shared->type) {
        case H5T_COMPOUND: {
            unsigned u; 

            
            for (u = 0; u < dt->shared->u.compnd.nmembs; u++)
                if (H5T__visit(dt->shared->u.compnd.memb[u].type, visit_flags, op, op_value) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "can't visit member datatype");
        } 
        break;

        case H5T_ARRAY:
        case H5T_VLEN:
        case H5T_ENUM:
        case H5T_COMPLEX:
            
            if (H5T__visit(dt->shared->parent, visit_flags, op, op_value) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "can't visit parent datatype");
            break;

        case H5T_NO_CLASS:
        case H5T_NCLASSES:
            
            HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL, "operation not defined for datatype class");
            break;

        case H5T_INTEGER:
        case H5T_FLOAT:
        case H5T_TIME:
        case H5T_STRING:
        case H5T_BITFIELD:
        case H5T_OPAQUE:
        case H5T_REFERENCE:
        default:
            
            if (visit_flags & H5T_VISIT_SIMPLE)
                if (op(dt, op_value) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "operator callback failed");
            break;
    } 

    
    if (is_composite && (visit_flags & H5T_VISIT_COMPOSITE_LAST))
        if (op(dt, op_value) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADITER, FAIL, "operator callback failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
