% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/losses.R
\name{loss_binary_crossentropy}
\alias{loss_binary_crossentropy}
\title{Computes the cross-entropy loss between true labels and predicted labels.}
\usage{
loss_binary_crossentropy(
  y_true,
  y_pred,
  from_logits = FALSE,
  label_smoothing = 0,
  axis = -1L,
  ...,
  reduction = "sum_over_batch_size",
  name = "binary_crossentropy",
  dtype = NULL
)
}
\arguments{
\item{y_true}{Ground truth values. shape = \verb{[batch_size, d0, .. dN]}.}

\item{y_pred}{The predicted values. shape = \verb{[batch_size, d0, .. dN]}.}

\item{from_logits}{Whether to interpret \code{y_pred} as a tensor of
\href{https://en.wikipedia.org/wiki/Logit}{logit} values. By default, we
assume that \code{y_pred} is probabilities (i.e., values in \verb{[0, 1)).}}

\item{label_smoothing}{Float in range \verb{[0, 1].} When 0, no smoothing occurs.
When > 0, we compute the loss between the predicted labels
and a smoothed version of the true labels, where the smoothing
squeezes the labels towards 0.5. Larger values of
\code{label_smoothing} correspond to heavier smoothing.}

\item{axis}{The axis along which to compute crossentropy (the features axis).
Defaults to \code{-1}.}

\item{...}{For forward/backward compatability.}

\item{reduction}{Type of reduction to apply to the loss. In almost all cases
this should be \code{"sum_over_batch_size"}. Supported options are
\code{"sum"}, \code{"sum_over_batch_size"}, \code{"mean"},
\code{"mean_with_sample_weight"} or \code{NULL}. \code{"sum"} sums the loss,
\code{"sum_over_batch_size"} and \code{"mean"} sum the loss and divide by the
sample size, and \code{"mean_with_sample_weight"} sums the loss and
divides by the sum of the sample weights. \code{"none"} and \code{NULL}
perform no aggregation. Defaults to \code{"sum_over_batch_size"}.}

\item{name}{Optional name for the loss instance.}

\item{dtype}{The dtype of the loss's computations. Defaults to \code{NULL}, which
means using \code{config_floatx()}. \code{config_floatx()} is a
\code{"float32"} unless set to different value
(via \code{config_set_floatx()}). If a \code{keras$DTypePolicy} is
provided, then the \code{compute_dtype} will be utilized.}
}
\value{
Binary crossentropy loss value. shape = \verb{[batch_size, d0, .. dN-1]}.
}
\description{
Use this cross-entropy loss for binary (0 or 1) classification applications.
The loss function requires the following inputs:
\itemize{
\item \code{y_true} (true label): This is either 0 or 1.
\item \code{y_pred} (predicted value): This is the model's prediction, i.e, a single
floating-point value which either represents a
\href{https://en.wikipedia.org/wiki/Logit}{logit}, (i.e, value in \verb{[-inf, inf]}
when \code{from_logits=TRUE}) or a probability (i.e, value in \verb{[0., 1.]} when
\code{from_logits=FALSE}).
}
}
\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{y_true <- rbind(c(0, 1), c(0, 0))
y_pred <- rbind(c(0.6, 0.4), c(0.4, 0.6))
loss <- loss_binary_crossentropy(y_true, y_pred)
loss
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([0.91629076 0.7135582 ], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}

\strong{Recommended Usage:} (set \code{from_logits=TRUE})

With \code{compile()} API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{model \%>\% compile(
    loss = loss_binary_crossentropy(from_logits=TRUE),
    ...
)
}\if{html}{\out{</div>}}

As a standalone function:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Example 1: (batch_size = 1, number of samples = 4)
y_true <- op_array(c(0, 1, 0, 0))
y_pred <- op_array(c(-18.6, 0.51, 2.94, -12.8))
bce <- loss_binary_crossentropy(from_logits = TRUE)
bce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.865458, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Example 2: (batch_size = 2, number of samples = 4)
y_true <- rbind(c(0, 1), c(0, 0))
y_pred <- rbind(c(-18.6, 0.51), c(2.94, -12.8))
# Using default 'auto'/'sum_over_batch_size' reduction type.
bce <- loss_binary_crossentropy(from_logits = TRUE)
bce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.865458, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'sample_weight' attribute
bce(y_true, y_pred, sample_weight = c(0.8, 0.2))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.2436386, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 0.243
# Using 'sum' reduction` type.
bce <- loss_binary_crossentropy(from_logits = TRUE, reduction = "sum")
bce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(1.730916, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'none' reduction type.
bce <- loss_binary_crossentropy(from_logits = TRUE, reduction = NULL)
bce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([0.23515666 1.4957594 ], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}

\strong{Default Usage:} (set \code{from_logits=FALSE})

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Make the following updates to the above "Recommended Usage" section
# 1. Set `from_logits=FALSE`
loss_binary_crossentropy() # OR ...('from_logits=FALSE')
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <LossFunctionWrapper(<function binary_crossentropy at 0x0>, kwargs=\{'from_logits': False, 'label_smoothing': 0.0, 'axis': -1\})>
##  signature: (y_true, y_pred, sample_weight=None)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 2. Update `y_pred` to use probabilities instead of logits
y_pred <- c(0.6, 0.3, 0.2, 0.8) # OR [[0.6, 0.3], [0.2, 0.8]]
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/losses/probabilistic_losses#binarycrossentropy-class}
}

Other losses: \cr
\code{\link{Loss}()} \cr
\code{\link{loss_binary_focal_crossentropy}()} \cr
\code{\link{loss_categorical_crossentropy}()} \cr
\code{\link{loss_categorical_focal_crossentropy}()} \cr
\code{\link{loss_categorical_hinge}()} \cr
\code{\link{loss_circle}()} \cr
\code{\link{loss_cosine_similarity}()} \cr
\code{\link{loss_ctc}()} \cr
\code{\link{loss_dice}()} \cr
\code{\link{loss_hinge}()} \cr
\code{\link{loss_huber}()} \cr
\code{\link{loss_kl_divergence}()} \cr
\code{\link{loss_log_cosh}()} \cr
\code{\link{loss_mean_absolute_error}()} \cr
\code{\link{loss_mean_absolute_percentage_error}()} \cr
\code{\link{loss_mean_squared_error}()} \cr
\code{\link{loss_mean_squared_logarithmic_error}()} \cr
\code{\link{loss_poisson}()} \cr
\code{\link{loss_sparse_categorical_crossentropy}()} \cr
\code{\link{loss_squared_hinge}()} \cr
\code{\link{loss_tversky}()} \cr
\code{\link{metric_binary_crossentropy}()} \cr
\code{\link{metric_binary_focal_crossentropy}()} \cr
\code{\link{metric_categorical_crossentropy}()} \cr
\code{\link{metric_categorical_focal_crossentropy}()} \cr
\code{\link{metric_categorical_hinge}()} \cr
\code{\link{metric_hinge}()} \cr
\code{\link{metric_huber}()} \cr
\code{\link{metric_kl_divergence}()} \cr
\code{\link{metric_log_cosh}()} \cr
\code{\link{metric_mean_absolute_error}()} \cr
\code{\link{metric_mean_absolute_percentage_error}()} \cr
\code{\link{metric_mean_squared_error}()} \cr
\code{\link{metric_mean_squared_logarithmic_error}()} \cr
\code{\link{metric_poisson}()} \cr
\code{\link{metric_sparse_categorical_crossentropy}()} \cr
\code{\link{metric_squared_hinge}()} \cr
}
\concept{losses}
