% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_ABC.R
\name{modelSelect}
\alias{modelSelect}
\title{Perform model selection with Approximate Bayesian Computation}
\usage{
modelSelect(target, index, sumstats, tol, method, warning = TRUE)
}
\arguments{
\item{target}{is a vector with the target summary statistics. These are
usually computed from observed data.}

\item{index}{is a vector of model indices. This can be a a character vector
of model names, repeated as many times as there are simulations for each
model. This vector will be coerced to factor and it must have the same
length as \code{nrow(sumstats)} to indicate which row of the \code{sumstats}
matrix belongs to which model.}

\item{sumstats}{is a vector or matrix containing the simulated summary
statistics for all the models. Each row or vector entry should be a
different simulation and each column of a matrix should be a different
statistic. The order must be the same as the order of the models in the
\code{index} vector.}

\item{tol}{is a numerical value, indicating the required proportion of points
nearest the target values (tolerance).}

\item{method}{a character string, either "rejection" or "regression",
indicating which algorithm should be used for model selection.}

\item{warning}{logical, if TRUE (default) warnings produced while running
this function, mainly related with accepting simulations for just one of
the models, will be displayed.}
}
\value{
a list with the following elements:

\item{method}{the method used for model selection.}

\item{indices}{a vector of model indices in the accepted region. In other
words, this vector contains the name of the accepted model for each
accepted point.}

\item{pred}{a vector of model probabilities.}

\item{ss}{the summary statistics in the accepted region.}

\item{weights}{vector of regression weights when method is regression.}

\item{nmodels}{the number of a priori simulations performed for each
model.}
}
\description{
Estimates posterior model probabilities using Approximate Bayesian
Computation (ABC).
}
\details{
Prior to using this function, simulations must have been performed under, at
least, two different models. When \code{method} is "rejection", the posterior
posterior probability of a given model is approximated by the proportion of
accepted simulations of that particular model. Note that this approximation
is only valid if all models where, a priori, equally likely and if the number
of simulations performed is the same for all models. When the \code{method} is set
to "regression", a multinomial logistic regression is used to estimate the
posterior model probabilities. This multinomial regression is implemented in
the \link[nnet]{multinom} function.
}
\examples{
# load the matrix with simulated parameter values
data(sumstats)

# select a random simulation to act as target just to test the function
target <- sumstats[10 ,]

# create a "fake" vector of model indices
# this assumes that half the simulations were from one model and the other half from other model
# this is not true but serves as an example of how to use this function
index <- c(rep("model1", nrow(sumstats)/2), rep("model2", nrow(sumstats)/2))

# perform model selection with ABC
modelSelect(target = target, index = index, sumstats = sumstats, tol = 0.01, method = "regression")

}
