% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_diff_abundance.R
\name{calculate_diff_abundance}
\alias{calculate_diff_abundance}
\title{Calculate differential abundance between conditions}
\usage{
calculate_diff_abundance(
  data,
  sample,
  condition,
  grouping,
  intensity_log2,
  missingness = missingness,
  comparison = comparison,
  mean = NULL,
  sd = NULL,
  n_samples = NULL,
  ref_condition = "all",
  filter_NA_missingness = TRUE,
  method = c("moderated_t-test", "t-test", "t-test_mean_sd", "proDA"),
  p_adj_method = "BH",
  retain_columns = NULL
)
}
\arguments{
\item{data}{a data frame containing at least the input variables that are required for the
selected method. Ideally the output of \code{assign_missingness} or \code{impute} is used.}

\item{sample}{a character column in the \code{data} data frame that contains the sample name.
Is not required if \code{method = "t-test_mean_sd"}.}

\item{condition}{a character or numeric column in the \code{data} data frame that contains the
conditions.}

\item{grouping}{a character column in the \code{data} data frame that contains precursor,
peptide or protein identifiers.}

\item{intensity_log2}{a numeric column in the \code{data} data frame that contains intensity
values. The intensity values need to be log2 transformed. Is not required if
\code{method = "t-test_mean_sd"}.}

\item{missingness}{a character column in the \code{data} data frame that contains missingness
information. Can be obtained by calling \code{assign_missingness()}. Is not required if
\code{method = "t-test_mean_sd"}. The type of missingness assigned to a comparison does not have
any influence on the statistical test. However, if \code{filter_NA_missingness = TRUE} and
\code{method = "proDA"}, then comparisons with missingness \code{NA} are filtered out prior
to p-value adjustment.}

\item{comparison}{a character column in the \code{data} data frame that contains information of
treatment/reference condition pairs. Can be obtained by calling \code{assign_missingness}.
Comparisons need to be in the form condition1_vs_condition2, meaning two compared conditions are
separated by \code{"_vs_"}. This column determines for which condition pairs differential
abundances are calculated. Is not required if \code{method = "t-test_mean_sd"}, in that case
please provide a reference condition with the ref_condition argument.}

\item{mean}{a numeric column in the \code{data} data frame that contains mean values for two
conditions. Is only required if \code{method = "t-test_mean_sd"}.}

\item{sd}{a numeric column in the \code{data} data frame that contains standard deviations for
two conditions. Is only required if \code{method = "t-test_mean_sd"}.}

\item{n_samples}{a numeric column in the \code{data} data frame that contains the number of
samples per condition for two conditions. Is only required if \code{method = "t-test_mean_sd"}.}

\item{ref_condition}{optional, character value providing the condition that is used as a
reference for differential abundance calculation. Only required for \code{method = "t-test_mean_sd"}.
Instead of providing one reference condition, "all" can be supplied, which will create all
pairwise condition pairs. By default \code{ref_condition = "all"}.}

\item{filter_NA_missingness}{a logical value, default is \code{TRUE}. For all methods except
\code{"t-test_mean_sd"} missingness information has to be provided. This information can be
for example obtained by calling \code{assign_missingness()}. If a reference/treatment pair has
too few samples to be considered robust based on user defined cutoffs, it is annotated with \code{NA}
as missingness by the \code{assign_missingness()} function. If this argument is \code{TRUE},
these \code{NA} reference/treatment pairs are filtered out. For \code{method = "proDA"} this
is done before the p-value adjustment.}

\item{method}{a character value, specifies the method used for statistical hypothesis testing.
Methods include Welch test (\code{"t-test"}), a Welch test on means, standard deviations and
number of replicates (\code{"t-test_mean_sd"}) and a moderated t-test based on the \code{limma}
package (\code{"moderated_t-test"}). More information on the moderated t-test can be found in
the \code{limma} documentation. Furthermore, the \code{proDA} package specific method (\code{"proDA"})
can be used to infer means across samples based on a probabilistic dropout model. This
eliminates the need for data imputation since missing values are inferred from the model. More
information can be found in the \code{proDA} documentation. We do not recommend using the
\code{moderated_t-test} or \code{proDA} method if the data was filtered for low CVs or imputation
was performed. Default is \code{method = "moderated_t-test"}.}

\item{p_adj_method}{a character value, specifies the p-value correction method. Possible
methods are c("holm", "hochberg", "hommel", "bonferroni", "BH", "BY", "fdr", "none"). Default
method is \code{"BH"}.}

\item{retain_columns}{a vector indicating if certain columns should be retained from the input
data frame. Default is not retaining additional columns \code{retain_columns = NULL}. Specific
columns can be retained by providing their names (not in quotations marks, just like other
column names, but in a vector). Please note that if you retain columns that have multiple
rows per grouped variable there will be duplicated rows in the output.}
}
\value{
A data frame that contains differential abundances (\code{diff}), p-values (\code{pval})
and adjusted p-values (\code{adj_pval}) for each protein, peptide or precursor (depending on
the \code{grouping} variable) and the associated treatment/reference pair. Depending on the
method the data frame contains additional columns:
\itemize{
\item "t-test": The \code{std_error} column contains the standard error of the differential
abundances. \code{n_obs} contains the number of observations for the specific protein, peptide
or precursor (depending on the \code{grouping} variable) and the associated treatment/reference pair.
\item "t-test_mean_sd": Columns labeled as control refer to the second condition of the
comparison pairs. Treated refers to the first condition. \code{mean_control} and \code{mean_treated}
columns contain the means for the reference and treatment condition, respectively. \code{sd_control}
and \code{sd_treated} columns contain the standard deviations for the reference and treatment
condition, respectively. \code{n_control} and \code{n_treated} columns contain the numbers of
samples for the reference and treatment condition, respectively. The \code{std_error} column
contains the standard error of the differential abundances. \code{t_statistic} contains the
t_statistic for the t-test.
\item "moderated_t-test": \code{CI_2.5} and \code{CI_97.5} contain the 2.5\% and 97.5\%
confidence interval borders for differential abundances. \code{avg_abundance} contains average
abundances for treatment/reference pairs (mean of the two group means). \code{t_statistic}
contains the t_statistic for the t-test. \code{B} The B-statistic is the log-odds that the
protein, peptide or precursor (depending on \code{grouping}) has a differential abundance
between the two groups. Suppose B=1.5. The odds of differential abundance is exp(1.5)=4.48, i.e,
about four and a half to one. The probability that there is a differential abundance is
4.48/(1+4.48)=0.82, i.e., the probability is about 82\% that this group is differentially
abundant. A B-statistic of zero corresponds to a 50-50 chance that the group is differentially
abundant.\code{n_obs} contains the number of observations for the specific protein, peptide or
precursor (depending on the \code{grouping} variable) and the associated treatment/reference pair.
\item "proDA": The \code{std_error} column contains the standard error of the differential
abundances. \code{avg_abundance} contains average abundances for treatment/reference pairs
(mean of the two group means). \code{t_statistic} contains the t_statistic for the t-test.
\code{n_obs} contains the number of observations for the specific protein, peptide or precursor
(depending on the \code{grouping} variable) and the associated treatment/reference pair.
}

For all methods execept \code{"proDA"}, the p-value adjustment is performed only on the
proportion of data that contains a p-value that is not \code{NA}. For \code{"proDA"} the
p-value adjustment is either performed on the complete dataset (\code{filter_NA_missingness = TRUE})
or on the subset of the dataset with missingness that is not \code{NA} (\code{filter_NA_missingness = FALSE}).
}
\description{
Performs differential abundance calculations and statistical hypothesis tests on data frames
with protein, peptide or precursor data. Different methods for statistical testing are available.
}
\examples{
set.seed(123) # Makes example reproducible

# Create synthetic data
data <- create_synthetic_data(
  n_proteins = 10,
  frac_change = 0.5,
  n_replicates = 4,
  n_conditions = 2,
  method = "effect_random",
  additional_metadata = FALSE
)

# Assign missingness information
data_missing <- assign_missingness(
  data,
  sample = sample,
  condition = condition,
  grouping = peptide,
  intensity = peptide_intensity_missing,
  ref_condition = "all",
  retain_columns = c(protein, change_peptide)
)

# Calculate differential abundances
# Using "moderated_t-test" and "proDA" improves
# true positive recovery progressively
diff <- calculate_diff_abundance(
  data = data_missing,
  sample = sample,
  condition = condition,
  grouping = peptide,
  intensity_log2 = peptide_intensity_missing,
  missingness = missingness,
  comparison = comparison,
  method = "t-test",
  retain_columns = c(protein, change_peptide)
)

head(diff, n = 10)
}
