% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/store_boot.R
\name{store_boot}
\alias{store_boot}
\title{Compute and Store Bootstrap Estimates}
\usage{
store_boot(
  object,
  type = "std.all",
  do_bootstrapping = TRUE,
  R = 1000,
  boot_type = "ordinary",
  parallel = c("no", "multicore", "snow"),
  ncpus = parallel::detectCores(logical = FALSE) - 1,
  iseed = NULL,
  keep.idx = FALSE,
  bootstrapLavaan_args = list()
)
}
\arguments{
\item{object}{A 'lavaan'-class
object, fitted with 'se = "boot"'.}

\item{type}{The type of standard
estimates. The same argument of
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
and support all values supported by
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.
Default is \code{"std.all"}.}

\item{do_bootstrapping}{If \code{TRUE} and
bootstrapping was not requested when
fitting the model, bootstrapping
will be done using
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}. Default
is \code{TRUE}.}

\item{R}{If \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
is called (see \code{do_bootstrapping}),
this is the number of bootstrap
samples, to be used by
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.}

\item{boot_type}{If \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
is called (see \code{do_bootstrapping}),
this is type of bootstrapping,
to be passed to the argument \code{type}
of
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Default is \code{"ordinary"}. See the
help page of \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
for details.}

\item{parallel}{If \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
is called (see \code{do_bootstrapping}),
whether parallel processing will
be used.
to be passed to the argument of the
same name in \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Default is \code{"no"}. Can be
\code{"snow"} or \code{"multicore"}. See the
help page of \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
for details.}

\item{ncpus}{If \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
is called (see \code{do_bootstrapping}),
and parallel processing is to be used,
this is the number of CPU cores to
use, to be passed to the argument of the
same name in \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Default is \code{parallel::detectCores(logical = FALSE) - 1},
the number of physical cores minus 1,
different from the default of
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} but identical
to the default of \code{\link[lavaan:sem]{lavaan::sem()}} and
\code{\link[lavaan:cfa]{lavaan::cfa()}}.}

\item{iseed}{If \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
is called (see \code{do_bootstrapping}),
this should be an integer used to
generate reproducible bootstrap
results, to be passed to the argument of the
same name in \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Default is \code{NULL} but it should nearly
always be set to an arbitrary integer.
See the
help page of \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}
for details.}

\item{keep.idx}{Whether the indices
of cases selected in each bootstrap
sample is to be stored. To be passed
to the argument of the same name
in \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Default is \code{FALSE}.}

\item{bootstrapLavaan_args}{A named
list of additional arguments to be
passed to \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}.
Note that the other arguments in
\code{\link[=store_boot]{store_boot()}} takes precedence,
overriding arguments of the same
names in this list, if any.}
}
\value{
The original \code{lavaan} object is
returned with the following objects
stored in the \code{external} slot:
\itemize{
\item \code{sbt_boot_std}: The matrix of
bootstrap estimates in the
standardized solution.
\item \code{sbt_boot_def}: The matrix of
bootstrap estimates of user-defined
parameters, if any.
\item \code{sbt_boot_ustd}: The matrix of
bootstrap estimates of free
parameters, if bootstrapping is
not requested when fitting the
model (i.e., \code{se} is not set to
\code{"boot"} or \code{"bootstrap"} when
fitting the model in \code{lavaan}).
}
}
\description{
This function computes bootstrap estimates
of a fitted structural equation model
and stores the estimates for further
processing.
}
\details{
The function \code{\link[=store_boot]{store_boot()}}
receives a
\link[lavaan:lavaan-class]{lavaan::lavaan} object, optionally
fitted with bootstrapping standard errors
requested, and compute and store
the bootstrap estimates of user-defined
parameters and estimates in the
standardized solution.

If bootstrapping was not requested
when fitting the model (i.e., \code{se}
not set to \code{"boot"} or \code{"bootstrap"}),
then bootstrapping will be conducted
using \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} to
compute bootstrap estimates of free
parameters. Otherwise, the stored
bootstrap estimates will be used in
subsequent steps.

For standardized solution bootstrap
estimates, it works by calling
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}
with the bootstrap estimates
of free parameters in each bootstrap sample
to compute the standardized estimates
in each sample.

For user-defined parameters, it works
by calling the function used to
compute user-defined parameters with
the bootstrap estimates of free
parameters in each bootstrap samples
to compute the user-defined parameters.

The bootstrap estimates are then
stored in the \code{external} slot
of the fit object for further
processing.
}
\examples{

library(lavaan)
set.seed(5478374)
n <- 50
x <- runif(n) - .5
m <- .40 * x + rnorm(n, 0, sqrt(1 - .40))
y <- .30 * m + rnorm(n, 0, sqrt(1 - .30))
dat <- data.frame(x = x, y = y, m = m)
model <-
'
m ~ a*x
y ~ b*m
ab := a*b
'

# Should set bootstrap to at least 2000 in real studies
fit <- sem(model, data = dat, fixed.x = FALSE,
           se = "boot",
           bootstrap = 100)
summary(fit)

fit <- store_boot(fit)

}
\author{
Shu Fai Cheung
\url{https://orcid.org/0000-0002-9871-9448}.
Based on \code{\link[semhelpinghands:standardizedSolution_boot_ci]{semhelpinghands::standardizedSolution_boot_ci()}},
which was originally proposed in an issue at GitHub
\url{https://github.com/simsem/semTools/issues/101#issue-1021974657},
inspired by a discussion at
the Google group for lavaan
\url{https://groups.google.com/g/lavaan/c/qQBXSz5cd0o/m/R8YT5HxNAgAJ}.
Unlike \code{\link[semhelpinghands:standardizedSolution_boot_ci]{semhelpinghands::standardizedSolution_boot_ci()}},
this function only computes and stores
the bootstrap estimates.
}
