\name{print.feasible}
\alias{print.feasible}
\title{Print method for \code{feasible} objects}
\description{
Produces a concisey summary of a feasible intervention object.
The printout summarizes information jointly over time and strategy,
using tables with strategies as rows and time points as columns. Separate
tables are printed for the proportion of infeasible mass (\code{\%infeasible})
and the mean feasible value (\code{Feasible}).
}
\usage{
\method{print}{feasible}(x, digits = 3, strategies = "all", times = "all", ...)
}
\arguments{
  \item{x}{A \code{"feasible"} object returned by \code{\link{feasible}}.}

  \item{digits}{Integer; number of digits used when printing numeric values.}

  \item{strategies}{Either \code{"all"} (default) or a numeric vector of strategy
    indices to include in the printed summary. When a numeric vector is supplied,
    all summaries and tables are restricted to these strategies.}

  \item{times}{Either \code{"all"} (default) or a numeric vector of time indices
    to include in the printed summary. When a numeric vector is supplied,
    all summaries and tables are restricted to these time points.}

  \item{...}{Ignored; provided for S3 method compatibility.}
}
\details{
The method extracts the \code{data.frame} stored in the \code{"summary"} attribute
of \code{x} and optionally restricts it to the selected \code{strategies}
and \code{times}. All reported values are based on this restricted data.

The summary data typically contains at least the following columns:
\itemize{
  \item \strong{time}: time index \code{t}.
  \item \strong{Strategy}: index of the intervention strategy.
  \item \strong{Abar}: target intervention value at time \code{t}.
  \item \strong{\%infeasible}: proportion of mass (on the 0--1 scale) falling below
        the estimated density threshold for the targeted \code{Abar}.
  \item \strong{Feasible}: mean of the mapped feasible values (after replacing
        low-density bins) for the targeted bin.
}

The output consists of:
\itemize{
  \item A short header showing how many strategies and time points exist in the
        underlying object, and how many are being displayed after subsetting
        via \code{strategies} and \code{times}.
  \item \strong{Table 1}: \code{\%infeasible} summarized by strategy (rows) and
        time (columns), printed as percentage.
  \item \strong{Table 2}: \code{Feasible} (mean feasible value) summarized by
        strategy (rows) and time (columns), printed on the original scale.
  \item A compact display of the \code{Abar} targets by strategy and time.
}

Within the selected subset, the method also checks whether each strategy uses
the same \code{Abar} at every selected time point. If that is the case, the
printout notes that each selected strategy corresponds to the same intervention
pattern over time. Otherwise, differences in \code{Abar} across time are made
visible by the \code{Abar}-by-time display.
}
\value{
Invisibly returns \code{x}.
}
\seealso{
\code{\link{feasible}}, \code{\link{summary.feasible}}, \code{\link{plot.feasible}}
}
\examples{
data(EFV)

\donttest{
Lnodes <- c("adherence.1","weight.1",
            "adherence.2","weight.2",
            "adherence.3","weight.3",
            "adherence.4","weight.4")
Ynodes <- c("VL.0","VL.1","VL.2","VL.3","VL.4")
Anodes <- c("efv.0","efv.1","efv.2","efv.3","efv.4")

## ------------------------------------------------------------------
## Example 1: Static grid of targets (vector abar)
## Each strategy uses the same target value at every time point
## ------------------------------------------------------------------

abar_static <- seq(0, 10, by = 2)

set.seed(123)
m_static <- feasible(X = EFV,
                     Anodes = Anodes,
                     Lnodes = Lnodes,
                     Ynodes = Ynodes,
                     d.method = "parametric",
                     abar = abar_static,
                     grid.size = NULL,
                     left.boundary = 0,
                     right.boundary = 10)

## Full time x strategy summary
print(m_static)

## Use fewer digits in the numeric summaries
print(m_static, digits = 2)

## Focus on a subset of strategies (e.g., 1 and 3)
print(m_static, strategies = c(1, 3))

## Focus on early time points only (e.g., times 1, 2)
print(m_static, times = c(1, 2))

## Combine selection: only strategies 1 and 3 over times 1, 2, 3
print(m_static, strategies = c(1, 3), times = 1:3)

## ------------------------------------------------------------------
## Example 2: Non-constant intervention strategies (matrix abar)
## Strategies can have different target values at different time points
## ------------------------------------------------------------------

## Rows define strategies, columns define time points.
## The first row increases over time, the second decreases, the third increases.
abar_matrix <- rbind(
  c(0, 2, 4, 6, 8),  # strategy 1
  c(9, 6, 2, 1, 0),  # strategy 2
  c(1, 3, 5, 7, 9)   # strategy 3
)

set.seed(456)
m_matrix <- feasible(X = EFV,
                     Anodes = Anodes,
                     Lnodes = Lnodes,
                     Ynodes = Ynodes,
                     d.method = "parametric",
                     abar = abar_matrix,
                     grid.size = 1,
                     left.boundary = 0,
                     right.boundary = 10)

## Time x strategy summary where targets vary over time within strategies
print(m_matrix)

## Focus on strategies 1 and 3 over a subset of time points
print(m_matrix, strategies = c(1, 3), times = 1:3)
}
}
\keyword{print}