#' diffCompSigSim
#'
#' This function is used to determine the similarity between two signatures
#' that have different underlying components. Uses ks-statistic based
#' measure to estimate similarity for normal distribution based components and
#' uses a correlation measure when comparing poisson distribution based
#' components.
#'
#' @param refComps Reference component parameters 
#' @param refWeights Reference component weights
#' @param valComps Component parameters to compare against
#' @param valWeights Component weights to compare agaisnt
#' @return Returns a correlation value
diffCompSigSim = function(refComps,refWeights,valComps,valWeights){
      numComps = length(refComps)
  
      # The number of samples to take from each distribution
      totalSamples = 5000
      
      # Goes through each component and compares the underlying distrobutions
      compCors = numeric(numComps)
      for (j in 1:numComps){
        refParams = refComps[[j]]
        valParams = valComps[[j]]
        
        # Checks if the component is a poisson vs. normal distribution
        pois=FALSE
        if (!is.null(nrow(refParams))) {
          refNum = ncol(refParams)
          valNum = ncol(valParams)
        }
        else {
          refNum = length(refParams)
          valNum = length(valParams)
          pois = TRUE
        }
        
        # Grabs the corresponding weights for the component distributions
        refParamWeights = refWeights[paste0(names(refComps)[j],1:refNum)]
        refParamWeights = refParamWeights/sum(refParamWeights)
        valParamWeights = valWeights[paste0(names(valComps)[j],1:valNum)]
        valParamWeights = valParamWeights/sum(valParamWeights)
        
        # Creats overall distributions for the component by taking weighted 
        # samples from each sub-component using either rpois or rnorm
        totalRefYVals = c()
        totalValYVals = totalRefYVals
        for (param in 1:refNum){
          numSamp = round(totalSamples*refParamWeights[param]) 
          if (pois){
            refYVals = stats::rpois(numSamp,refParams[param])
            totalRefYVals = c(totalRefYVals,refYVals)
          }
          else{
            refYVals = stats::rnorm(numSamp,refParams[1,param],refParams[2,param])
            totalRefYVals = c(totalRefYVals,refYVals)
          }
        }
        
        # Repeats same process for the other samples
        for (param in 1:valNum){
          numSamp = round(totalSamples*valParamWeights[param])
          if (pois){
            valYVals = stats::rpois(numSamp,valParams[param])
            totalValYVals = c(totalValYVals,valYVals)
          }
          else{
            valYVals = stats::rnorm(numSamp,valParams[1,param],valParams[2,param])
            totalValYVals = c(totalValYVals,valYVals)
          }
        }
    
        
        if (pois){
          totalVals = numeric(max(c(totalRefYVals,totalValYVals))+1)
          names(totalVals) = as.character(0:max(c(totalRefYVals,totalValYVals)))
          
          tableRefYVals = totalVals
          tableRefYVals[names(table(totalRefYVals))] = table(totalRefYVals)
          
          tableValYVals = totalVals
          tableValYVals[names(table(totalValYVals))] = table(totalValYVals)
          compCors[j] = cor(tableRefYVals,tableValYVals)
        }
        else {
          compCors[j] = 1 - stats::ks.test(totalRefYVals,totalValYVals)$statistic
        }
      }
  
  return(mean(compCors))
}
