# ============================================================
#  GROWTH SIGNALS — FINAL STABILITY PATCH FOR GACE 1.0.0
# ============================================================

#' Internal helper: compute YoY lag
#' @keywords internal
#' @noRd
.gace_freq_to_yoy_lag <- function(freq_name, freq_value = NULL) {
  
  if (!is.null(freq_value) && freq_value > 1L) {
    if (freq_value == 12L) return(12L)
    if (freq_value == 4L)  return(4L)
    if (freq_value >= 50L && freq_value <= 53L) return(52L)
  }
  
  freq_name <- tolower(freq_name %||% "")
  if (freq_name %in% c("month", "monthly", "m"))    return(12L)
  if (freq_name %in% c("week", "weekly", "w"))      return(52L)
  if (freq_name %in% c("quarter", "qtr", "q"))      return(4L)
  if (freq_name %in% c("year", "yearly", "annual")) return(1L)
  
  12L
}

# ============================================================
#  MAIN HYBRID GROWTH SIGNAL BUILDER (patched)
# ============================================================
#' @keywords internal
#' @noRd
#' @importFrom utils tail
#' @importFrom stats coef
.gace_compute_growth_signals <- function(y,
                                         freq_name  = NULL,
                                         freq_value = NULL,
                                         min_points = 8L) {
  
  n <- length(y)
  if (n < min_points) return(rep(0, n))
  
  yoy_lag <- .gace_freq_to_yoy_lag(freq_name, freq_value)
  
  # ------------------------------------------------------------
  # 1. Year-over-year growth
  # ------------------------------------------------------------
  yoy <- rep(NA_real_, n)
  if (n > yoy_lag + 2L) {
    for (t in (yoy_lag + 1L):n) {
      denom <- max(abs(y[t - yoy_lag]), 1e-8)
      yoy[t] <- (y[t] / denom) - 1
    }
  }
  
  # ------------------------------------------------------------
  # 2. Short-term (MoM/WoW) growth
  # ------------------------------------------------------------
  st <- rep(NA_real_, n)
  if (n >= 2L) {
    for (t in 2:n) {
      denom <- max(abs(y[t - 1]), 1e-8)
      st[t] <- (y[t] / denom) - 1
    }
  }
  
  # ============================================================
  # 3. Rolling growth — ADAPTIVE FOR TREND BREAK DETECTION (FIX C)
  # ============================================================
  roll <- rep(NA_real_, n)
  
  # Compute recent volatility (CV over last 6–8 points)
  recent_len <- min(8L, n)
  recent_cv <- sd(tail(y, recent_len), na.rm = TRUE) /
    max(mean(tail(y, recent_len), na.rm = TRUE), 1e-8)
  
  # Baseline windows
  window_long_base  <- min(8L, floor(n / 2))
  window_short_base <- max(2L, floor(window_long_base / 2))
  
  # Adaptive shortening when volatility is high → quicker trend reaction
  if (recent_cv > 0.15) {
    window_long  <- max(4L, floor(window_long_base * 0.6))
    window_short <- max(2L, floor(window_short_base * 0.7))
  } else {
    window_long  <- window_long_base
    window_short <- window_short_base
  }
  
  if (window_long >= 4L && window_short >= 2L) {
    for (t in (window_long + 1L):n) {
      prev_idx  <- (t - window_long):(t - window_short)
      curr_idx  <- (t - window_short + 1L):t
      
      prev_mean <- mean(y[prev_idx], na.rm = TRUE)
      curr_mean <- mean(y[curr_idx], na.rm = TRUE)
      
      denom <- max(abs(prev_mean), 1e-8)
      roll[t] <- (curr_mean / denom) - 1
    }
  }
  
  # ============================================================
  # 4. Drift (trend) — DYNAMIC DRIFT CAP (Option 2)
  # ============================================================
  drift <- rep(0, n)
  t_seq <- seq_len(n)
  lm_fit <- try(stats::lm(y ~ t_seq), silent = TRUE)
  
  if (!inherits(lm_fit, "try-error")) {
    slope <- unname(coef(lm_fit)[2L])
    level <- max(mean(y, na.rm = TRUE), 1e-8)
    drift_val <- slope / level
    
    # CV-based adaptive drift limit
    cv <- sd(y, na.rm = TRUE) / max(mean(y, na.rm = TRUE), 1e-8)
    
    # Allow 2% → 10% depending on volatility
    drift_cap <- max(min(2 * cv, 0.10), 0.02)
    
    drift_val <- max(min(drift_val, drift_cap), -drift_cap)
    
    drift <- rep(drift_val, n)
  }
  
  # ------------------------------------------------------------
  # Combine components
  # ------------------------------------------------------------
  mat <- cbind(yoy, st, roll, drift)
  
  # hard limit: ±50% per component
  trim_col <- function(col, limit = 0.50) {
    idx <- which(abs(col) > limit & is.finite(col))
    if (length(idx) > 0) col[idx] <- sign(col[idx]) * limit
    col
  }
  
  for (j in seq_len(ncol(mat))) {
    mat[, j] <- trim_col(mat[, j])
  }
  
  # Hybrid average
  growth_raw <- apply(mat, 1, function(v) {
    good <- v[is.finite(v)]
    if (length(good) >= 2) mean(good) else drift[length(drift)]
  })
  
  growth_raw[!is.finite(growth_raw)] <- 0
  growth_raw
}