% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/biomass_calc.R
\name{biomass_calc}
\alias{biomass_calc}
\title{Total Biomass, Carbon and CO2 Estimation for Tree Species}
\usage{
biomass_calc(data, na_action = c("error", "omit"), output = NULL)
}
\arguments{
\item{data}{A data frame containing volume and species information for each tree.
Must include:
\itemize{
\item \code{species_code}: species name in uppercase Latin format (e.g. \code{"PICEA_ABIES"}),
matched against a density table.
\item At least one volume column:
\itemize{
\item For CNPF method (trunk volume):
\itemize{
\item Dagnelie equations: \code{dagnelie_vc22_2}, \code{dagnelie_vc22_1g}, \code{dagnelie_vc22_1}
(priority order: \code{dagnelie_vc22_2} > \code{dagnelie_vc22_1g} > \code{dagnelie_vc22_1})
\item Vallet equation: \code{vallet_vc22}
\item Rondeux equation: \code{rondeux_vc22}
\item Algan equation: \code{algan_vc22}
}
\item For Vallet method (total aboveground volume): \code{vallet_vta}
}
}

If multiple trunk volumes are provided, CNPF is computed separately for each source.
If only one is available, the corresponding method is applied.
All volume columns must be numeric and expressed in cubic meters (m3).}

\item{na_action}{How to handle missing values. \code{"error"} (default) stops if any required
value is missing. \code{"omit"} removes rows with missing values.}

\item{output}{Optional file path where the resulting data frame should be
exported as a CSV. If NULL (default), no file is written.
Export is handled by the utility function \code{export_output()} and
failures trigger warnings without interrupting execution.}
}
\value{
A data frame with one row per tree, including:
\itemize{
\item \code{species_code}: species name in uppercase Latin format.
\item \code{dagnelie_vc22_1}, \code{dagnelie_vc22_1g}, \code{dagnelie_vc22_2}, \code{vallet_vc22}, \code{rondeux_vc22},\code{algan_vc22}:
optional trunk volume inputs (in m3).
\item \code{vallet_vta}: optional total aboveground volume (in m3) for Vallet method.
\item \code{vc22_dagnelie}: selected trunk volume used for CNPF (Dagnelie), based on priority.
\item \code{vc22_source}: name of the Dagnelie column used to populate \code{vc22}.
}
\subsection{CNPF method outputs:}{
\itemize{
\item From Dagnelie (priority selection):
\itemize{
\item \code{cnpf_dagnelie_bag}, \code{cnpf_dagnelie_bbg}, \code{cnpf_dagnelie_btot},
\code{cnpf_dagnelie_c}, \code{cnpf_dagnelie_co2}
}
\item From Vallet trunk volume (\code{vallet_vc22}):
\itemize{
\item \code{cnpf_vallet_bag}, \code{cnpf_vallet_bbg}, \code{cnpf_vallet_btot},
\code{cnpf_vallet_c}, \code{cnpf_vallet_co2}
}
\item From Rondeux trunk volume (\code{rondeux_vc22}):
\itemize{
\item \code{cnpf_rondeux_bag}, \code{cnpf_rondeux_bbg}, \code{cnpf_rondeux_btot},
\code{cnpf_rondeux_c}, \code{cnpf_rondeux_co2}
}
\item From Algan trunk volume (\code{algan_vc22}):
\itemize{
\item \code{cnpf_algan_bag}, \code{cnpf_algan_bbg}, \code{cnpf_algan_btot},
\code{cnpf_algan_c}, \code{cnpf_algan_co2}
}
}
}

\subsection{Vallet method outputs (if \code{vallet_vta} is available and species is compatible):}{
\itemize{
\item \code{vallet_bag}, \code{vallet_bbg}, \code{vallet_btot}, \code{vallet_c}, \code{vallet_co2}
}
}
}
\description{
Computes total biomass (aboveground + root), carbon content and CO2 equivalent
for tree species using CNPF (with multiple trunk volume sources) and Vallet methods.
}
\details{
\itemize{
\item The density table provides:
\itemize{
\item \code{density}: wood density in tonnes of dry matter per cubic meter (t/m3).
\item \code{con_broad}: species group, either \code{"conifer"} or \code{"broadleaf"}.
}
\item The expansion factor \code{feb} is  derived from \code{con_broad}:
\itemize{
\item \code{feb = 1.3} for conifers
\item \code{feb = 1.56} for broadleaves
}
\item Dagnelie trunk volume (\code{vc22_dagnelie}) is automatically selected from the best available column,
in the following priority: \code{dagnelie_vc22_2} > \code{dagnelie_vc22_1g} > \code{dagnelie_vc22_1}.
\item CNPF outputs are computed separately for each trunk volume source (Dagnelie, Vallet, Rondeux, Algan).
\item Vallet method is applied only to a predefined list of compatible species using \code{vallet_vta}.
\item If required columns are missing, the corresponding method is skipped with a warning.
\item Warnings are also displayed if trunk volume columns exist but contain missing values (\code{NA}).
\item All biomass values are expressed in tonnes of dry matter (t),
carbon in tonnes of carbon (t C), and CO2 in tonnes of CO2 equivalent (t CO2).
}
}
\examples{
data <- data.frame(
  species_code = c("PICEA_ABIES", "QUERCUS_ROBUR", "FAGUS_SYLVATICA"),
  dagnelie_vc22_2 = c(1.1, NA, NA),
  dagnelie_vc22_1g = c(NA, NA, NA),
  dagnelie_vc22_1 = c(NA, 0.9, NA),
  vallet_vc22 = c(NA, 1.2, NA),
  rondeux_vc22 = c(NA, NA, 1.0),
  algan_vc22 = c(NA,0.8,NA),
  vallet_vta = c(1.5, NA, 1.3)
)

output_path <- tempfile(fileext = ".csv")
results <- biomass_calc(data, output = output_path)
if (file.exists(output_path)) {
  message("CSV file successfully created.")
} else {
  warning("CSV file was not created.")
}

}
