#' Map observations to BMUs (Best Matching Units) using Gower distance
#'
#' Computes, for each observation, the index of the best-matching neuron (BMU)
#' in a trained Gower-SOM codebook and the corresponding Gower distance. Also
#' converts BMU indices to grid coordinates (row, col).
#'
#' @param data A data.frame of observations to map. Must be typed consistently
#'   with the training data (numeric, factor, etc.).
#' @param codebook A data.frame (or matrix coerced to data.frame) with one row per
#'   neuron and the same columns as `data` (i.e., the trained SOM weights).
#' @param n_rows,n_cols Integers, SOM grid dimensions.
#'
#' @return A data.frame with columns:
#' \describe{
#'   \item{bmu}{Integer BMU index (1..n_rows*n_cols).}
#'   \item{distance}{Numeric Gower distance to the BMU.}
#'   \item{row}{BMU grid row (1..n_rows).}
#'   \item{col}{BMU grid column (1..n_cols).}
#' }
#'
#' @examples
#' \dontrun{
#' set.seed(1)
#' df <- data.frame(x1=rnorm(10), x2=rnorm(10), g=factor(sample(letters[1:3],10,TRUE)))
#' fit <- gsom_Training(df, grid_rows=3, grid_cols=3, num_iterations=5, batch_size=5)
#' res <- get_bmu_gower(df, codebook = fit$weights, n_rows = 3, n_cols = 3)
#' head(res)
#' }
#'
#' @export
#' @importFrom gower gower_dist

get_bmu_gower <- function(data, codebook, n_rows, n_cols) {
  data <- as.data.frame(data)
  codebook <- as.data.frame(codebook)
  stopifnot(nrow(codebook) == n_rows * n_cols)

  # ensure factor levels match for categorical columns
  common_cols <- intersect(colnames(data), colnames(codebook))
  data <- data[ , common_cols, drop = FALSE]
  codebook <- codebook[ , common_cols, drop = FALSE]

  n <- nrow(data)
  bmus <- integer(n)
  dmin <- numeric(n)

  for (i in seq_len(n)) {
    d <- as.numeric(gower_dist(data[i, , drop = FALSE], codebook))
    bmus[i] <- which.min(d)
    dmin[i] <- min(d)
  }

  # index -> (row, col)
  idx_to_coords <- function(index, n_rows, n_cols) {
    row <- ((index - 1) %% n_rows) + 1
    col <- ((index - 1) %/% n_rows) + 1
    c(row = row, col = col)
  }
  coords <- t(vapply(bmus, idx_to_coords, numeric(2), n_rows = n_rows, n_cols = n_cols))

  out <- data.frame(bmu = bmus, distance = dmin, row = coords[,1], col = coords[,2])
  rownames(out) <- rownames(data)

  return(out)
}
