#' Replicate mass uptake curve
#'
#' @description Plot the mass uptake curve for selected
#' peptide to see the difference between replicates.
#' 
#' @param dat data imported by the \code{\link{read_hdx}} function
#' @param protein selected protein
#' @param state selected biological state for given protein
#' @param sequence selected peptide sequence for given protein 
#' in given biological state
#' @param aggregated \code{logical}, indicator if presented
#' data is aggregated on replicate level
#' @param log_x \code{logical}, indicator if the X axis values 
#' are transformed to log10
#' @param interactive \code{logical}, whether plot should have an interactive 
#' layer created with with ggiraph, which would add tooltips to the plot in an
#' interactive display (HTML/Markdown documents or shiny app)
#' 
#' @details The function \code{\link{plot_replicate_mass_uptake}}
#' generates a plot showing the mass uptake for selected protein 
#' in replicates of the experiments. The values can be presented 
#' in two ways: as aggregated values for each replicate, or before
#' aggregation - measured values for charge values within a replicate.
#' The mass uptake is generated by subtracting the MHP mass of a peptide
#' from measured mass and the mass uptake is presented in Daltons. 
#'
#' @return a \code{\link[ggplot2]{ggplot}} object.
#' 
#' @seealso 
#' \code{\link{read_hdx}}
#' \code{\link{calculate_exp_masses_per_replicate}}
#' 
#' @examples 
#' plot_replicate_mass_uptake(alpha_dat)
#' plot_replicate_mass_uptake(alpha_dat, aggregated = TRUE)
#' 
#' @export plot_replicate_mass_uptake

plot_replicate_mass_uptake <- function(dat, 
                                       protein = dat[["Protein"]][1],
                                       state = dat[["State"]][1],
                                       sequence = dat[["Sequence"]][1],
                                       aggregated = FALSE,
                                       log_x = TRUE,
                                       interactive = getOption("hadex_use_interactive_plots")){
  
  
  if(aggregated) {
    
    mass_uptake_dat <- dat %>%
      filter(Protein == protein, 
             State == state,
             Sequence == sequence,
             Exposure > 0.001) %>%
      calculate_exp_masses_per_replicate() %>%
      mutate(mass_uptake = avg_exp_mass - MHP) 
      
    if(interactive){
      
      mass_uptake_plot <- ggplot(mass_uptake_dat) +
        geom_point_interactive(aes(x = Exposure, y = mass_uptake,
                                   tooltip = glue(
                                     "{Sequence}
                                       Position: {Start}-{End}
                                       Value: {round(mass_uptake, 2)}"
                                   ))) 
      
    } else {
      
      mass_uptake_plot <- ggplot(mass_uptake_dat) +
        geom_point(aes(x = Exposure, y = mass_uptake)) 
    }
      
      
  } else {
    
    mass_uptake_dat <- dat %>%
      filter(Protein == protein, 
             State == state,
             Sequence == sequence,
             Exposure > 0.001) %>%
      mutate(exp_mass = Center*z - z*1.00727647,
             weighted_Inten = scale(Inten),
             mass_uptake = exp_mass - MHP) 
    
    if(interactive){
      
      mass_uptake_plot <- ggplot(mass_uptake_dat) +
        geom_point_interactive(aes(x = Exposure, y = mass_uptake, colour = as.factor(z),
                                   tooltip = glue(
                                     "{Sequence}
                                       Position: {Start}-{End}
                                       Value: {round(mass_uptake, 2)}
                                       Charge: {z}"
                                   ))) +
        labs(colour = "Charge") +
        theme(legend.position = "bottom")
      
    } else {
      
      mass_uptake_plot <- ggplot(mass_uptake_dat) +
        geom_point(aes(x = Exposure, y = mass_uptake, colour = as.factor(z))) +
        labs(colour = "Charge") +
        theme(legend.position = "bottom")
    }
    
    
    
      
  }
  
  mass_uptake_plot <- mass_uptake_plot +
    labs(x = "Time points [min]",
         y = "Mass uptake [Da]") +
    coord_cartesian(ylim = c(0, NA))
    
  if(log_x){ mass_uptake_plot <- mass_uptake_plot + scale_x_log10() }
  
  return(HaDeXify(mass_uptake_plot))
  
}