# =============================================================================
# INSTABILITY DETECTION (NEW in KWELA 1.0.0)
# =============================================================================
# Deterministic treatment-level instability assessment for matrix interference.
# All metrics are deterministic (no RNG). Thresholds are configurable.

#' Compute Instability Flags for a Treatment
#'
#' Evaluates 6 deterministic metrics to detect matrix interference that
#' compromises classification reliability. A treatment is flagged as unstable
#' when it behaves unlike both positive and negative controls.
#'
#' @param trt_mp Finite MP values for treatment wells
#' @param trt_ttt TTT values for treatment wells (may contain NA/Inf)
#' @param pc_mp,nc_mp Positive/negative control MP values (finite)
#' @param pc_ttt,nc_ttt Positive/negative control TTT values
#' @param trt_raf RAF values for treatment wells (NULL if unavailable)
#' @param trt_mp_raf MP values corresponding to RAF wells (NULL if unavailable)
#' @param pc_raf_mp_ratios RAF/MP ratios from positive controls (NULL if unavailable)
#' @param crossing_threshold TTT crossing threshold for this treatment's group
#' @param strictness "moderate" (2+ flags), "strict" (1+), or "lenient" (3+)
#' @param has_raf Whether RAF data is available
#' @return List with components:
#'   \describe{
#'     \item{unstable}{Logical: TRUE if treatment is flagged}
#'     \item{reasons}{Character vector of triggered criteria}
#'     \item{n_flags}{Number of triggered criteria}
#'     \item{min_flags_required}{Minimum flags needed for instability}
#'     \item{metrics}{Named list of computed metric values}
#'   }
#' @export
#' @examples
#' set.seed(42)
#' flags <- compute_instability_flags(
#'   trt_mp = rnorm(8, 50, 20),
#'   trt_ttt = rnorm(8, 40, 15),
#'   pc_mp = rnorm(8, 100, 10),
#'   nc_mp = rnorm(8, 20, 5),
#'   pc_ttt = rnorm(8, 8, 1),
#'   nc_ttt = rnorm(8, 72, 5),
#'   crossing_threshold = 40,
#'   strictness = "moderate"
#' )
compute_instability_flags <- function(
    trt_mp, trt_ttt, pc_mp, nc_mp, pc_ttt, nc_ttt,
    trt_raf = NULL, trt_mp_raf = NULL, pc_raf_mp_ratios = NULL,
    crossing_threshold = NA_real_,
    strictness = "moderate",
    has_raf = FALSE
) {
  reasons <- character(0)
  metrics <- list()

  # Threshold sets by strictness
  thresholds <- switch(strictness,
    "strict" = list(fano = 1.5, cross_lo = 0.15, cross_hi = 0.85,
                    wass = 2.0, ttt_disp = 1.5, min_flags = 1),
    "lenient" = list(fano = 3.0, cross_lo = 0.35, cross_hi = 0.65,
                     wass = 3.5, ttt_disp = 3.0, min_flags = 3),
    # moderate (default)
    list(fano = 2.0, cross_lo = 0.25, cross_hi = 0.75,
         wass = 2.5, ttt_disp = 2.0, min_flags = 2)
  )

  n_trt <- length(trt_mp)
  nc_spread <- NA_real_  # Initialize for use across metrics

  # 1. Fano factor deviation
  if (n_trt >= 2 && length(pc_mp) >= 2) {
    mu_trt <- mean(trt_mp, na.rm = TRUE)
    mu_pc <- mean(pc_mp, na.rm = TRUE)
    if (is.finite(mu_trt) && abs(mu_trt) > .Machine$double.eps &&
        is.finite(mu_pc) && abs(mu_pc) > .Machine$double.eps) {
      fano_trt <- stats::var(trt_mp, na.rm = TRUE) / mu_trt
      fano_pc <- stats::var(pc_mp, na.rm = TRUE) / mu_pc
      fano_diff <- abs(fano_trt - fano_pc)
      metrics$fano_trt <- fano_trt
      metrics$fano_pc <- fano_pc
      metrics$fano_diff <- fano_diff
      if (is.finite(fano_diff) && fano_diff > thresholds$fano) {
        reasons <- c(reasons, sprintf("fano_deviation=%.2f (>%.1f)", fano_diff, thresholds$fano))
      }
    }
  }

  # 2. Crossing variability
  trt_ttt_all <- trt_ttt
  if (length(trt_ttt_all) >= 3 && is.finite(crossing_threshold)) {
    crossed <- is.finite(trt_ttt_all) & trt_ttt_all < crossing_threshold
    cross_rate <- mean(crossed)
    metrics$cross_rate <- cross_rate
    if (cross_rate > thresholds$cross_lo && cross_rate < thresholds$cross_hi) {
      reasons <- c(reasons, sprintf("crossing_instability=%.2f (%.2f-%.2f)",
                                     cross_rate, thresholds$cross_lo, thresholds$cross_hi))
    }
  }

  # 3. Wasserstein distance from BOTH controls
  if (n_trt >= 2 && length(pc_mp) >= 2 && length(nc_mp) >= 2) {
    wass_pc <- wasserstein_1d(trt_mp, pc_mp)
    wass_nc <- wasserstein_1d(trt_mp, nc_mp)
    nc_spread <- robust_scale(nc_mp)
    if (is.finite(nc_spread) && nc_spread > .Machine$double.eps) {
      wass_pc_norm <- wass_pc / nc_spread
      wass_nc_norm <- wass_nc / nc_spread
    } else {
      wass_pc_norm <- wass_pc
      wass_nc_norm <- wass_nc
    }
    metrics$wass_pc_norm <- wass_pc_norm
    metrics$wass_nc_norm <- wass_nc_norm
    if (is.finite(wass_pc_norm) && is.finite(wass_nc_norm) &&
        wass_pc_norm > thresholds$wass && wass_nc_norm > thresholds$wass) {
      reasons <- c(reasons, sprintf("wass_both_distant (pc=%.2f, nc=%.2f, >%.1f)",
                                     wass_pc_norm, wass_nc_norm, thresholds$wass))
    }
  }

  # 4. Energy distance from BOTH controls
  if (n_trt >= 2 && length(pc_mp) >= 2 && length(nc_mp) >= 2) {
    en_pc <- energy_distance(trt_mp, pc_mp)
    en_nc <- energy_distance(trt_mp, nc_mp)
    if (is.finite(nc_spread) && nc_spread > .Machine$double.eps) {
      en_pc_norm <- en_pc / nc_spread^2
      en_nc_norm <- en_nc / nc_spread^2
    } else {
      en_pc_norm <- en_pc
      en_nc_norm <- en_nc
    }
    metrics$energy_pc_norm <- en_pc_norm
    metrics$energy_nc_norm <- en_nc_norm
    if (is.finite(en_pc_norm) && is.finite(en_nc_norm) &&
        en_pc_norm > thresholds$wass && en_nc_norm > thresholds$wass) {
      reasons <- c(reasons, sprintf("energy_both_distant (pc=%.2f, nc=%.2f)",
                                     en_pc_norm, en_nc_norm))
    }
  }

  # 5. TTT dispersion
  trt_ttt_fin <- trt_ttt[is.finite(trt_ttt)]
  pc_ttt_fin <- pc_ttt[is.finite(pc_ttt)]
  if (length(trt_ttt_fin) >= 2 && length(pc_ttt_fin) >= 2) {
    ttt_mad_trt <- robust_scale(trt_ttt_fin)
    ttt_mad_pc <- robust_scale(pc_ttt_fin)
    if (is.finite(ttt_mad_trt) && is.finite(ttt_mad_pc) &&
        ttt_mad_pc > .Machine$double.eps) {
      ttt_disp_ratio <- ttt_mad_trt / ttt_mad_pc
      metrics$ttt_disp_ratio <- ttt_disp_ratio
      if (ttt_disp_ratio > thresholds$ttt_disp) {
        reasons <- c(reasons, sprintf("ttt_dispersion=%.2f (>%.1f)", ttt_disp_ratio, thresholds$ttt_disp))
      }
    }
  }

  # 6. RAF-MP inconsistency
  if (has_raf && !is.null(trt_raf) && !is.null(trt_mp_raf) && !is.null(pc_raf_mp_ratios)) {
    valid <- is.finite(trt_raf) & is.finite(trt_mp_raf) & trt_mp_raf > .Machine$double.eps
    if (sum(valid) >= 2 && length(pc_raf_mp_ratios) >= 2) {
      trt_ratios <- trt_raf[valid] / trt_mp_raf[valid]
      trt_ratio_mad <- robust_scale(trt_ratios)
      pc_ratio_mad <- robust_scale(pc_raf_mp_ratios)
      if (is.finite(trt_ratio_mad) && is.finite(pc_ratio_mad) &&
          pc_ratio_mad > .Machine$double.eps) {
        raf_incon <- trt_ratio_mad / pc_ratio_mad
        metrics$raf_inconsistency <- raf_incon
        if (raf_incon > thresholds$ttt_disp) {
          reasons <- c(reasons, sprintf("raf_mp_inconsistency=%.2f", raf_incon))
        }
      }
    }
  }

  list(
    unstable = length(reasons) >= thresholds$min_flags,
    reasons = reasons,
    n_flags = length(reasons),
    min_flags_required = thresholds$min_flags,
    metrics = metrics
  )
}
