# =============================================================================
# MODULE 4: PROPER SCORING RULES
# =============================================================================

#' Continuous Ranked Probability Score (CRPS)
#'
#' The primary metric for stochastic process evaluation.
#'
#' @param forecast_samples Numeric vector of samples from forecast distribution
#' @param observation Single observed value
#' @return CRPS value (non-negative), or NA_real_ if insufficient data
#' @export
#' @references
#' Gneiting T, Raftery AE (2007). Strictly proper scoring rules, prediction,
#' and estimation. Journal of the American Statistical Association 102(477):359-378.
#' @examples
#' crps_empirical(rnorm(100, 5, 1), 5.5)
crps_empirical <- function(forecast_samples, observation) {
  forecast_samples <- forecast_samples[is.finite(forecast_samples)]
  if (length(forecast_samples) < 2 || !is.finite(observation)) return(NA_real_)
  n <- length(forecast_samples)
  term1 <- mean(abs(forecast_samples - observation))
  term2_pairs <- mean(stats::dist(forecast_samples))
  term2_allij <- term2_pairs * (n - 1) / n
  max(term1 - 0.5 * term2_allij, 0)
}

#' Dawid-Sebastiani Score
#'
#' Combines variance and mean error for forecast distribution evaluation.
#'
#' @param observation Observed value
#' @param predicted_mean Predicted mean
#' @param predicted_var Predicted variance
#' @return DS score, or NA_real_ if inputs invalid
#' @export
#' @references
#' Dawid AP, Sebastiani P (1999). Coherent dispersion criteria for optimal
#' experimental design. Annals of Statistics 27(1):65-81.
#' @examples
#' dawid_sebastiani(5.5, 5, 1)
dawid_sebastiani <- function(observation, predicted_mean, predicted_var) {
  if (!is.finite(observation) || !is.finite(predicted_mean) ||
      !is.finite(predicted_var) || predicted_var <= 0) return(NA_real_)
  (observation - predicted_mean)^2 / predicted_var + log(predicted_var)
}

#' Log Predictive Score
#'
#' Assumes Gaussian predictive distribution.
#'
#' @param observation Observed value
#' @param predicted_mean Predicted mean
#' @param predicted_sd Predicted standard deviation
#' @return Negative log density, or NA_real_ if inputs invalid
#' @export
#' @examples
#' log_predictive_score(5.5, 5, 1)
log_predictive_score <- function(observation, predicted_mean, predicted_sd) {
  if (!is.finite(observation) || !is.finite(predicted_mean) ||
      !is.finite(predicted_sd) || predicted_sd <= 0) return(NA_real_)
  -stats::dnorm(observation, predicted_mean, predicted_sd, log = TRUE)
}

#' Interval Score
#'
#' Evaluates prediction intervals with penalties for width and miscoverage.
#'
#' @param observation Observed value
#' @param lower Lower bound of prediction interval
#' @param upper Upper bound of prediction interval
#' @param alpha Nominal miscoverage rate (default 0.1 for 90% interval)
#' @return Interval score, or NA_real_ if inputs invalid
#' @export
#' @examples
#' interval_score(5.5, 3, 7, alpha = 0.1)
interval_score <- function(observation, lower, upper, alpha = 0.1) {
  if (!is.finite(observation) || !is.finite(lower) || !is.finite(upper))
    return(NA_real_)
  width <- upper - lower
  penalty_lower <- (2 / alpha) * (lower - observation) * (observation < lower)
  penalty_upper <- (2 / alpha) * (observation - upper) * (observation > upper)
  width + penalty_lower + penalty_upper
}
