% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse_IRSAR.RF.R
\name{analyse_IRSAR.RF}
\alias{analyse_IRSAR.RF}
\title{Analyse IRSAR RF measurements}
\usage{
analyse_IRSAR.RF(
  object,
  sequence_structure = c("NATURAL", "REGENERATED"),
  RF_nat.lim = NULL,
  RF_reg.lim = NULL,
  method = "FIT",
  method_control = NULL,
  test_parameters = NULL,
  n.MC = 10,
  txtProgressBar = TRUE,
  plot = TRUE,
  plot_reduced = FALSE,
  ...
)
}
\arguments{
\item{object}{\linkS4class{RLum.Analysis} or a \link{list} of \linkS4class{RLum.Analysis}-objects (\strong{required}):
input object containing data for protocol analysis. The function expects to
find at least two curves in the \linkS4class{RLum.Analysis} object: (1) \code{RF_nat}, (2) \code{RF_reg}.
If a \code{list} is provided as input all other parameters can be provided as
\code{list} as well to gain full control.}

\item{sequence_structure}{\link{vector} \link{character} (\emph{with default}):
specifies the general sequence structure. Allowed steps are \code{NATURAL} and
\code{REGENERATED}, and at least one of each must appear. In addition, any other
character is allowed in the sequence structure; such curves will be ignored
during the analysis. If \code{sequence_structure} is shorter than the number of
curves in \code{object}, it is recycled. If multiple \code{NATURAL} and \code{REGENERATED}
steps are specified, the corresponding measurements are stacked.}

\item{RF_nat.lim}{\link{vector} (\emph{with default}):
set minimum and maximum channel range for natural signal fitting and sliding.
If only one value is provided this will be treated as minimum value and the
maximum limit will be added automatically.}

\item{RF_reg.lim}{\link{vector} (\emph{with default}):
set minimum and maximum channel range for regenerated signal fitting and sliding.
If only one value is provided this will be treated as minimum value and the
maximum limit will be added automatically.}

\item{method}{\link{character} (\emph{with default}): select method applied for the
data analysis. Possible options are \code{"FIT"}, \code{"SLIDE"}, \code{"VSLIDE"};
\code{"NONE"} can be used to disable the analysis and plot the natural points
at their original position.}

\item{method_control}{\link{list} (\emph{optional}):
parameters to control the method, that can be passed to the chosen method.
These are for (1) \code{method = "FIT"}: \code{'trace'}, \code{'maxiter'}, \code{'warnOnly'}, \code{'minFactor'} and for
(2) \code{method = "SLIDE"}: \code{'correct_onset'}, \code{'show_density'},  \code{'show_fit'}, \code{'trace'}.
See details.}

\item{test_parameters}{\link{list} (\emph{with default}):
set test parameters. Supported parameters are: \code{curves_ratio},
\code{residuals_slope} (only for \code{method = "SLIDE"} and \code{"VSLIDE"}),
\code{curves_bounds}, \code{dynamic_ratio}, \code{lambda}, \code{beta} and \code{delta.phi}.
All input: \link{numeric} values, \code{NA} and \code{NULL} (see Details for further
information).}

\item{n.MC}{\link{numeric} (\emph{with default}):
number of Monte Carlo runs for the estimation of the start parameter
(\code{method = "FIT"}) or of the error (\code{method = "SLIDE"} and \code{"VSLIDE"}).
This value can be set to \code{NULL} to skip the MC runs.
Note: Large values will significantly increase the computation time.}

\item{txtProgressBar}{\link{logical} (\emph{with default}):
enable/disable the progress bar during MC runs.}

\item{plot}{\link{logical} (\emph{with default}):
enable/disable the plot output.}

\item{plot_reduced}{\link{logical} (\emph{optional}):
provides a reduced plot output if enabled to allow common R plot combinations,
e.g., \code{par(mfrow(...))}. If \code{TRUE} no residual plot
is returned; it has no effect if \code{plot = FALSE}}

\item{...}{further arguments that will be passed to the plot output.
Currently supported arguments are \code{main}, \code{mtext}, \code{xlab}, \code{ylab},
\code{xlim}, \code{ylim}, \code{log}, \code{legend} (\code{TRUE/FALSE}),
\code{legend.pos}, \code{legend.text} (passes argument to x,y in
\link[graphics:legend]{graphics::legend}), \code{xaxt}, \code{verbose} (\code{TRUE/FALSE}).}
}
\value{
The function returns numerical output and an (\emph{optional}) plot.

-----------------------------------\cr
\verb{[ NUMERICAL OUTPUT ]}\cr
-----------------------------------\cr

\strong{\code{RLum.Results}}-object

\strong{slot:} \strong{\verb{@data}}

\verb{[.. $data : data.frame]}

\tabular{lll}{
\strong{Column} \tab \strong{Type} \tab \strong{Description}\cr
\code{DE} \tab \code{numeric} \tab the obtained equivalent dose\cr
\code{DE.ERROR} \tab \code{numeric} \tab (only \code{method = "SLIDE"}) standard deviation obtained from MC runs \cr
\code{DE.LOWER} \tab \code{numeric}\tab 2.5\% quantile for De values obtained by MC runs \cr
\code{DE.UPPER} \tab \code{numeric}\tab 97.5\% quantile for De values obtained by MC runs  \cr
\code{DE.STATUS}  \tab \code{character}\tab test parameter status\cr
\code{RF_NAT.LIM}  \tab \code{character}\tab used \code{RF_nat} curve limits \cr
\code{RF_REG.LIM} \tab \code{character}\tab used \code{RF_reg} curve limits\cr
\code{POSITION} \tab \code{integer}\tab (\emph{optional}) position of the curves\cr
\code{DATE} \tab \code{character}\tab (\emph{optional}) measurement date\cr
\code{SEQUENCE_NAME} \tab \code{character}\tab (\emph{optional}) sequence name\cr
\code{UID} \tab \code{character}\tab unique data set ID
}

\verb{[.. $De.MC : numeric]}

A \code{numeric} vector with all the De values obtained by the MC runs.

\verb{[.. $test_parameters : data.frame]}

\tabular{lll}{
\strong{Column} \tab \strong{Type} \tab \strong{Description}\cr
\code{POSITION} \tab \code{numeric} \tab aliquot position \cr
\code{PARAMETER} \tab \code{character} \tab test parameter name \cr
\code{THRESHOLD} \tab \code{numeric} \tab set test parameter threshold value \cr
\code{VALUE} \tab \code{numeric} \tab the calculated test parameter value (to be compared with the threshold)\cr
\code{STATUS} \tab \code{character} \tab test parameter status either \code{"OK"} or \code{"FAILED"} \cr
\code{SEQUENCE_NAME} \tab \code{character} \tab name of the sequence, so far available \cr
\code{UID} \tab \code{character}\tab unique data set ID
}

\verb{[.. $fit : data.frame]}

An \link{nls} object produced by the fitting.

\verb{[.. $slide : list]}

A \link{list} with data produced during the sliding. Some elements are previously
reported with the summary object data. List elements are:

\tabular{lll}{
\strong{Element} \tab \strong{Type} \tab \strong{Description}\cr
\code{De} \tab \code{numeric} \tab the final De obtained with the sliding approach \cr
\code{De.MC} \tab \code{numeric} \tab all De values obtained by the MC runs \cr
\code{residuals} \tab \code{numeric} \tab the obtained residuals for each channel of the curve \cr
\code{trend.fit} \tab \code{lm} \tab fitting results produced by the fitting of the residuals \cr
\code{RF_nat.slid} \tab \code{matrix} \tab the slid \code{RF_nat} curve \cr
\code{t_n.id} \tab \code{numeric} \tab the index of the t_n offset \cr
\code{I_n} \tab \code{numeric} \tab the vertical intensity offset if a vertical slide was applied \cr
\code{algorithm_error} \tab \code{numeric} \tab the vertical sliding suffers from a systematic effect induced by the used
algorithm. The returned value is the standard deviation of all obtained De values while expanding the
vertical sliding range. It can be added as systematic error to the final De error; so far wanted.\cr
\code{vslide_range} \tab \code{numeric} \tab the range used for the vertical sliding \cr
\code{num_slide_windows} \tab \code{integer} \tab the number of windows used for the vertical sliding \cr
\code{squared_residuals} \tab \code{numeric} \tab the squared residuals (horizontal sliding)
}

\strong{slot:} \strong{\verb{@info}}

The original function call (\link[methods:LanguageClasses]{methods::language}-object)

The output (\code{data}) should be accessed using the function \link{get_RLum}.

------------------------\cr
\verb{[ PLOT OUTPUT ]}\cr
------------------------\cr

The slid IR-RF curves with the finally obtained De
}
\description{
The function analyses IRSAR RF measurements on K-feldspar samples performed
using the protocol according to Erfurt et al. (2003) and beyond.
}
\details{
The function performs an IRSAR analysis described for K-feldspar samples by
Erfurt et al. (2003) assuming a negligible sensitivity change of the RF
signal.

\strong{General Sequence Structure} (according to Erfurt et al., 2003)
\enumerate{
\item Measuring IR-RF intensity of the natural dose for a few seconds (\eqn{RF_{nat}})
\item Bleach the samples under solar conditions for at least 30 min without changing the geometry
\item Waiting for at least one hour
\item Regeneration of the IR-RF signal to at least the natural level (measuring (\eqn{RF_{reg}})
\item Fitting data with a stretched exponential function
\item Calculate the palaeodose \eqn{D_{e}} using the parameters from the fitting
}

Three methods are supported to obtain the \eqn{D_{e}}:
\code{method = "FIT"}, \code{method = "SLIDE"} and \code{method = "VSLIDE"}:

\strong{\code{method = "FIT"}}

The principle is described above and follows the original suggestions by
Erfurt et al., 2003. For the fitting, the mean count value of the \code{RF_nat}
curve is used.

Function used for the fitting (according to Erfurt et al. (2003)):

\deqn{\phi(D) = \phi_{0}-\Delta\phi(1-exp(-\lambda*D))^\beta}

with
\eqn{\phi(D)} the dose dependent IR-RF flux,
\eqn{\phi_{0}} the initial IR-RF flux,
\eqn{\Delta\phi} the dose dependent change of the IR-RF flux,
\eqn{\lambda} the exponential parameter, \eqn{D} the dose and
\eqn{\beta} the dispersive factor.

To obtain the palaeodose
\eqn{D_{e}} the function is changed to:

\deqn{D_{e} = ln(-(\phi(D) - \phi_{0})/(-\lambda*\phi)^{1/\beta}+1)/-\lambda}

The fitting is done using the \code{port} algorithm of the \link{nls} function.

\strong{\code{method = "SLIDE"}}

For this method, the natural curve is slid along the x-axis until
congruence with the regenerated curve is reached. Unlike fitting, this
allows for working with the original data without the need for any physical
model. This approach was introduced for RF curves by Buylaert et al., 2012
and Lapp et al., 2012.

Here the sliding is done by searching for the minimum of the sum of
squared residuals.
For the mathematical details of the implementation see Frouin et al., 2017

\strong{\code{method = "VSLIDE"}}

Same as \code{"SLIDE"} but searching also vertically for the best match (i.e. in xy-direction.)
See Kreutzer et al. (2017) and Murari et al. (2021). By default, the vertical sliding
range is set automatically, but can be set manually by changing the
\code{vslide_range} parameter (see \code{method_control}).

\strong{\code{method_control}}

To keep the generic argument list as clear as possible, parameters to control
the methods for De estimation are preset with meaningful default values,
which can however be modified using the \code{method_control} argument, e.g.,
\code{method_control = list(trace = TRUE)}. Supported parameters are:

\emph{For FIT}
\itemize{
\item \code{trace} (\link{logical}, default: \code{FALSE}): as in \link{nls}; shows sum of squared
residuals.
\item \code{maxiter} (\link{integer}, default: 500): as in \link{nls}.
\item \code{warnOnly} (\link{logical}, default: \code{FALSE}): as in \link{nls}.
\item \code{minFactor} (\link{numeric}, default: \code{1 / 4096}): as in \link{nls}.
}

\emph{For SLIDE or VSLIDE}
\itemize{
\item \code{trace} (\link{logical}, default: \code{FALSE}): as in \link{nls}; shows sum of squared
residuals.
\item \code{trace_vslide} (\link{logical}, default: \code{FALSE}): enable/disable the tracing
of the vertical sliding.
\item \code{correct_onset} (\link{logical}, default: \code{TRUE}): whether the curves should
be shifted along the x-axis by the first channel, as light is expected in
the first channel.
\item \code{show_density} (\link{logical}, default: \code{TRUE}): enable/disable KDE plots for
MC run results. Nothing is shown if the distribution is too narrow.
\item \code{show_fit} (\link{logical}, default: \code{FALSE}): enable/disable the plot of the
fitted curve routinely obtained during the evaluation.
\item \code{n.MC} (\link{integer}, default: 1000): number of Monte Carlo runs within the
sliding (assessing the possible minimum values). \strong{Note}: This parameter
is not the same as the function argument \code{n.MC}.
\item \code{vslide_range} (\link{numeric} or \link{character}, default: "auto"): boundaries
for the vertical curve sliding. The argument expects a vector with absolute
minimum and maximum (e.g., \code{c(-1000,1000)}). The default "auto" mode
detects the reasonable vertical sliding range (\emph{recommended}). \code{NULL}
disables the vertical sliding.
\item \code{num_slide_windows} (\link{integer}, default: 3): number of  differently-sized
windows tested when sliding: the higher the value (up to a maximum of 10),
the more time is spent in searching the global optimum. The default setting
attempts to strike a balance between quality of the fit and computation
speed.
\item \code{cores} (\link{numeric} or \link{character}, default: \code{NULL}): number of cores
allocated for a parallel processing of the Monte-Carlo runs. The default
value corresponds to single-threaded computation; the recommended values is
\code{"auto"}, which assigns all but two of the available cores.
}

\strong{Error estimation}

For \strong{\code{method = "FIT"}}, the \eqn{D_{e}} error range is obtained by using
the 2.5 \% (lower) and the 97.5 \% (upper) quantiles of the \eqn{RF_{nat}}
curve.

For \strong{\code{method = "SLIDE"}} and \strong{\code{method = "VSLIDE"}}, the error is obtained
by bootstrapping the residuals of the slid curve to construct new natural
curves for a Monte Carlo simulation. The error is returned in two ways:
(a) the standard deviation of the \eqn{D_{e}} obtained from the MC runs and
(b) the confidence interval using the 2.5 \% (lower) and the 97.5 \% (upper)
quantiles. The results of the MC runs are returned with the function output.

\strong{Test parameters}

The argument \code{test_parameters} allows to pass thresholds for several test
parameters, which will be evaluated during the function run. If a threshold
is set and it is exceeded, the test parameter status will be set to
\code{"FAILED"}. This argument is intentionally not termed \code{'rejection criteria'}
as not all test parameters are evaluated for both methods and some parameters
are calculated but not evaluated by default.

\code{NA} and \code{NULL} are the allowed values for all parameters. If the parameter
is set to \code{NA}, the value is calculated but the result will not be evaluated,
therefore it will have no effect on the status (\code{"OK"} or \code{"FAILED"}) of
the parameter. Setting the parameter to \code{NULL} disables the parameter
entirely and the parameter will also be removed from the function output.
This might be useful in cases where a particular parameter requires a long
computation time. Currently supported parameters are:
\itemize{
\item \code{curves_ratio} (\link{numeric}, default: \code{1.001}):
the ratio of \eqn{RF_{nat}} to \eqn{RF_{reg}} is calculated over the range
spanned by \eqn{RF_{nat}}, and should not exceed the threshold value.
\item \code{intersection_ratio} (\link{numeric}, default: \code{NA}):
calculated as absolute difference from 1 of the ratio of the integral of
the normalised RF-curves. This value indicates intersection of the RF-curves
and should be close to 0 if the curves have a similar shape. For this
calculation first the corresponding time-count pair value on the RF_reg
curve is obtained using the maximum count value of the \code{RF_nat} curve and
only this segment (fitting to the \code{RF_nat} curve) on the RF_reg curve is
taken for further calculating this ratio. If nothing is found at all, \code{Inf}
is returned.
\item \code{residuals_slope} (\link{numeric}, default: \code{NA}; only for \code{method = "SLIDE"}
and \verb{"VSLIDE}"):
a linear function is fitted on the residuals after sliding.
The corresponding slope can be used to discard values as a high (positive,
negative) slope may indicate that both curves are fundamentally different
and the method cannot be applied at all.
By default, the value of this parameter is calculated but not evaluated.
\item \code{curves_bounds} (\link{numeric}, default: \eqn{max(RF_{reg_counts})}):
this measure uses the maximum time (x) value of the regenerated curve.
The maximum time (x) value of the natural curve cannot be larger than this
value. However, although this is not recommended the value can be changed
or disabled.
\item \code{dynamic_ratio} (\link{numeric}, default: \code{NA}):
the dynamic ratio of the regenerated curve is calculated as ratio of the
minimum and maximum count values.
\item \code{lambda}, \code{beta} and \code{delta.phi} \link{numeric} (default: \code{NA}):
the stretched exponential function suggested by Erfurt et al. (2003)
describing the decay of the RF signal, comprises several parameters that
might be useful to evaluate the shape of the curves. For \code{method = "FIT"},
this parameter is obtained during the fitting; for \code{method = "SLIDE"} a
rather rough estimation is made using the function \link[minpack.lm:nlsLM]{minpack.lm::nlsLM} and
the equation given above.
Note: As this procedure requests more computation time, it is performed only
if all three parameters are set.
}
}
\note{
This function assumes that there is no sensitivity change during the
measurements (natural vs. regenerated signal), which is in contrast to the
findings by Buylaert et al. (2012).
}
\section{Function version}{
 0.7.10
}

\examples{

##load data
data(ExampleData.RLum.Analysis, envir = environment())

##(1) perform analysis using the method 'FIT'
results <- analyse_IRSAR.RF(object = IRSAR.RF.Data)

##show De results and test parameter results
get_RLum(results, data.object = "data")
get_RLum(results, data.object = "test_parameters")

##(2) perform analysis using the method 'SLIDE'
data(ExampleData.RF70Curves, envir = environment())
results <- analyse_IRSAR.RF(
object = RF70Curves,
method = "SLIDE",
 n.MC = 1)

\dontrun{
##(3) perform analysis using the method 'VSLIDE' and method control option
## 'trace
results <- analyse_IRSAR.RF(
 object = RF70Curves,
 method = "VSLIDE",
 method_control = list(trace = TRUE))
}

} 

\section{How to cite}{
Kreutzer, S., 2025. analyse_IRSAR.RF(): Analyse IRSAR RF measurements. Function version 0.7.10. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.1.2. https://r-lum.github.io/Luminescence/
}

\references{
Buylaert, J.P., Jain, M., Murray, A.S., Thomsen, K.J., Lapp, T.,
2012. IR-RF dating of sand-sized K-feldspar extracts: A test of accuracy.
Radiation Measurements 44 (5-6), 560-565. \doi{10.1016/j.radmeas.2012.06.021}

Erfurt, G., Krbetschek, M.R., 2003. IRSAR - A single-aliquot
regenerative-dose dating protocol applied to the infrared radiofluorescence
(IR-RF) of coarse- grain K-feldspar. Ancient TL 21, 35-42.
\doi{10.26034/la.atl.2003.358}

Erfurt, G., 2003. Infrared luminescence of Pb+ centres in potassium-rich
feldspars. physica status solidi (a) 200, 429-438. \doi{10.1002/pssa.200306700}

Erfurt, G., Krbetschek, M.R., 2003. Studies on the physics of the infrared
radioluminescence of potassium feldspar and on the methodology of its
application to sediment dating. Radiation Measurements 37, 505-510.
\doi{10.1016/s1350-4487(03)00058-1}

Erfurt, G., Krbetschek, M.R., Bortolot, V.J., Preusser, F., 2003. A fully
automated multi-spectral radioluminescence reading system for geochronometry
and dosimetry. Nuclear Instruments and Methods in Physics Research Section
B: Beam Interactions with Materials and Atoms 207, 487-499.
\doi{10.1016/s0168-583x(03)01121-2}

Frouin, M., Huot, S., Kreutzer, S., Lahaye, C., Lamothe, M., Philippe, A., Mercier, N., 2017.
An improved radiofluorescence single-aliquot regenerative dose protocol for K-feldspars.
Quaternary Geochronology 38, 13-24. \doi{10.1016/j.quageo.2016.11.004}

Kreutzer, S., Murari, M.K., Frouin, M., Fuchs, M., Mercier, N., 2017.
Always remain suspicious: a case study on tracking down a technical artefact while measuring IR-RF.
Ancient TL 35, 20–30. \doi{10.26034/la.atl.2017.510}

Murari, M.K., Kreutzer, S., Fuchs, M., 2018. Further investigations on IR-RF:
Dose recovery and correction. Radiation Measurements 120, 110–119.
doi: 10.1016/j.radmeas.2018.04.017 \doi{10.1016/j.radmeas.2018.04.017}

Lapp, T., Jain, M., Thomsen, K.J., Murray, A.S., Buylaert, J.P., 2012. New
luminescence measurement facilities in retrospective dosimetry. Radiation
Measurements 47, 803-808.
\doi{10.1016/j.radmeas.2012.02.006}

Trautmann, T., 2000. A study of radioluminescence kinetics of natural
feldspar dosimeters: experiments and simulations. Journal of Physics D:
Applied Physics 33, 2304-2310.
\doi{10.1088/0022-3727/33/18/315}

Trautmann, T., Krbetschek, M.R., Dietrich, A., Stolz, W., 1998.
Investigations of feldspar radioluminescence: potential for a new dating
technique. Radiation Measurements 29, 421-425.
\doi{10.1016/s1350-4487(98)00012-2}

Trautmann, T., Krbetschek, M.R., Dietrich, A., Stolz, W., 1999. Feldspar
radioluminescence: a new dating method and its physical background. Journal
of Luminescence 85, 45-58.
\doi{10.1016/s0022-2313(99)00152-0}

Trautmann, T., Krbetschek, M.R., Stolz, W., 2000. A systematic study of the
radioluminescence properties of single feldspar grains. Radiation
Measurements 32, 685-690.
\doi{10.1016/s1350-4487(00)00077-9}

** Further reading**

Murari, M.K., Kreutzer, S., King, G.E., Frouin, M., Tsukamoto, S., Schmidt, C., Lauer, T.,
Klasen, N., Richter, D., Friedrich, J., Mercier, N., Fuchs, M., 2021.
Infrared radiofluorescence (IR-RF) dating: A review. Quaternary Geochronology 64,
101155. \doi{10.1016/j.quageo.2021.101155}
}
\seealso{
\linkS4class{RLum.Analysis}, \linkS4class{RLum.Results}, \link{get_RLum},
\link{nls}, \link[minpack.lm:nlsLM]{minpack.lm::nlsLM}, \link[parallel:mclapply]{parallel::mclapply}, \link{ExampleData.RF70Curves}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{datagen}
