#' @title Check Whether a Q-Matrix is Identifiable
#'
#' @description
#' Checks whether a given Q-matrix satisfies the conditions for \strong{strict identifiability}
#' and \strong{generic identifiability} under cognitive diagnosis models (CDMs),
#' including the DINA, DINO, and saturated models, based on theoretical results from Gu & Xu (2021).
#'
#' This function evaluates both joint strict identifiability and various forms of generic identifiability,
#' including global and local cases, by verifying structural conditions on the Q-matrix.
#'
#' @details
#' The identifiability of the Q-matrix is essential for valid parameter estimation in CDMs.
#' According to Gu & Xu (2021), identifiability can be categorized into:
#'
#' \describe{
#'   \item{Joint Strict Identifiability}{
#'     Ensures that both the Q-matrix and model parameters can be uniquely determined from the data,
#'     currently only applicable under DINA or DINO models.
#'     Requires three conditions:
#'     \itemize{
#'       \item \strong{Completeness (A)}: The Q-matrix contains a \eqn{K \times K} identity submatrix (after row/column permutations).
#'       \item \strong{Distinctness (B)}: All columns of the remaining part (excluding the identity block) are distinct.
#'       \item \strong{Repetition (C)}: Each attribute appears in at least three items (i.e., column sums \eqn{\geq 3}).
#'     }
#'   }
#'
#'   \item{Joint Generic Identifiability (DINA/DINO)}{
#'     Means uniqueness holds for "almost all" parameter values (except on a set of measure zero).
#'     Applies when exactly one attribute has precisely two non-zero entries.
#'     The Q-matrix must have the structure:
#'     \eqn{\mathbf{Q} = \begin{pmatrix}
#'         1    & \mathbf{0}^\top \\
#'         1    & \mathbf{v}^\top \\
#'         \mathbf{0} & \mathbf{Q}^*
#'       \end{pmatrix}}.
#'     Then:
#'     \itemize{
#'       \item If \eqn{\mathbf{v} = \mathbf{0}}, then \eqn{\mathbf{Q}^*} must satisfy either:
#'         (i) Joint Strict Identifiability, or
#'         (ii) contain at least two identity submatrices → \strong{Globally generically identifiable}.
#'       \item If \eqn{\mathbf{v} \neq \mathbf{0}, \mathbf{1}}, then \eqn{\mathbf{Q}^*} must satisfy 
#'             Joint Strict Identifiability → \strong{Locally generically identifiable}.
#'     }
#'   }
#'
#'   \item{Joint Generic Identifiability (Saturated Model)}{
#'     For general saturated models, requires:
#'     \itemize{
#'       \item \strong{Generic Completeness (D)}: Two different \eqn{K \times K} submatrices exist,
#'             each having full rank and containing diagonal ones after permutation
#'             (indicating sufficient independent measurement of attributes).
#'       \item \strong{Generic Repetition (E)}: At least one '1' exists outside these two submatrices.
#'     }
#'   }
#' }
#' 
#' This function first reconstructs the Q-matrix into standard form,
#' then checks each condition accordingly.
#' 
#' @param Q An \eqn{I \times K} binary Q-matrix (\code{matrix} or \code{data.frame}), where
#'          each row represents an item and each column an attribute.
#'          Entries indicate whether an attribute is required (1) or not (0).
#' @param verbose Logical; if \code{TRUE}, prints warning messages during checking process
#'            (e.g., insufficient items, missing patterns).
#'            
#' @return
#' An object of class \code{"is.Qident"} — a list containing:
#' \describe{
#'  \item{\code{Q.orig}}{Original input Q-matrix.}
#'  \item{\code{Q.reconstructed}}{Reconstructed Q-matrix sorted by attribute pattern.}
#'  \item{\code{arguments}}{A list containing all input arguments}
#'  \item{\code{strictIdentifiability.obj}}{Results of checking Joint Strict Identifiability under DINA/DINO.
#'     Contains:
#'    \itemize{
#'       \item \code{completeness}: TRUE if \eqn{K \times K} identity submatrix exists.
#'       \item \code{distinctness}: TRUE if remaining columns are distinct.
#'       \item \code{repetition}: TRUE if every attribute appears more than 3 items.
#'    }
#'    All three must be TRUE for joint strict identifiability.}
#'  \item{\code{genericIdentifiability.obj}}{Results for Joint Generic Identifiability under saturated models.
#'    Includes:
#'    \itemize{
#'       \item \code{genericCompleteness}: TRUE if two different generic complete \eqn{K \times K} submatrices exist.
#'       \item \code{genericRepetition}: TRUE if at least one '1' exists outside those submatrices.
#'       \item \code{Q1}, \code{Q2}: Identified generic complete submatrices (if found).
#'       \item \code{Q.star}: Remaining part after removing rows in \code{Q1} and \code{Q2}.
#'    }
#'    Both \code{genericCompleteness} and \code{genericRepetition} must be TRUE.}
#'  \item{\code{genericIdentifiability.DINA.obj}}{Results for Joint Generic Identifiability under DINA/DINO.
#'     Includes:
#'     \itemize{
#'       \item \code{locallyGenericIdentifiability}: TRUE if local generic identifiability holds.
#'       \item \code{globallyGenericIdentifiability}: TRUE if global generic identifiability holds.
#'       \item \code{Q.reconstructed.DINA}: Reconstructed Q-matrix with low-frequency attribute moved to first column.}
#'     }
#' }
#' 
#' @author Haijiang Qin <Haijiang133@outlook.com>
#' 
#' @references
#' Gu, Y., & Xu, G. (2021). Sufficient and necessary conditions for the identifiability of the Q-matrix.
#' \emph{Statistica Sinica}, 31, 449–472. \url{https://www.jstor.org/stable/26969691}
#'
#' @seealso
#' \code{\link[Qval]{sim.Q}}, \code{\link[GDINA]{attributepattern}}
#'
#' @examples
#' library(Qval)
#' set.seed(123)
#'
#' # Simulate a 5-attribute, 20-item Q-matrix
#' Q <- sim.Q(5, 20)
#'
#' # Check identifiability
#' result <- is.Qident(Q, verbose = TRUE)
#'
#' # View summary
#' print(result)
#'
#' @export
is.Qident <- function(Q, verbose=TRUE){
  
  if(!is.matrix(Q) & !is.data.frame(Q)){
    stop("Error: Q must be a matrix or data.frame.")
  }
  
  is.Qidentcall <- match.call()
  
  Q <- as.matrix(Q)
  reconstruct.Q.obj <- reconstruct.Q(Q)
  num.identitySubmatrices <- reconstruct.Q.obj$num.identitySubmatrices
  pattern <- reconstruct.Q.obj$pattern
  Q.rec = reconstruct.Q.obj$Q.reconstructed
  Q.rec.pattern <- reconstruct.Q.obj$Q.pattern.reconstructe
  
  I <- nrow(Q.rec)
  K <- ncol(Q.rec)
  if(any(is.null(colnames(Q.rec)) | is.na(colnames(Q.rec)))){
    rownames(Q.rec) <- 1:I
  }
  
  strictIdentifiability.obj <- is.StrictIdentifiability(Q.rec, pattern, Q.rec.pattern, verbose)
  genericIdentifiability.obj <- is.genericIdentifiability(Q.rec, verbose)
  genericIdentifiability.DINA.obj <- is.genericIdentifiability.DINA(Q.rec, pattern, 
                                                                Q.rec.pattern, 
                                                                strictIdentifiability.obj, 
                                                                verbose)
  
  res <- list(Q.orig=Q, 
              Q.reconstructed=Q.rec, 
              strictIdentifiability.obj=strictIdentifiability.obj, 
              genericIdentifiability.obj=genericIdentifiability.obj, 
              genericIdentifiability.DINA.obj=genericIdentifiability.DINA.obj, 
              call = is.Qidentcall, 
              arguments = list(
                Q=Q, verbose=verbose
              ))
  class(res) <- "is.Qident"
  
  return(res)

}

#' @importFrom GDINA attributepattern
reconstruct.Q <- function(Q, pattern=NULL){
  I <- nrow(Q)
  K <- ncol(Q)
  
  if(is.null(pattern)){
    pattern <- GDINA::attributepattern(K)
  }
  
  Q.reconstructed <- NULL
  Q.pattern <- apply(Q, 1, function(x) get_Pattern(x, pattern))
  
  num.pattern1 <- table(Q.pattern[which(rowSums(Q) == 1)])
  num.identitySubmatrices <- ifelse(length(num.pattern1), min(num.pattern1), 0)
  if(length(num.pattern1) > 0){
    while (any(num.pattern1 > 0)) {
      Q.pattern.cur <- as.numeric(names(num.pattern1)[which(num.pattern1>0)])
      Q.reconstructed <- rbind(Q.reconstructed, pattern[Q.pattern.cur, ])
      num.pattern1 <- num.pattern1 - 1
    }
  }
  
  Q.reconstructed <- rbind(Q.reconstructed, pattern[sort(Q.pattern[which(rowSums(Q) > 1)]), ])
  Q.pattern.reconstructe <- apply(Q.reconstructed, 1, function(x) get_Pattern(x, pattern))

  return(list(Q=Q, Q.reconstructed=Q.reconstructed, 
              num.identitySubmatrices=num.identitySubmatrices, 
              pattern=pattern, Q.pattern=Q.pattern, 
              Q.pattern.reconstructe=Q.pattern.reconstructe))
}

is.completeness <- function(Q, pattern, Q.pattern, verbose){
  if(nrow(Q) < ncol(Q)){
    if(verbose)
      warning("The Q-matrix does not have enough items to achieve Generic Completeness.")
    return(FALSE)
  }
  num.pattern1 <- table(Q.pattern[which(rowSums(Q) == 1)])
  if(identical(as.numeric(c(1:ncol(Q))), (as.numeric(names(num.pattern1)) - 1))){
    return(TRUE)
  }
  return(FALSE)
}

is.distinctness <- function(Q.star){
  Q.star.t <- t(Q.star)
  if(nrow(unique(Q.star.t)) == ncol(Q.star)){
    return(TRUE)
  }
  return(FALSE)
}

is.repetition <- function(Q){
  if(all(colSums(Q) >= 3)){
    return(TRUE)
  }
  return(FALSE)
}

is.StrictIdentifiability <- function(Q, pattern, Q.pattern, verbose){
  
  K <- ncol(Q)
  I <- nrow(Q)
  
  completeness <- is.completeness(Q, pattern, Q.pattern, verbose)
  
  distinctness <- FALSE
  if(completeness){
    Q.star <- Q[(K+1):I, ]
    distinctness <- is.distinctness(Q.star)
  }
  
  repetition <- is.repetition(Q)
  
  return(list(completeness=completeness, 
              distinctness=distinctness, 
              repetition=repetition))
}

is.genericIdentifiability.DINA <- function(Q, pattern, Q.pattern, strictIdentifiability.obj, verbose){
  
  I <- nrow(Q)
  K <- ncol(Q)
  num.attributes <- colSums(Q)
  
  if (all(unlist(strictIdentifiability.obj))) {
    return(list(locallyGenericIdentifiability=TRUE, 
                globallyGenericIdentifiability=TRUE, 
                Q.reconstructed.DINA=Q))
  }
  
  attribute.k2 <- which(num.attributes == 2)
  if(any(num.attributes < 2) | length(attribute.k2) >= 2 | any(!unlist(strictIdentifiability.obj)[1:2])){
    return(list(locallyGenericIdentifiability=FALSE, 
                globallyGenericIdentifiability=FALSE, 
                Q.reconstructed.DINA=NA))
  }
  
  
  locallyGenericIdentifiability <- globallyGenericIdentifiability <- FALSE
  Q.temp <- Q
  items.posi <- which(Q.temp[, attribute.k2] == 1)
  Q.temp[, 1] <- Q[, attribute.k2]
  Q.temp[, attribute.k2] <- Q[, 1]
  qv <- rowSums(Q.temp[1:2, ])
  
  if(all(qv != 1)){
    return(list(locallyGenericIdentifiability=FALSE, 
                globallyGenericIdentifiability=FALSE, 
                Q.reconstructed.DINA=NA))
  }else if(all(qv == 1)){
    Q.star <- Q.temp[, -1]
    Q.star <- Q.star[-c(1, 2), ]
    
    pattern.star <- GDINA::attributepattern(ncol(Q.star))
    Q.star.pattern <- apply(Q.star, 1, function(x) get_Pattern(x, pattern.star))
    StrictIdentifiability.Qstar.obj <- is.StrictIdentifiability(Q.star, pattern.star, Q.star.pattern, verbose)
    
    Q.star.rec.obj <- reconstruct.Q(Q.star)
    
    if(all(unlist(StrictIdentifiability.Qstar.obj)) | Q.star.rec.obj$num.identitySubmatrices >= 2){
      return(list(locallyGenericIdentifiability=TRUE, 
                  globallyGenericIdentifiability=TRUE, 
                  Q.reconstructed.DINA=Q.temp))
    }
    
  }else if(all(qv %in% c(1:(K-1)))){
    Q.star <- Q.temp[, -1]
    Q.star <- Q.star[-c(1, 2), ]
    
    pattern.star <- GDINA::attributepattern(ncol(Q.star))
    Q.star.pattern <- apply(Q.star, 1, function(x) get_Pattern(x, pattern.star))
    StrictIdentifiability.Qstar.obj <- is.StrictIdentifiability(Q.star, pattern.star, Q.star.pattern, verbose)
    if(all(unlist(StrictIdentifiability.Qstar.obj))){
      return(list(locallyGenericIdentifiability=TRUE, 
                  globallyGenericIdentifiability=FALSE, 
                  Q.reconstructed.DINA=Q.temp))
    }
  }
  
  return(list(locallyGenericIdentifiability=FALSE, 
              globallyGenericIdentifiability=FALSE, 
              Q.reconstructed.DINA=NA))
}

is.genericCompleteness <- function(Q, verbose) {
  I <- nrow(Q)
  K <- ncol(Q)
  
  if (I < 2 * K) {
    if(verbose)
      warning("Not enough items for Generic Completeness.")
    gr <- is.genericRepetition(Q, verbose)
    return(list(genericCompleteness = FALSE, genericRepetition = gr, 
                Q1 = NA, Q2 = NA, Q.star = if (gr) Q else NA))
  }
  
  if(any(colSums(Q) == 1)){
    if(verbose)
      warning("Attribute ", paste0(names(which(colSums(Q) == 1)), ", "), 
              " was contained only 2 times in the Q-matrix, \nthus cannot satisfy both Generic Completeness and Generic Repetition.")
    return(list(genericCompleteness = FALSE, genericRepetition = FALSE, 
                Q1 = NA, Q2 = NA, Q.star = NA))
  }
  
  attributes.posi <- lapply(1:K, function(k) which(Q[, k] == 1))
  if(any(colSums(Q) == 2)){ 
    if(verbose)
      warning("Attribute ", paste0(names(which(colSums(Q) == 2)), ", "), 
              " was contained only 2 times in the Q-matrix, \nthus cannot satisfy Generic Repetition.")
    backtrack.Completeness <- function(k, S1, S2) {
      if (k > K) {
        Q1 <- Q[S1, , drop = FALSE]
        Q2 <- Q[S2, , drop = FALSE]
        if (sum(abs(Q1 - Q2)) == 0) return(NULL)
        
        remain_idx <- setdiff(1:I, c(S1, S2))
        Q.star <- if (length(remain_idx) == 0) matrix(0, 0, K) else Q[remain_idx, , drop = FALSE]
        return(list(Q1 = Q1, Q2 = Q2, Q.star = Q.star))
      }

      available <- setdiff(attributes.posi[[k]], c(S1, S2))
      if (length(available) < 2) return(NULL)
      
      for (i in 1:(length(available)-1)) {
        for (j in (i+1):length(available)) {
          a <- available[i]
          b <- available[j]
          res1 <- backtrack.Completeness(k + 1, c(S1, a), c(S2, b))
          if (!is.null(res1)) return(res1)
          res2 <- backtrack.Completeness(k + 1, c(S1, b), c(S2, a))
          if (!is.null(res2)) return(res2)
        }
      }
      return(NULL)
    }
    
    result <- backtrack.Completeness(1, integer(0), integer(0))
    if (is.null(result)) {
      gr <- is.genericRepetition(Q, verbose)
      return(list(genericCompleteness = FALSE, genericRepetition = gr, 
                  Q1 = NA, Q2 = NA, Q.star = if (gr) Q else NA))
    } else {
      return(list(genericCompleteness = TRUE, genericRepetition = FALSE, 
                  Q1 = result$Q1, Q2 = result$Q2, Q.star = result$Q.star))
    }
  }
  
  backtrack.CompletenessAndRepetition <- function(k, S1, S2) {
    if (k > K) {
      Q1 <- Q[S1, , drop = FALSE]
      Q2 <- Q[S2, , drop = FALSE]
      if (sum(abs(Q1 - Q2)) == 0) return(NULL)
      
      remain_idx <- setdiff(1:I, c(S1, S2))
      Q.star <- if (length(remain_idx) == 0) matrix(0, 0, K) else Q[remain_idx, , drop = FALSE]
      if (is.genericRepetition(Q, verbose)) {
        return(list(Q1 = Q1, Q2 = Q2, Q.star = Q.star))
      }
      return(NULL)
    }
    
    available <- setdiff(attributes.posi[[k]], c(S1, S2))
    if (length(available) < 2) return(NULL)
    
    for (i in 1:(length(available)-1)) {
      for (j in (i+1):length(available)) {
        a <- available[i]
        b <- available[j]
        res1 <- backtrack.CompletenessAndRepetition(k + 1, c(S1, a), c(S2, b))
        if (!is.null(res1)) return(res1)
        res2 <- backtrack.CompletenessAndRepetition(k + 1, c(S1, b), c(S2, a))
        if (!is.null(res2)) return(res2)
      }
    }
    return(NULL)
  }
  result <- backtrack.CompletenessAndRepetition(1, integer(0), integer(0))
  if (is.null(result)) {
    gr <- is.genericRepetition(Q, verbose)
    return(list(genericCompleteness = FALSE, genericRepetition = gr, Q1 = NA, Q2 = NA, Q.star = if (gr) Q else NA))
  } else {
    return(list(genericCompleteness = TRUE, genericRepetition = TRUE, 
                Q1 = result$Q1, Q2 = result$Q2, Q.star = result$Q.star))
  }
}

is.genericRepetition <- function(Q, verbose){
  if(nrow(Q) == 0){
    if(verbose)
      warning("Not enough items for Generic Repetition.")
    return(FALSE)
  }else{
    if(all(colSums(Q) >= 3)){
      return(TRUE)
    }
  }
  return(FALSE)
}

is.genericIdentifiability <- function(Q, verbose){
  genericCompleteness.obj <- is.genericCompleteness(Q, verbose)
  return(genericCompleteness.obj)
}