% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IndexPriority.R
\name{get.priority}
\alias{get.priority}
\title{Priority of Attribute}
\usage{
get.priority(
  Y = NULL,
  Q = NULL,
  att.str = NULL,
  CDM.obj = NULL,
  mono.constraint = FALSE,
  model = "GDINA"
)
}
\arguments{
\item{Y}{A required \eqn{N} × \eqn{I} matrix or \code{data.frame} consisting of the responses of \code{N} individuals
to \eqn{N} × \eqn{I} items. Missing values need to be coded as \code{NA}.}

\item{Q}{A required binary \eqn{I} × \eqn{K} matrix containing the attributes not required or required 
master the items. The \code{i}th row of the matrix is a binary indicator vector indicating which
attributes are not required (coded by 0) and which attributes are required (coded by 1) to master
item \eqn{i}.}

\item{att.str}{Specify attribute structures. \code{NULL}, by default, means there is no structure. Attribute structure 
needs be specified as a list - which will be internally handled by \code{\link[GDINA]{att.structure}} function. 
See examples. It can also be a matrix giving all permissible attribute profiles.}

\item{CDM.obj}{An object of class \code{CDM.obj}. When it is not NULL, it enables rapid validation
of the Q-matrix without the need for parameter estimation. @seealso \code{\link[Qval]{CDM}}.}

\item{mono.constraint}{Logical indicating whether monotonicity constraints should be fulfilled in estimation.
Default = \code{FALSE}.}

\item{model}{Type of model to fit; can be \code{"GDINA"}, \code{"LCDM"}, \code{"DINA"}, \code{"DINO"}
, \code{"ACDM"}, \code{"LLM"}, or \code{"rRUM"}. Default = \code{"GDINA"}.
@seealso \code{\link[Qval]{CDM}}.}
}
\value{
A matrix containing all attribute priorities.
}
\description{
This function will provide the priorities of attributes for all items.
}
\details{
The calculation of priorities is straightforward (Qin & Guo, 2025): the priority of an attribute is the 
regression coefficient obtained from a LASSO multinomial logistic regression, with the attribute 
as the independent variable and the response data from the examinees as the dependent variable.  
The formula (Tu et al., 2022) is as follows:

\deqn{
 \log[\frac{P(X_{pi} = 1 | \boldsymbol{\Lambda}_{p})}{P(X_{pi} = 0 | \boldsymbol{\Lambda}_{p})}] = 
 logit[P(X_{pi} = 1 | \boldsymbol{\Lambda}_{p})] = 
 \beta_{i0} + \beta_{i1} \Lambda_{p1} + \ldots + \beta_{ik} \Lambda_{pk} + \ldots + \beta_{iK} \Lambda_{pK}
}

Where \eqn{X_{pi}} represents the response of examinee \eqn{p} on item \eqn{i},  
\eqn{\boldsymbol{\Lambda}_{p}} denotes the marginal mastery probabilities of examinee \eqn{p}  
(which can be obtained from the return value \code{alpha.P} of the \code{\link[Qval]{CDM}} function),  
\eqn{\beta_{i0}} is the intercept term, and \eqn{\beta_{ik}} represents the regression coefficient.  

The LASSO loss function can be expressed as:

\deqn{l_{lasso}(\boldsymbol{X}_i | \boldsymbol{\Lambda}) = l(\boldsymbol{X}_i | \boldsymbol{\Lambda}) - \lambda |\boldsymbol{\beta}_i|}

Where \eqn{l_{lasso}(\boldsymbol{X}_i | \boldsymbol{\Lambda})} is the penalized likelihood,  
\eqn{l(\boldsymbol{X}_i | \boldsymbol{\Lambda})} is the original likelihood,  
and \eqn{\lambda} is the tuning parameter for penalization (a larger value imposes a stronger penalty on 
\eqn{\boldsymbol{\beta}_i = [\beta_{i1}, \ldots, \beta_{ik}, \ldots, \beta_{iK}]}).  
The priority for attribute \eqn{i} is defined as: \eqn{\boldsymbol{priority}_i = \boldsymbol{\beta}_i = [\beta_{i1}, \ldots, \beta_{ik}, \ldots, \beta_{iK}]}
}
\examples{
\donttest{
set.seed(123)
library(Qval)

## generate Q-matrix and data
K <- 5
I <- 20
IQ <- list(
  P0 = runif(I, 0.1, 0.3),
  P1 = runif(I, 0.7, 0.9)
)

Q <- sim.Q(K, I)
data <- sim.data(Q = Q, N = 500, IQ = IQ, model = "GDINA", distribute = "horder")
MQ <- sim.MQ(Q, 0.1)

CDM.obj <- CDM(data$dat, MQ)

priority <- get.priority(data$dat, Q, CDM.obj=CDM.obj)
head(priority)
}


}
\references{
Qin, H., & Guo, L. (2025). Priority attribute algorithm for Q-matrix validation: A didactic. Behavior Research Methods, 57(1), 31. DOI: 10.3758/s13428-024-02547-5.

Tu, D., Chiu, J., Ma, W., Wang, D., Cai, Y., & Ouyang, X. (2022). A multiple logistic regression-based (MLR-B) Q-matrix validation method for cognitive diagnosis models: A confirmatory approach. Behavior Research Methods. DOI: 10.3758/s13428-022-01880-x.
}
\seealso{
\code{\link[Qval]{validation}}
}
