% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/is.Qident.R
\name{is.Qident}
\alias{is.Qident}
\title{Check Whether a Q-Matrix is Identifiable}
\usage{
is.Qident(Q, verbose = TRUE)
}
\arguments{
\item{Q}{An \eqn{I \times K} binary Q-matrix (\code{matrix} or \code{data.frame}), where
each row represents an item and each column an attribute.
Entries indicate whether an attribute is required (1) or not (0).}

\item{verbose}{Logical; if \code{TRUE}, prints warning messages during checking process
(e.g., insufficient items, missing patterns).}
}
\value{
An object of class \code{"is.Qident"} — a list containing:
\describe{
 \item{\code{Q.orig}}{Original input Q-matrix.}
 \item{\code{Q.reconstructed}}{Reconstructed Q-matrix sorted by attribute pattern.}
 \item{\code{arguments}}{A list containing all input arguments}
 \item{\code{strictIdentifiability.obj}}{Results of checking Joint Strict Identifiability under DINA/DINO.
    Contains:
   \itemize{
      \item \code{completeness}: TRUE if \eqn{K \times K} identity submatrix exists.
      \item \code{distinctness}: TRUE if remaining columns are distinct.
      \item \code{repetition}: TRUE if every attribute appears more than 3 items.
   }
   All three must be TRUE for joint strict identifiability.}
 \item{\code{genericIdentifiability.obj}}{Results for Joint Generic Identifiability under saturated models.
   Includes:
   \itemize{
      \item \code{genericCompleteness}: TRUE if two different generic complete \eqn{K \times K} submatrices exist.
      \item \code{genericRepetition}: TRUE if at least one '1' exists outside those submatrices.
      \item \code{Q1}, \code{Q2}: Identified generic complete submatrices (if found).
      \item \code{Q.star}: Remaining part after removing rows in \code{Q1} and \code{Q2}.
   }
   Both \code{genericCompleteness} and \code{genericRepetition} must be TRUE.}
 \item{\code{genericIdentifiability.DINA.obj}}{Results for Joint Generic Identifiability under DINA/DINO.
    Includes:
    \itemize{
      \item \code{locallyGenericIdentifiability}: TRUE if local generic identifiability holds.
      \item \code{globallyGenericIdentifiability}: TRUE if global generic identifiability holds.
      \item \code{Q.reconstructed.DINA}: Reconstructed Q-matrix with low-frequency attribute moved to first column.}
    }
}
}
\description{
Checks whether a given Q-matrix satisfies the conditions for \strong{strict identifiability}
and \strong{generic identifiability} under cognitive diagnosis models (CDMs),
including the DINA, DINO, and saturated models, based on theoretical results from Gu & Xu (2021).

This function evaluates both joint strict identifiability and various forms of generic identifiability,
including global and local cases, by verifying structural conditions on the Q-matrix.
}
\details{
The identifiability of the Q-matrix is essential for valid parameter estimation in CDMs.
According to Gu & Xu (2021), identifiability can be categorized into:

\describe{
  \item{Joint Strict Identifiability}{
    Ensures that both the Q-matrix and model parameters can be uniquely determined from the data,
    currently only applicable under DINA or DINO models.
    Requires three conditions:
    \itemize{
      \item \strong{Completeness (A)}: The Q-matrix contains a \eqn{K \times K} identity submatrix (after row/column permutations).
      \item \strong{Distinctness (B)}: All columns of the remaining part (excluding the identity block) are distinct.
      \item \strong{Repetition (C)}: Each attribute appears in at least three items (i.e., column sums \eqn{\geq 3}).
    }
  }

  \item{Joint Generic Identifiability (DINA/DINO)}{
    Means uniqueness holds for "almost all" parameter values (except on a set of measure zero).
    Applies when exactly one attribute has precisely two non-zero entries.
    The Q-matrix must have the structure:
    \eqn{\mathbf{Q} = \begin{pmatrix}
        1    & \mathbf{0}^\top \\
        1    & \mathbf{v}^\top \\
        \mathbf{0} & \mathbf{Q}^*
      \end{pmatrix}}.
    Then:
    \itemize{
      \item If \eqn{\mathbf{v} = \mathbf{0}}, then \eqn{\mathbf{Q}^*} must satisfy either:
        (i) Joint Strict Identifiability, or
        (ii) contain at least two identity submatrices → \strong{Globally generically identifiable}.
      \item If \eqn{\mathbf{v} \neq \mathbf{0}, \mathbf{1}}, then \eqn{\mathbf{Q}^*} must satisfy 
            Joint Strict Identifiability → \strong{Locally generically identifiable}.
    }
  }

  \item{Joint Generic Identifiability (Saturated Model)}{
    For general saturated models, requires:
    \itemize{
      \item \strong{Generic Completeness (D)}: Two different \eqn{K \times K} submatrices exist,
            each having full rank and containing diagonal ones after permutation
            (indicating sufficient independent measurement of attributes).
      \item \strong{Generic Repetition (E)}: At least one '1' exists outside these two submatrices.
    }
  }
}

This function first reconstructs the Q-matrix into standard form,
then checks each condition accordingly.
}
\examples{
library(Qval)
set.seed(123)

# Simulate a 5-attribute, 20-item Q-matrix
Q <- sim.Q(5, 20)

# Check identifiability
result <- is.Qident(Q, verbose = TRUE)

# View summary
print(result)

}
\references{
Gu, Y., & Xu, G. (2021). Sufficient and necessary conditions for the identifiability of the Q-matrix.
\emph{Statistica Sinica}, 31, 449–472. \url{https://www.jstor.org/stable/26969691}
}
\seealso{
\code{\link[Qval]{sim.Q}}, \code{\link[GDINA]{attributepattern}}
}
\author{
Haijiang Qin <Haijiang133@outlook.com>
}
