#' Julia Backend Integration for RegimeChange
#'
#' This file provides the R-Julia bridge for high-performance computation.
#' When Julia is available and initialized, methods automatically dispatch
#' to optimized Julia implementations.
#'
#' @name julia-backend
#' @noRd
NULL

#' Initialize Julia Backend
#'
#' Explicitly initialize the Julia backend for improved performance.
#' This loads the RegimeChangeJulia module and makes Julia functions available.
#'
#' @param force Logical. Force reinitialization even if already initialized.
#'   Default is FALSE.
#' @param num_threads Integer or NULL. Number of Julia threads to use.
#'   Default is NULL (auto-detect).
#'
#' @return Logical indicating success (invisibly).
#' @export
#'
#' @examples
#' \dontrun{
#' init_julia()
#' julia_available()
#' result <- detect_regimes(data, method = "pelt")
#' }
init_julia <- function(force = FALSE, num_threads = NULL) {
  
  if (!requireNamespace("JuliaCall", quietly = TRUE)) {
    cli::cli_warn(c(
      "JuliaCall package not installed.",
      "i" = "Please install the JuliaCall package.",
      "i" = "Also requires Julia >= 1.6 installed on system."
    ))
    return(invisible(FALSE))
  }
  
  if (.pkg_env$julia_initialized && !force) {
    cli::cli_inform("Julia backend already initialized.")
    return(invisible(TRUE))
  }
  
  tryCatch({
    if (!is.null(num_threads)) {
      old_threads <- Sys.getenv("JULIA_NUM_THREADS")
      Sys.setenv(JULIA_NUM_THREADS = as.character(num_threads))
      on.exit(Sys.setenv(JULIA_NUM_THREADS = old_threads), add = TRUE)
    }
    
    cli::cli_inform("Initializing Julia...")
    JuliaCall::julia_setup(verbose = FALSE)
    
    JuliaCall::julia_library("Statistics")
    JuliaCall::julia_library("LinearAlgebra")
    
    julia_src <- system.file("julia", "RegimeChangeJulia.jl",
                             package = "RegimeChange")
    
    if (file.exists(julia_src)) {
      JuliaCall::julia_source(julia_src)
      JuliaCall::julia_command("using .RegimeChangeJulia")
      
      .pkg_env$julia_initialized <- TRUE
      .pkg_env$julia_available <- TRUE
      
      cli::cli_alert_success("Julia backend initialized successfully.")
      cli::cli_inform(c(
        "i" = "Julia version: {JuliaCall::julia_eval('VERSION')}",
        "i" = "Threads available: {JuliaCall::julia_eval('Threads.nthreads()')}"
      ))
      
      invisible(TRUE)
    } else {
      cli::cli_warn("Julia module not found in package installation.")
      invisible(FALSE)
    }
    
  }, error = function(e) {
    cli::cli_warn(c(
      "Failed to initialize Julia backend.",
      "x" = conditionMessage(e),
      "i" = "The package will use R implementations instead."
    ))
    .pkg_env$julia_initialized <- FALSE
    invisible(FALSE)
  })
}

#' Check if Julia Backend is Available
#'
#' @return Logical indicating if Julia is available and initialized.
#' @export
julia_available <- function() {
  
  .pkg_env$julia_available && .pkg_env$julia_initialized
}

#' Get Julia Backend Status
#'
#' @return List with Julia status information.
#' @export
julia_status <- function() {
  
  list(
    available = .pkg_env$julia_available,
    initialized = .pkg_env$julia_initialized,
    version = if (julia_available()) {
      tryCatch(
        as.character(JuliaCall::julia_eval("string(VERSION)")),
        error = function(e) NA_character_
      )
    } else NA_character_,
    threads = if (julia_available()) {
      tryCatch(
        as.integer(JuliaCall::julia_eval("Threads.nthreads()")),
        error = function(e) NA_integer_
      )
    } else NA_integer_
  )
}

#' @noRd
pelt_julia <- function(data, penalty, min_segment, cost_type = "mean") {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  data_jl <- as.numeric(data)
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.pelt_detect",
    data_jl,
    as.numeric(penalty),
    as.integer(min_segment),
    cost_type = cost_type
  )
  
  as.integer(result)
}

#' @noRd
fpop_julia <- function(data, penalty, min_segment) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.fpop_detect",
    as.numeric(data),
    as.numeric(penalty),
    as.integer(min_segment)
  )
  
  as.integer(result)
}

#' @noRd
bocpd_julia <- function(data, hazard_rate, mu0, kappa0, alpha0, beta0,
                        threshold = 0.5, max_runlen = 500L) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  JuliaCall::julia_command(sprintf(
    "prior = RegimeChangeJulia.NormalGammaPrior(%f, %f, %f, %f)",
    mu0, kappa0, alpha0, beta0
  ))
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.bocpd_detect",
    as.numeric(data),
    as.numeric(hazard_rate),
    prior = JuliaCall::julia_eval("prior"),
    threshold = as.numeric(threshold),
    max_runlen = as.integer(max_runlen)
  )
  
  list(
    changepoints = as.integer(result$changepoints),
    prob_change = as.numeric(result$prob_change),
    map_runlen = as.integer(result$map_runlen)
  )
}

#' @noRd
cusum_julia <- function(data, threshold, baseline_n = 50L, two_sided = TRUE) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.cusum_detect",
    as.numeric(data),
    as.numeric(threshold),
    baseline_n = as.integer(baseline_n),
    two_sided = two_sided
  )
  
  list(
    changepoints = as.integer(result$changepoints),
    S_plus = as.numeric(result$S_plus),
    S_minus = as.numeric(result$S_minus)
  )
}

#' @noRd
kernel_cpd_julia <- function(data, penalty, kernel = "rbf",
                             bandwidth = 0, min_segment = 10L) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.kernel_cpd_detect",
    as.numeric(data),
    as.numeric(penalty),
    kernel = kernel,
    bandwidth = as.numeric(bandwidth),
    min_segment = as.integer(min_segment)
  )
  
  as.integer(result)
}

#' @noRd
wbs_julia <- function(data, penalty, n_intervals = 100L,
                      min_segment = 5L, threshold = 0) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.wbs_detect",
    as.numeric(data),
    as.numeric(penalty),
    n_intervals = as.integer(n_intervals),
    min_segment = as.integer(min_segment),
    threshold = as.numeric(threshold)
  )
  
  as.integer(result)
}

#' @noRd
segneigh_julia <- function(data, n_changepoints, min_segment = 2L,
                           cost_type = "mean") {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.segneigh_detect",
    as.numeric(data),
    as.integer(n_changepoints),
    min_segment = as.integer(min_segment),
    cost_type = cost_type
  )
  
  as.integer(result)
}

#' @noRd
pelt_multivariate_julia <- function(data, penalty, min_segment) {
  if (!julia_available()) {
    cli::cli_abort("Julia backend not available. Call init_julia() first.")
  }
  
  if (!is.matrix(data)) {
    data <- as.matrix(data)
  }
  
  result <- JuliaCall::julia_call(
    "RegimeChangeJulia.pelt_multivariate",
    data,
    as.numeric(penalty),
    as.integer(min_segment)
  )
  
  as.integer(result)
}

#' @noRd
select_backend <- function(method, n, use_julia = "auto") {
  if (is.logical(use_julia)) {
    if (use_julia && julia_available()) {
      return("julia")
    } else {
      return("r")
    }
  }
  
  if (use_julia == "auto" && julia_available()) {
    if (n > 1000) {
      return("julia")
    }
  }
  
  return("r")
}

#' @noRd
run_with_backend <- function(method, data, use_julia = "auto", ...) {
  n <- if (is.matrix(data)) nrow(data) else length(data)
  backend <- select_backend(method, n, use_julia)
  
  if (backend == "julia") {
    cli::cli_inform("Using Julia backend for {method}")
  }
  
  switch(method,
         "pelt" = if (backend == "julia") pelt_julia(data, ...) else pelt(data, ...),
         "fpop" = if (backend == "julia") fpop_julia(data, ...) else fpop_detect(data, ...),
         "bocpd" = if (backend == "julia") bocpd_julia(data, ...) else bocpd(data, ...),
         "cusum" = if (backend == "julia") cusum_julia(data, ...) else cusum(data, ...),
         "kernel" = if (backend == "julia") kernel_cpd_julia(data, ...) else kernel_cpd_detect(data, ...),
         "wbs" = if (backend == "julia") wbs_julia(data, ...) else wild_binary_segmentation(data, ...),
         cli::cli_abort("Unknown method: {method}")
  )
}

#' Benchmark Julia vs R Performance
#'
#' Compare execution time between Julia and R implementations.
#'
#' @param data Test data.
#' @param method Method to benchmark.
#' @param n_reps Number of repetitions.
#' @param ... Additional arguments for the method.
#'
#' @return Data frame with timing results (invisibly).
#' @export
#'
#' @examples
#' \donttest{
#' data <- c(rnorm(500), rnorm(500, 3))
#' benchmark_backends(data, method = "pelt", penalty = log(1000))
#' }
benchmark_backends <- function(data, method = "pelt", n_reps = 10, ...) {
  results <- data.frame(
    backend = character(),
    rep = integer(),
    time_ms = numeric(),
    stringsAsFactors = FALSE
  )
  
  for (i in seq_len(n_reps)) {
    t_start <- Sys.time()
    run_with_backend(method, data, use_julia = FALSE, ...)
    t_end <- Sys.time()
    
    results <- rbind(results, data.frame(
      backend = "R",
      rep = i,
      time_ms = as.numeric(difftime(t_end, t_start, units = "secs")) * 1000
    ))
  }
  
  if (julia_available()) {
    run_with_backend(method, data, use_julia = TRUE, ...)
    
    for (i in seq_len(n_reps)) {
      t_start <- Sys.time()
      run_with_backend(method, data, use_julia = TRUE, ...)
      t_end <- Sys.time()
      
      results <- rbind(results, data.frame(
        backend = "Julia",
        rep = i,
        time_ms = as.numeric(difftime(t_end, t_start, units = "secs")) * 1000
      ))
    }
  }
  
  summary_stats <- stats::aggregate(
    time_ms ~ backend,
    data = results,
    FUN = function(x) c(mean = mean(x), sd = sd(x), median = median(x))
  )
  
  cli::cli_h2("Benchmark Results")
  print(summary_stats)
  
  invisible(results)
}