# Tests for frequentist changepoint detection methods
# Test file: tests/testthat/test-frequentist.R

# Helper function to generate test data with known changepoints
generate_test_data <- function(n = 200, changepoints = 100, means = c(0, 3), sd = 1) {
  if (length(changepoints) == 1 && changepoints < n) {
    c(rnorm(changepoints, means[1], sd), 
      rnorm(n - changepoints, means[2], sd))
  } else {
    rnorm(n, means[1], sd)
  }
}

# =============================================================================
# CUSUM Tests
# =============================================================================

test_that("CUSUM returns proper structure", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- cusum(data)
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

test_that("CUSUM handles no changepoint case", {
  set.seed(123)
  data <- rnorm(200)  # No changepoint
  
  result <- cusum(data)
  
  expect_type(result, "list")
})

# =============================================================================
# PELT Tests
# =============================================================================

test_that("PELT returns proper structure", {
  set.seed(123)
  data <- generate_test_data(n = 200, changepoints = 100)
  
  result <- pelt(data)
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

test_that("PELT supports different penalty types", {
  set.seed(123)
  data <- generate_test_data()
  
  result_bic <- pelt(data, penalty = "bic")
  result_aic <- pelt(data, penalty = "aic")
  
  expect_type(result_bic, "list")
  expect_type(result_aic, "list")
})

test_that("PELT detects variance changes", {
  set.seed(123)
  data <- c(rnorm(100, 0, 1), rnorm(100, 0, 3))  # Variance change
  
  result <- pelt(data, type = "variance")
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

test_that("PELT respects min_segment parameter", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- pelt(data, min_segment = 50)
  
  expect_type(result, "list")
})

# =============================================================================
# Binary Segmentation Tests
# =============================================================================

test_that("Binary segmentation returns proper structure", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- binary_segmentation(data)
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

test_that("Binary segmentation respects n_changepoints", {
  set.seed(123)
  data <- generate_test_data(n = 300, changepoints = 100)
  
  result <- binary_segmentation(data, n_changepoints = 2)
  
  expect_true(length(result$changepoints) <= 2)
})

# =============================================================================
# Wild Binary Segmentation Tests
# =============================================================================

test_that("WBS returns proper structure", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- wild_binary_segmentation(data)
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

test_that("WBS respects n_changepoints", {
  set.seed(123)
  data <- generate_test_data(n = 300, changepoints = 100)
  
  result <- wild_binary_segmentation(data, n_changepoints = 1)
  
  expect_true(length(result$changepoints) <= 1)
})

# =============================================================================
# Cost Function Tests
# =============================================================================

test_that("cost_univariate computes correctly", {
  data <- c(1, 2, 3, 4, 5)
  cost <- cost_univariate(data, type = "mean")
  
  expect_type(cost, "double")
  expect_true(is.finite(cost))
})

test_that("cost_univariate handles edge cases", {
  # Single observation
  expect_equal(cost_univariate(5, type = "mean"), 0)
  
  # Constant data
  data <- rep(3, 10)
  expect_equal(cost_univariate(data, type = "mean"), 0)
})

# =============================================================================
# Detect Ensemble Tests
# =============================================================================

test_that("detect_ensemble combines multiple methods", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- detect_ensemble(data, methods = c("pelt", "cusum"))
  
  expect_type(result, "list")
  expect_true("changepoints" %in% names(result))
})

# =============================================================================
# Edge Cases
# =============================================================================

test_that("methods handle short time series", {
  data <- rnorm(30)
  
  result <- pelt(data, min_segment = 5)
  expect_type(result, "list")
})

test_that("methods handle constant data", {
  data <- rep(5, 100)
  
  result <- pelt(data, type = "mean")
  
  # Should detect no changepoints
  expect_equal(length(result$changepoints), 0)
})

test_that("methods handle ts objects", {
  set.seed(123)
  data <- ts(generate_test_data(), frequency = 12)
  
  result <- pelt(data)
  
  expect_type(result, "list")
})

