% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pxlib_open_file.R
\name{pxlib_open_file}
\alias{pxlib_open_file}
\title{Open a Paradox Database File}
\usage{
pxlib_open_file(path, encoding = NULL, password = NULL)
}
\arguments{
\item{path}{A character string specifying the path to the Paradox (.db) file.}

\item{encoding}{An optional character string specifying the source encoding
(e.g., "cp866", "cp1252"). If \code{NULL} (default), encoding is determined
from the file header.}

\item{password}{An optional character string specifying the password for
encrypted files. If the file is encrypted and no password is provided,
an error will be thrown. Default is \code{NULL}.}
}
\value{
An external pointer of class \code{"pxdoc_t"} representing the opened
Paradox file, or \code{NULL} if the file could not be opened (with a warning).
}
\description{
Opens a Paradox database (.db) file and prepares it for reading. This function
serves as the entry point for interacting with a Paradox database.
}
\details{
This function initializes a connection to a Paradox file via the underlying C library.
It automatically performs two key setup tasks:
\enumerate{
\item \strong{Encoding Override:} It allows the user to specify the character encoding of the
source file via the \code{encoding} parameter. This is crucial for legacy files
where the encoding stored in the header may be incorrect. If \code{encoding} is
\code{NULL}, the function will attempt to use the codepage from the file header.
\item \strong{BLOB File Attachment:} It automatically searches for an associated BLOB file
(with a \code{.mb} extension, case-insensitively) in the same directory and,
if found, attaches it to the database handle.
}
\subsection{Encryption Handling}{

This function automatically handles encrypted Paradox files:
\enumerate{
\item If the file is \strong{not encrypted}, the \code{password} parameter is ignored
\item If the file \strong{is encrypted} and \code{password} is provided, it validates the password
\item If the file \strong{is encrypted} and \code{password} is \code{NULL}, an error is thrown
\item If the provided password is \strong{incorrect}, an error is thrown
}

When a file is successfully opened with the correct password, all subsequent
operations (like \code{pxlib_get_data()}) will automatically decrypt the data.
}

\subsection{Resource Management}{

It's important to always close the file handle using \code{pxlib_close_file()}
when you're done to prevent resource leaks. Using \code{on.exit()} is recommended:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{px_doc <- pxlib_open_file("myfile.db", password = "secret")
on.exit(pxlib_close_file(px_doc), add = TRUE)
# ... work with the file ...
}\if{html}{\out{</div>}}
}
}
\examples{
# Example 1: Open a bundled demo file (biolife.db)
db_path <- system.file("extdata", "biolife.db", package = "Rparadox")
pxdoc <- pxlib_open_file(db_path)
if (!is.null(pxdoc)) {
  # normally you'd read data here
  pxlib_close_file(pxdoc)
}

# Example 2: Open a file with overridden encoding (of_cp866.db)
db_path2 <- system.file("extdata", "of_cp866.db", package = "Rparadox")
pxdoc2 <- pxlib_open_file(db_path2, encoding = "cp866")
if (!is.null(pxdoc2)) {
  # read some data ...
  pxlib_close_file(pxdoc2)
}

# Example 3: Open an encrypted file with password
db_path3 <- system.file("extdata", "country_encrypted.db", package = "Rparadox")
px_doc <- pxlib_open_file(db_path3, password = "rparadox")
data <- pxlib_get_data(px_doc)
pxlib_close_file(px_doc)

}
