##########################################################################################################################################

test_that("Bounds are equal to 0, 1 or -1 when there is no selection bias or
          observed effect.", {

  # Bound for the relative risk in the total population should equal 1.
  expect_equal(as.numeric(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                                  pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 1,
                                  RR_SU_11 = 1, RR_SU_00 = 1,
                                  RR_SU_10 = 1, RR_SU_01 = 1)[1, 2]), 1)

  # Bound for the risk difference in the total population should equal -1.
  expect_equal(as.numeric(SVbound(whichEst = "RD_tot", pY1_T1_S1 = 0.5,
                                  pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 1,
                                  RR_SU_11 = 1, RR_SU_00 = 1,
                                  RR_SU_10 = 1, RR_SU_01 = 1)[1, 2]), -1)


  # Bound for the relative risk in the subpopulation should equal 1.
  expect_equal(as.numeric(SVbound(whichEst = "RR_sub", pY1_T1_S1 = 0.5,
                                  pY1_T0_S1 = 0.5, RR_UY_S1 = 1, RR_TU_1 = 1,
                                  RR_TU_0 = 1)[1, 2]), 1)

  # Bound for the risk difference in the subpopulation should equal 0.
  expect_equal(as.numeric(SVbound(whichEst = "RD_sub", pY1_T1_S1 = 0.5,
                                  pY1_T0_S1 = 0.5, RR_UY_S1 = 1, RR_TU_1 = 1,
                                  RR_TU_0 = 1)[1, 2]), 0)
})

##########################################################################################################################################


test_that("SVbound throws an error if the estimand is not correctly specified.", {

  expect_error(SVbound(whichEst = "RR_t", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 1,
                       RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1), 'The estimand must be')

})

##########################################################################################################################################


test_that("SVbound throws an error if the wrong sensitivity parameter is equal to NULL.", {

  #Sensitivity parameters missing.
  expect_error(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = NULL, RR_UY_T0 = 1,
                       RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "When the total population is of interest, RR_UY_T1")
  expect_error(SVbound(whichEst = "RD_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = NULL,
                       RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "When the total population is of interest, RR_UY_T1")
  expect_error(SVbound(whichEst = "RR_sub", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_S1 = 1, RR_TU_1 = NULL, 
                       RR_TU_0 = NULL),
               "When the subpopulation is of interest, RR_UY_S1")
  expect_error(SVbound(whichEst = "RD_sub", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_S1 = NULL, RR_TU_1 = 1,
                       RR_TU_0 = 1),
               "When the subpopulation is of interest, RR_UY_S1")


})

##########################################################################################################################################

test_that("SVbound throws an error if the input takes on incorrect values.", {

  # Sensitivity parameters smaller than 1.
  expect_error(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 0.5, RR_UY_T0 = 1,
                       RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "All sensitivity parameters must be greater")
  expect_error(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 0.5,
                       RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "All sensitivity parameters must be greater")
  expect_error(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 1,
                       RR_SU_11 = 0.5, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "All sensitivity parameters must be greater")
  expect_error(SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_T1 = 1, RR_UY_T0 = 1,
                       RR_SU_11 = 1, RR_SU_00 = 0.5, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "All sensitivity parameters must be greater")
  expect_error(SVbound(whichEst = "RR_sub", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_S1 = 0.5, RR_TU_1 = 1,
                       RR_TU_0 = 1),
               "All sensitivity parameters must be greater")
  expect_error(SVbound(whichEst = "RR_sub", pY1_T1_S1 = 0.5,
                       pY1_T0_S1 = 0.5, RR_UY_S1 = 1, RR_TU_1 = 0.5,
                       RR_TU_0 = 1),
               "All sensitivity parameters must be greater")


  # Probabilities not valid.
  expect_error(SVbound(whichEst = "RD_tot", pY1_T1_S1 = 0.5, pY1_T0_S1 = -1,
                       RR_UY_T1 = 1, RR_UY_T0 = 1, RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "cannot be smaller than 0 or larger than 1")

  expect_error(SVbound(whichEst = "RD_tot", pY1_T1_S1 = 0.5, pY1_T0_S1 = 2,
                       RR_UY_T1 = 1, RR_UY_T0 = 1, RR_SU_11 = 1, RR_SU_00 = 1, 
                       RR_SU_10 = 1, RR_SU_01 = 1),
               "cannot be smaller than 0 or larger than 1")

  expect_error(SVbound(whichEst = "RD_sub", pY1_T1_S1 = -1, pY1_T0_S1 = 0.5,
                       RR_UY_S1 = 1, RR_TU_1 = 1, RR_TU_0 = 1),
               "cannot be smaller than 0 or larger than 1")

  expect_error(SVbound(whichEst = "RD_sub", pY1_T1_S1 = 2, pY1_T0_S1 = 0.5,
                       RR_UY_S1 = 1, RR_TU_1 = 1, RR_TU_0 = 1),
               "cannot be smaller than 0 or larger than 1")

})

##########################################################################################################################################

test_that("SVbound() works with both direct and sensitivityparametersM() input", {
  
  # Example model setup
  V <- matrix(c(1, 0, 0.85, 0.15), ncol = 2)
  U <- matrix(c(1, 0, 0.5, 0.5), ncol = 2)
  Tr <- c(-6.2, 1.75)
  Y <- c(-5.2, 5.0, -1.0)
  S <- matrix(c(1.2, 2.2, 0.0, 0.5, 2.0, -2.75, -4.0, 0.0), ncol = 4)
  probT1 <- 0.286
  probT0 <- 0.004
  
  # Run sensitivity parameter computation for subpopulation.
  sensval <- sensitivityparametersM(
    whichEst = "RR_sub", whichBound = "SV",
    Vval = V, Uval = U, Tcoef = Tr, Ycoef = Y, Scoef = S,
    Mmodel = "L", pY1_T1_S1 = probT1, pY1_T0_S1 = probT0
  )
  
  # Ensure the object has the expected class
  expect_s3_class(sensval, "sensparams")
  
  # Convert to list for reference comparison
  senslist <- as.list(sensval)
  
  # Compute SV bound using manual parameter input for the subpopulation.
  res_manual <- SVbound(
    whichEst = "RR_sub",
    pY1_T1_S1 = probT1, pY1_T0_S1 = probT0,
    RR_UY_S1 = senslist$value[3],
    RR_TU_1  = senslist$value[4],
    RR_TU_0  = senslist$value[5]
  )
  
  # Compute SV bound directly from sensparams object for the subpopulation
  res_auto <- SVbound(
    sens = sensval,
    whichEst = "RR_sub",
    pY1_T1_S1 = probT1,
    pY1_T0_S1 = probT0
  )
  
  # Compare the two outputs
  expect_equal(res_auto, res_manual, tolerance = 1e-10)
  
  
  # Run sensitivity parameter computation for total population.
  sensvalTot <- sensitivityparametersM(
    whichEst = "RR_tot", whichBound = "SV",
    Vval = V, Uval = U, Tcoef = Tr, Ycoef = Y, Scoef = S,
    Mmodel = "L", pY1_T1_S1 = probT1, pY1_T0_S1 = probT0
  )
  
  # Ensure the object has the expected class
  expect_s3_class(sensvalTot, "sensparams")
  
  # Convert to list for reference comparison
  senslistTot <- as.list(sensvalTot)
  
  # Compute SV bound using manual parameter input for the total population.
  res_manualTot <- SVbound(
    whichEst = "RR_tot",
    pY1_T1_S1 = probT1, pY1_T0_S1 = probT0,
    RR_UY_T1 = senslistTot$value[5], RR_UY_T0 = senslistTot$value[6],
    RR_SU_11 = senslistTot$value[7], RR_SU_00 = senslistTot$value[8],
    RR_SU_10 = senslistTot$value[9], RR_SU_01 = senslistTot$value[10]
  )
  
  # Compute SV bound directly from sensparams object for the total population
  res_autoTot <- SVbound(
    sens = sensvalTot,
    whichEst = "RR_tot",
    pY1_T1_S1 = probT1,
    pY1_T0_S1 = probT0
  )
  
  # Compare the two outputs
  expect_equal(res_autoTot, res_manualTot, tolerance = 1e-10)
  
  
})


