% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_xxtpt_to_hours.R
\name{convert_xxtpt_to_hours}
\alias{convert_xxtpt_to_hours}
\title{Convert \code{XXTPT} Strings to Hours}
\usage{
convert_xxtpt_to_hours(
  xxtpt,
  treatment_duration = 0,
  range_method = "midpoint"
)
}
\arguments{
\item{xxtpt}{A character vector of timepoint descriptions from SDTM
\code{--TPT} variables (e.g., \code{PCTPT}, \code{VSTPT}, \code{EGTPT}, \code{ISTPT}, \code{LBTPT}).
Can contain \code{NA} values.

\describe{
\item{Permitted values}{character vector}
\item{Default value}{none}
}}

\item{treatment_duration}{Numeric value(s) specifying the duration of treatment
in hours. Used to convert "EOI/EOT" (End of Infusion/Treatment) patterns
and patterns describing time after end of treatment. Must be non-negative.
Can be either:
\itemize{
\item A single value (used for all timepoints), or
\item A vector of the same length as \code{xxtpt} (one value per timepoint)
}

Default is 0 hours (for instantaneous treatments like oral medications).

\describe{
\item{Permitted values}{numeric scalar or vector (non-negative)}
\item{Default value}{\code{0}}
}}

\item{range_method}{Method for converting time ranges to single values.
Options are "midpoint" (default), "start", or "end". For example,
"0-6h" with midpoint returns 3, with start returns 0, with end returns 6.

\describe{
\item{Permitted values}{character scalar ("midpoint", "start", or "end")}
\item{Default value}{\code{"midpoint"}}
}}
}
\value{
A numeric vector of timepoints in hours. Returns \code{NA_real_} for:
\itemize{
\item Input \code{NA} values
\item Unrecognized timepoint formats
\item Non-time descriptors (e.g., "Morning", "Evening")
}

Returns \code{numeric(0)} for empty input.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Converts CDISC timepoint strings (e.g., \code{PCTPT}, \code{VSTPT}, \code{EGTPT},
\code{ISTPT}, \code{LBTPT}) into numeric hours for analysis. The function handles
common dose-centric formats including pre-dose, post-dose (hours/minutes),
days, time ranges, and treatment-related time markers.
}
\details{
The function recognizes the following patterns (all case-insensitive):

\strong{Special Cases:}
\itemize{
\item \code{"Screening"} -> 0
\item \code{"Pre-dose"}, \code{"Predose"}, \code{"Pre-treatment"}, \code{"Pre-infusion"},
\code{"Pre-inf"}, \code{"Before"}, \code{"Infusion"}, \code{"0H"} -> 0
\item \code{"EOI"}, \code{"EOT"}, \code{"End of Infusion"}, \code{"End of Treatment"},
\code{"After End of Infusion"}, \code{"After End of Treatment"} -> \code{treatment_duration}
(default: 0)
\item \code{"Morning"}, \code{"Evening"} -> \code{NA_real_}
\item Unrecognized values -> \code{NA_real_}
}

\strong{Time Ranges:}
Time ranges are converted based on the \code{range_method} parameter:
\itemize{
\item \code{"0-6h Post-dose"} with \code{range_method = "midpoint"} (default) -> 3
\item \code{"0-6h Post-dose"} with \code{range_method = "start"} -> 0
\item \code{"0-6h Post-dose"} with \code{range_method = "end"} -> 6
\item \code{"0-4H PRIOR START OF INFUSION"} with midpoint -> -2 (negative for prior)
\item \code{"8-16H POST START OF INFUSION"} with midpoint -> 12
\item \code{"0-4H AFTER EOI"} with midpoint and treatment_duration=1 -> 3 (1 + 2)
\item \code{"0-4H EOT"} with midpoint and treatment_duration=0 -> 2
\item \code{"4-8H AFTER END OF INFUSION"} with midpoint and treatment_duration=1 -> 7 (1 + 6)
\item \code{"4-8H POST INFUSION"} with midpoint and treatment_duration=1 -> 7 (1 + 6)
\item \code{"4-8H POST-INF"} with midpoint and treatment_duration=1 -> 7 (1 + 6)
}

\strong{Time-based Conversions:}
\itemize{
\item \strong{Days}: \code{"Day 1"} -> 24, \code{"2D"} -> 48, \code{"30 DAYS AFTER LAST"} -> 720
(requires unit indicator; bare numbers like \code{"2"} return \code{NA})
\item \strong{Hours + Minutes}: \code{"1H30M"} -> 1.5
\item \strong{Hours}: \code{"2 hours"} -> 2, \code{"1 HOUR POST"} -> 1
\item \strong{Minutes}: \code{"30M"} -> 0.5, \code{"30 MIN POST"} -> 0.5
\item \strong{Predose}: \code{"5 MIN PREDOSE"} -> -0.0833, \code{"5 MIN PRE-DOSE"} -> -0.0833
\item \strong{Before treatment}: \code{"5 MIN BEFORE"} -> -0.0833
\item \strong{Post EOI/EOT}: \code{"1 HOUR POST EOI"} -> treatment_duration + 1,
\code{"24 HR POST INF"} -> treatment_duration + 24,
\code{"24 HR POST-INF"} -> treatment_duration + 24,
\code{"1 HOUR AFTER EOT"} -> treatment_duration + 1
\item \strong{After end}: \code{"30MIN AFTER END OF INFUSION"} -> treatment_duration + 0.5
\item \strong{Start of infusion/treatment}: \code{"8H PRIOR START OF INFUSION"} -> -8,
\code{"8H BEFORE START OF TREATMENT"} -> -8
\item \strong{Pre EOI/EOT}: \code{"10MIN PRE EOI"} -> treatment_duration - 1/6,
\code{"10MIN BEFORE EOT"} -> treatment_duration - 1/6
}

\strong{Supported Unit Formats:}
\itemize{
\item Hours: H, h, HR, hr, HOUR, hour (with optional plurals)
\item Minutes: M, m, MIN, min, MINUTE, minute (with optional plurals)
\item Days: D, d, DAY, day (with optional plurals)
\item Flexible whitespace and optional "Post-dose", "POST", "After last"
suffixes
\item Hyphens in compound terms: "PRE-DOSE", "POST-INF", "POST-INFUSION"
}

\strong{Understanding POST/AFTER Patterns:}

It's important to distinguish between patterns relative to treatment \strong{start}
versus treatment \strong{end}:
\itemize{
\item \strong{Relative to START} (treatment_duration NOT added):
\itemize{
\item \code{"1H POST"}, \code{"1H AFTER"}, \code{"30M POST"} -> Time from dose/treatment start
\item These patterns assume treatment starts at time 0
\item Example: \code{"1H POST"} -> 1 hour (regardless of treatment_duration)
}
\item \strong{Relative to END} (treatment_duration IS added):
\itemize{
\item \code{"1H POST EOI"}, \code{"1H AFTER EOT"}, \code{"1H POST INFUSION"} -> Time from
treatment end
\item These patterns account for when treatment ends (start + duration)
\item Example: \code{"1H POST EOI"} with treatment_duration=2 -> 3 hours (2 + 1)
}
}

This distinction follows standard pharmacokinetic conventions where "post-dose"
refers to time from treatment initiation, while "post end of infusion" refers
to time from treatment completion.

\strong{Vectorized Treatment Duration:}

When \code{treatment_duration} is a vector, each timepoint uses its corresponding
treatment duration value. This is useful when different records have different
treatment durations (e.g., different infusion lengths).
}
\seealso{
Date/Time Computation Functions that returns a vector:
\code{\link{compute_age_years}()},
\code{\link{compute_dtf}()},
\code{\link{compute_duration}()},
\code{\link{compute_tmf}()},
\code{\link{convert_date_to_dtm}()},
\code{\link{convert_dtc_to_dt}()},
\code{\link{convert_dtc_to_dtm}()},
\code{\link{impute_dtc_dt}()},
\code{\link{impute_dtc_dtm}()}
}
\concept{com_date_time}
\keyword{com_date_time}
\keyword{experimental}
\section{Examples}{
\subsection{Basic timepoint patterns}{

Convert basic dose-centric patterns to hours

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(c(
  "Screening",
  "Pre-dose",
  "Pre-treatment",
  "Before",
  "30M",
  "1H",
  "2H POSTDOSE",
  "Day 1"
))
#> [1]  0.0  0.0  0.0  0.0  0.5  1.0  2.0 24.0}\if{html}{\out{</div>}}}
\subsection{Predose and before patterns}{

Convert predose/before patterns that return negative times

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(c("5 MIN PREDOSE", "5 MIN PRE-DOSE", "1 HOUR BEFORE"))
#> [1] -0.08333333 -0.08333333 -1.00000000}\if{html}{\out{</div>}}}
\subsection{Treatment-related patterns (oral medications)}{

With default treatment_duration = 0 for oral medications

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(c(
  "EOT",
  "1 HOUR POST EOT",
  "1 HOUR AFTER EOT",
  "After End of Treatment"
))
#> [1] 0 1 1 0}\if{html}{\out{</div>}}}
\subsection{Infusion-related patterns}{

With treatment_duration = 1 hour for IV infusions

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(
  c(
    "EOI",
    "1 HOUR POST EOI",
    "24 HR POST INF",
    "24 HR POST-INF",
    "30MIN AFTER END OF INFUSION",
    "8H PRIOR START OF INFUSION",
    "10MIN PRE EOI"
  ),
  treatment_duration = 1
)
#> [1]  1.0000000  2.0000000 25.0000000 25.0000000  1.5000000 -8.0000000  0.8333333}\if{html}{\out{</div>}}}
\subsection{Vectorized treatment duration}{

Different treatment durations per timepoint

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(
  c("EOI", "1 HOUR POST EOI", "EOI", "1 HOUR POST EOI"),
  treatment_duration = c(1, 1, 2, 2)
)
#> [1] 1 2 2 3}\if{html}{\out{</div>}}}
\subsection{Time ranges with midpoint method}{

Default midpoint method for ranges

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(c(
  "0-6h Post-dose",
  "0-4H PRIOR START OF INFUSION",
  "8-16H POST START OF INFUSION"
))
#> [1]  3 -2 12}\if{html}{\out{</div>}}}
\subsection{Time ranges with custom methods}{

Specify start or end method for ranges

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours("0-6h Post-dose", range_method = "end")
#> [1] 6
convert_xxtpt_to_hours("0-6h Post-dose", range_method = "start")
#> [1] 0}\if{html}{\out{</div>}}}
\subsection{Ranges relative to EOI/EOT}{

Time ranges after end of infusion/treatment

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(
  c(
    "0-4H AFTER EOI",
    "0-4H POST EOI",
    "4-8H AFTER END OF INFUSION",
    "4-8H AFTER EOT",
    "4-8H POST INFUSION",
    "4-8H POST-INF"
  ),
  treatment_duration = 1
)
#> [1] 3 3 7 7 7 7}\if{html}{\out{</div>}}}
\subsection{POST vs POST EOI distinction}{

Difference between POST (from start) and POST EOI (from end)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{convert_xxtpt_to_hours(
  c("Pre-dose", "1H POST", "2H POST", "4H POST"),
  treatment_duration = 2
)
#> [1] 0 1 2 4

convert_xxtpt_to_hours(
  c("Pre-dose", "EOI", "1H POST EOI", "2H POST EOI"),
  treatment_duration = 2
)
#> [1] 0 2 3 4

convert_xxtpt_to_hours(
  c("1H POST", "1H POST EOI", "1H POST INFUSION"),
  treatment_duration = 2
)
#> [1] 1 3 3}\if{html}{\out{</div>}}}}

