# Prior simulation for appraise

#' Simulate bias priors (xi)
#'
#' Generates Monte Carlo samples from the prior distributions
#' specified for each bias type.
#'
#' @param bias_spec Output from build_bias_specification()
#' @param n_draws Number of Monte Carlo draws
#'
#' @return A numeric matrix of dimension n_draws x NN
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{build_bias_specification}}
#'   \item \code{\link{run_appraise_model}}
#'   \item \code{vignette("appraise-introduction")}
#' }
#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272.
#'
#' @examples
#' ## Simulate prior draws for two biases
#'
#' bias_spec <- build_bias_specification(
#'   num_biases = 2,
#'   b_types = "Confounding",
#'   e_types = "Measurement Errors",
#'   ab_params = list(
#'     Confounding = c(2, 5)
#'   ),
#'   ex_params = list(
#'     `Measurement Errors` = 1.2
#'   )
#' )
#'
#' xi <- simulate_bias_priors(bias_spec, n_draws = 1000)
#'
#' ## Dimensions correspond to (draws × biases)
#' dim(xi)
#'
#' ## Inspect prior distribution for first bias
#' hist(xi[, 1], main = "Prior for Confounding Bias", xlab = "Bias magnitude")
#'
#' @export
simulate_bias_priors <- function(bias_spec, n_draws = 5000) {

  if (!is.list(bias_spec) || is.null(bias_spec$NN)) {
    stop("bias_spec must be output of build_bias_specification()", call. = FALSE)
  }

  NN <- bias_spec$NN

  xi <- matrix(NA_real_, nrow = n_draws, ncol = NN)

  for (i in seq_len(NN)) {

    if (bias_spec$b[i] != 999) {
      xi[, i] <- stats::rbeta(
        n_draws,
        bias_spec$ab_values[2 * i - 1],
        bias_spec$ab_values[2 * i]
      )

    } else if (bias_spec$s[i] != 999) {
      if (!requireNamespace("sn", quietly = TRUE)) {
        stop("Package 'sn' is required for skew-normal priors", call. = FALSE)
      }
      xi[, i] <- sn::rsn(
        n = n_draws,
        xi = bias_spec$skn_values[3 * i - 2],
        omega = bias_spec$skn_values[3 * i - 1],
        alpha = bias_spec$skn_values[3 * i]
      )

    } else if (bias_spec$d[i] != 999) {
      if (!requireNamespace("VGAM", quietly = TRUE)) {
        stop("Package 'VGAM' is required for Laplace priors", call. = FALSE)
      }
      xi[, i] <- VGAM::rlaplace(
        n_draws,
        bias_spec$de_values[2 * i - 1],
        bias_spec$de_values[2 * i]
      )

    } else if (bias_spec$e[i] != 999) {
      xi[, i] <- stats::rexp(
        n_draws,
        bias_spec$ex_values[i]
      )

    } else if (bias_spec$en[i] != 999) {
      xi[, i] <- -stats::rexp(
        n_draws,
        bias_spec$exneg_values[i]
      )
    }
  }

  colnames(xi) <- paste0("bias_", seq_len(NN))
  xi
}
