% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynamic.R
\name{blvim}
\alias{blvim}
\title{Compute an equilibrium solution of the Boltzmann-Lotka-Volterra model}
\usage{
blvim(
  costs,
  X,
  alpha,
  beta,
  Z,
  bipartite = TRUE,
  origin_data = NULL,
  destination_data = NULL,
  epsilon = 0.01,
  iter_max = 50000,
  conv_check = 100,
  precision = 1e-06,
  quadratic = FALSE
)
}
\arguments{
\item{costs}{a cost matrix}

\item{X}{a vector of production constraints}

\item{alpha}{the return to scale parameter}

\item{beta}{the inverse cost scale parameter}

\item{Z}{a vector of initial destination attractivenesses}

\item{bipartite}{when \code{TRUE} (default value), the origin and destination
locations are considered to be distinct. When \code{FALSE}, a single set of
locations plays the both roles. This has only consequences in functions
specific to this latter case such as \code{\link[=terminals]{terminals()}}.}

\item{origin_data}{\code{NULL} or a list of additional data about the origin
locations (see details)}

\item{destination_data}{\code{NULL} or a list of additional data about the
destination locations (see details)}

\item{epsilon}{the update intensity}

\item{iter_max}{the maximal number of steps of the BLV dynamic}

\item{conv_check}{number of iterations between to convergence test}

\item{precision}{convergence threshold}

\item{quadratic}{selects the update rule, see details.}
}
\value{
an object of class \code{sim}(and \code{sim_blvim}) for spatial interaction
model that contains the matrix of flows between the origin and the
destination locations as well as the final attractivenesses computed by the
model.
}
\description{
This function computes flows between origin locations and destination
locations at an equilibrium solution of A. Wilson's Boltzmann-Lotka-Volterra
(BLV) interaction model. The BLV dynamic model is initialised with the
production constraints at the origin locations and the initial values of the
the attractiveness of destination locations. Iterations update the
attractivenesses according the received flows.
}
\details{
In a step of the BLV model, flows are computed according to the production
constrained entropy maximising model proposed by A. Wilson and implemented in
\code{\link[=static_blvim]{static_blvim()}}. Then the flows received at a destination are computed as
follows

\deqn{\forall j,\quad D_j=\sum_{i=1}^{n}Y_{ij},}

for destination \eqn{j}. This enables updating the attractivenesses by making
them closer to the received flows, i.e. trying to reduce \eqn{|D_j-Z_j|}.

A. Wilson and co-authors proposed two different update strategies:
\enumerate{
\item The original model proposed in Harris & Wilson (1978) updates the
\eqn{Z_j} as follows \deqn{Z_j^{t+1} = Z_j^{t} + \epsilon (D^{t}_j-Z^{t}_j)}
\item In Wilson (2008), the update is given by \deqn{Z_j^{t+1} = Z_j^{t} +
\epsilon (D^{t}_j-Z^{t}_j)Z^{t}_j}
}

In both cases, \eqn{\epsilon} is given by the \code{epsilon} parameter. It should
be smaller than 1. The first update is used when the \code{quadratic} parameter is
\code{FALSE} which is the default value. The second update is used when
\code{quadratic} is \code{TRUE}.

Updates are performed until convergence or for a maximum of \code{iter_max}
iterations. Convergence is checked every \code{conv_check} iterations. The
algorithm is considered to have converged if \deqn{\|Z^{t+1}-Z^t\|<\delta
(\|Z^{t+1}\|+\delta),} where \eqn{\delta} is given by the \code{precision}
parameter.
}
\section{Location data}{
While models in this package do not use location data beyond \code{X} and \code{Z},
additional data can be stored and used when analysing spatial interaction
models.
\subsection{Origin and destination location names}{

Spatial interaction models can store names for origin and destination
locations, using \code{\link[=origin_names<-]{origin_names<-()}} and \code{\link[=destination_names<-]{destination_names<-()}}. Names
are taken by default from names of the cost matrix \code{costs}. More precisely,
\code{rownames(costs)} is used for origin location names and \code{colnames(costs)} for
destination location names.
}

\subsection{Origin and destination location positions}{

Spatial interaction models can store the positions of the origin and
destination locations, using \code{\link[=origin_positions<-]{origin_positions<-()}} and
\code{\link[=destination_positions<-]{destination_positions<-()}}.
}

\subsection{Specifying location data}{

In addition to the functions mentioned above, location data can be specified
directly using the \code{origin_data} and \code{destination_data} parameters. Data are
given by a list whose components are not interpreted excepted the following
ones:
\itemize{
\item \code{names} is used to specify location names and its content has to follow
the restrictions documented in \code{\link[=origin_names<-]{origin_names<-()}} and
\code{\link[=destination_names<-]{destination_names<-()}}
\item \code{positions} is used to specify location positions and its content has
to follow the restrictions documented in \code{\link[=origin_positions<-]{origin_positions<-()}} and
\code{\link[=destination_positions<-]{destination_positions<-()}}
}
}
}

\examples{
distances <- french_cities_distances[1:10, 1:10] / 1000 ## convert to km
production <- rep(1, 10)
attractiveness <- log(french_cities$area[1:10])
## rescale to production
attractiveness <- attractiveness / sum(attractiveness) * sum(production)
flows <- blvim(distances, production, 1.5, 1 / 250, attractiveness)
flows
}
\references{
Harris, B., & Wilson, A. G. (1978). "Equilibrium Values and
Dynamics of Attractiveness Terms in Production-Constrained
Spatial-Interaction Models", Environment and Planning A: Economy and Space,
10(4), 371-388. \doi{10.1068/a100371}

Wilson, A. (2008), "Boltzmann, Lotka and Volterra and spatial structural
evolution: an integrated methodology for some dynamical systems", J. R.
Soc. Interface.5865-871 \doi{10.1098/rsif.2007.1288}
}
\seealso{
\code{\link[=grid_blvim]{grid_blvim()}} for systematic exploration of parameter influence,
\code{\link[=static_blvim]{static_blvim()}} for the static model.
}
