
#' @export
generics::forecast

#' @title Forecasting using Hierarchical Panel Vector Autoregressions
#'
#' @description Samples from the joint predictive density of the dependent 
#' variables for all countries at forecast horizons 
#' from 1 to \code{horizon} specified as an argument of the function. 
#' Also implements conditional forecasting based on the provided projections
#' for some of the variables.
#' 
#' @details 
#' The package provides a range of options regarding the forecasting procedure.
#' They are dependent on the model and forecast specifications and include 
#' Bayesian forecasting many periods ahead, conditional forecasting, and 
#' forecasting for models with exogenous variables.
#' 
#' \strong{One-period-ahead predictive density.}
#' The model assumptions provided in the documentation for \code{\link{bpvars}} 
#' determine the country-specific one-period ahead conditional predictive density 
#' for the unknown vector \eqn{\mathbf{y}_{c.t+1}} given the data available at 
#' time \eqn{t} and the parameters of the model. It is multivariate normal with
#' the mean \eqn{\mathbf{A}_c' \mathbf{x}_{c.t+1}} and the covariance matrix 
#' \eqn{\mathbf{\Sigma}_c}
#' \deqn{p(\mathbf{y}_{c.t+1} | \mathbf{x}_{c.t+1}, \mathbf{A}_c, \mathbf{\Sigma}_c) = N_N(\mathbf{A}_c' \mathbf{x}_{c.t+1}, \mathbf{\Sigma}_c)}
#' where \eqn{\mathbf{x}_{c.t+1}} includes the lagged
#' values of \eqn{\mathbf{y}_{c.t+1}}, the constant term, and, potentially,
#' exogenous variables if they were specified by the user. 
#' 
#' \strong{Bayesian predictive density.}
#' The one-period ahead predictive density is used to sample from the joint 
#' predictive density of the unknown future values. This predictive density is
#' defined as a joint density of \eqn{\mathbf{y}_{c.t+h}} at horizons 
#' \eqn{h = 1,\dots,H}, where \eqn{H} corresponds to the value of argument 
#' \code{horizon}, given the data available at time \eqn{t}:
#' \deqn{p( \mathbf{y}_{c.T_c + H}, \dots, \mathbf{y}_{c.T_c + 1} | \mathbf{Y}_c, \mathbf{X}_c) = 
#' \int p(\mathbf{y}_{c.T_c + H}, \dots, \mathbf{y}_{c.T_c + 1} | \mathbf{Y}_c, \mathbf{X}_c, \mathbf{A}_c, \boldsymbol\Sigma_c)
#' p( \mathbf{A}_c, \boldsymbol\Sigma_c | \mathbf{Y}_c, \mathbf{X}_c) d(\mathbf{A}_c, \boldsymbol\Sigma_c)}
#' Therefore, the Bayesian forecast does not depend on the parameter values as
#' the parameters are integrated out with respect to their posterior distribution.
#' Consequently, Bayesian forecasts incorporate the uncertainty with respect to
#' estimation. Sampling from the density is facilitated using the draws from the 
#' posterior density and sequential sampling from the one-period ahead 
#' predictive density.
#' 
#' \strong{Conditional forecasting} of some of the variables given the future 
#' values of the remaining variables is implemented following 
#' Waggoner and Zha (1999) and is based on the conditional normal density given
#' the future projections of some of the variables created basing on the 
#' one-period ahead predictive density.
#' 
#' \strong{Exogenous variables.}
#' Forecasting with models for which specification argument 
#' \code{exogenous_variables} was specified required providing the future values
#' of these exogenous variables in the argument \code{exogenous_forecast} of the
#' \code{\link{forecast.PosteriorBVARPANEL}} function.
#' 
#' \strong{Truncated forecasts for variables of type 'rate'.}
#' The package provides the option to truncate the forecasts for variables of 
#' for which the corresponding element of argument \code{type} of the function 
#' \code{specify_bvarPANEL$new()} is set to \code{"rate"}. The one-period-ahead
#' predictive normal density for such variables is truncated to values from 
#' interval  \eqn{[0,100]}. 
#' 
#' @method forecast PosteriorBVARPANEL
#' 
#' @param object posterior estimation outcome - an object of class 
#' \code{PosteriorBVARPANEL} obtained by running the \code{estimate} function.
#' @param horizon a positive integer, specifying the forecasting horizon.
#' @param exogenous_forecast not used here ATM; included for compatibility with 
#' generic \code{forecast}.
#' @param conditional_forecast a list of length \code{C} containing 
#' \code{horizon x N} matrices with forecasted values for selected variables. 
#' These matrices should only contain \code{numeric} or \code{NA} values. The 
#' entries with \code{NA} values correspond to the values that are forecasted 
#' conditionally on the realisations provided as \code{numeric} values.
#' @param ... not used
#' 
#' @return A list of class \code{ForecastsPANEL} with \code{C} elements containing 
#' the draws from the country-specific predictive density and data in a form of 
#' object class \code{Forecasts} that includes:
#' 
#' \describe{
#'  \item{forecasts}{an \code{NxhorizonxS} array with the draws from the 
#'  country-specific predictive density}
#'  \item{forecast_mean}{an \code{NxhorizonxS} array with the mean of the 
#'  country-specific predictive density}
#'  \item{forecast_cov}{an \code{NxNxhorizonxS} array with the covariance of the 
#'  country-specific predictive density}
#'  \item{Y}{a \code{T_cxN} matrix with the country-specific data}
#' }
#'
#' @references
#' Waggoner, D. F., & Zha, T. (1999) 
#' Conditional forecasts in dynamic multivariate models, 
#' \emph{Review of Economics and Statistics}, \bold{81}(4), 639-651,
#' \doi{10.1162/003465399558508}.
#'
#' @seealso \code{\link{specify_bvarPANEL}}, \code{\link{estimate.PosteriorBVARPANEL}}, 
#' \code{\link{summary.ForecastsPANEL}}, \code{\link{plot.ForecastsPANEL}}
#'
#' @author Tomasz Woźniak \email{wozniak.tom@pm.me}
#' 
#' @examples
#' # specify the model
#' specification = specify_bvarPANEL$new(
#'   ilo_dynamic_panel[1:5], 
#'   exogenous = ilo_exogenous_variables[1:5]
#' )
#' burn_in       = estimate(specification, 5)             # run the burn-in; use say S = 10000
#' posterior     = estimate(burn_in, 5)                   # estimate the model; use say S = 10000
#' 
#' # forecast 5 years ahead
#' predictive    = forecast(posterior, 5, exogenous_forecast = ilo_exogenous_forecasts[1:5])
#' 
#' @export
forecast.PosteriorBVARPANEL = function(
    object, 
    horizon = 1, 
    exogenous_forecast = NULL,
    conditional_forecast = NULL,
    ...
) {
  
  stopifnot(
    "Argument horizon must be a positive integer number." = 
      horizon > 0 & horizon %% 1 == 0
  )
  
  posterior_A_c_cpp     = object$posterior$A_c_cpp
  posterior_Sigma_c_cpp = object$posterior$Sigma_c_cpp
  Y_c             = object$posterior$Y
  N               = dim(posterior_A_c_cpp[1,1][[1]])[2]
  K               = dim(posterior_A_c_cpp[1,1][[1]])[1]
  C               = dim(posterior_A_c_cpp[1,1][[1]])[3]
  S               = dim(posterior_A_c_cpp)[1]
  p               = object$last_draw$p
  c_names         = names(object$last_draw$data_matrices$Y)
  
  d               = K - N * p - 1
  if (d == 0 ) {
    # this will not be used for forecasting, but needs to be provided
    exogenous_forecast = list()
    for (c in 1:C) exogenous_forecast[[c]] = matrix(NA, horizon, 1)
  } else {
    stopifnot("Forecasted values of exogenous variables are missing." 
              = (d > 0) & !is.null(exogenous_forecast))
    stopifnot("The matrix of exogenous_forecast does not have a correct number of columns." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){ncol(x)}))) == d)
    stopifnot("Provide exogenous_forecast for all forecast periods specified by argument horizon." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){nrow(x)}))) == horizon)
    stopifnot("Argument exogenous has to be a matrix." 
              = all(simplify2array(lapply(exogenous_forecast, function(x){is.matrix(x) & is.numeric(x)}))))
    stopifnot("Argument exogenous cannot include missing values." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){any(is.na(x))}))) == FALSE)
  }
  
  if ( is.null(conditional_forecast) ) {
    conditional_forecast = list()
    for (c in 1:C) conditional_forecast[[c]] = matrix(NA, horizon, N)
  } else {
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data." 
              = is.list(conditional_forecast) & length(conditional_forecast) == C
    )
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data."
              = all(c_names == names(conditional_forecast))
    )
    stopifnot("Argument conditional_forecast must be a list with matrices with numeric values."
              = all(sapply(conditional_forecast, function(x) is.matrix(x) & is.numeric(x)))
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of rows equal to the value of argument horizon."
              = unique(sapply(conditional_forecast, function(x) nrow(x) )) == horizon
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of columns equal to the number of columns in the used data."
              = unique(sapply(conditional_forecast, function(x) ncol(x) )) == N
    )
  }
  
  type      = object$last_draw$data_matrices$type
  LB        = rep(-Inf, N)
  UB        = rep(Inf, N)
  rates_id  = which(type == "rate")
  if (length(rates_id) > 0) {
    LB[rates_id] = 0
    UB[rates_id] = 100
  }
  
  # perform forecasting
  fff           = .Call(`_bpvars_forecast_bvarPANEL`, 
                        posterior_A_c_cpp, 
                        posterior_Sigma_c_cpp, 
                        Y_c, 
                        conditional_forecast, 
                        exogenous_forecast, 
                        horizon,
                        LB,
                        UB,
                        TRUE,
                        p
                       )
                          
  forecasts       = list()
  Ymean           = .Call(`_bpvars_mean_field`, Y_c )
  
  for (c in 1:C) {
    fore            = list()
    fore_tmp        = aperm(fff$forecasts_cpp[c,1][[1]], c(2,1,3))
    fore$forecasts  = fore_tmp
    
    fmean_tmp       = aperm(fff$forecast_mean_cpp[c,1][[1]], c(2,1,3))
    fore$forecast_mean = fmean_tmp
    
    cov_tmp        = array(NA, c(N, N, horizon, S))
    for (s in 1:S) {
      cov_tmp[,,,s] = fff$forecast_cov_cpp[c,s][[1]]
    }
    fore$forecast_cov = cov_tmp
    
    fore$Y          = t(Ymean[[c]])
    class(fore)     = "Forecasts"
    forecasts[[c]]  = fore
  }
  names(forecasts)  = c_names
  class(forecasts)  = "ForecastsPANEL"
  
  return(forecasts)
}




#' @inherit forecast.PosteriorBVARPANEL
#' @method forecast PosteriorBVARGROUPPANEL
#' 
#' @param object posterior estimation outcome - an object of class 
#' \code{PosteriorBVARGROUPPANEL} obtained by running the \code{estimate} function.
#' 
#' @seealso \code{\link{specify_bvarGroupPANEL}}, \code{\link{estimate.PosteriorBVARGROUPPANEL}}, 
#' \code{\link{summary.ForecastsPANEL}}, \code{\link{plot.ForecastsPANEL}}
#'
#' @examples
#' # specify the model
#' specification = specify_bvarGroupPANEL$new(
#'                   ilo_dynamic_panel[1:5], 
#'                   exogenous = ilo_exogenous_variables[1:5],
#'                   group_allocation = country_grouping_incomegroup[1:5]
#'                 )
#' burn_in       = estimate(specification, 5)             # run the burn-in; use say S = 10000
#' posterior     = estimate(burn_in, 5)                   # estimate the model; use say S = 10000
#' 
#' # forecast 5 years ahead
#' predictive    = forecast(
#'                   posterior, 
#'                   horizon = 5, 
#'                   exogenous_forecast = ilo_exogenous_forecasts[1:5]
#'                 )
#' 
#' @export
forecast.PosteriorBVARGROUPPANEL = function(
    object, 
    horizon = 1, 
    exogenous_forecast = NULL,
    conditional_forecast = NULL,
    ...
) {
  
  stopifnot(
    "Argument horizon must be a positive integer number." = 
      horizon > 0 & horizon %% 1 == 0
  )
  
  posterior_A_c_cpp     = object$posterior$A_c_cpp
  posterior_Sigma_c_cpp = object$posterior$Sigma_c_cpp
  
  Y_c             = object$posterior$Y
  N               = dim(posterior_A_c_cpp[1,1][[1]])[2]
  K               = dim(posterior_A_c_cpp[1,1][[1]])[1]
  C               = dim(posterior_A_c_cpp[1,1][[1]])[3]
  S               = dim(posterior_A_c_cpp)[1]
  p               = object$last_draw$p
  c_names         = names(object$last_draw$data_matrices$Y)
  
  d               = K - N * p - 1
  if (d == 0 ) {
    # this will not be used for forecasting, but needs to be provided
    exogenous_forecast = list()
    for (c in 1:C) exogenous_forecast[[c]] = matrix(NA, horizon, 1)
  } else {
    stopifnot("Forecasted values of exogenous variables are missing." 
              = (d > 0) & !is.null(exogenous_forecast))
    stopifnot("The matrix of exogenous_forecast does not have a correct number of columns." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){ncol(x)}))) == d)
    stopifnot("Provide exogenous_forecast for all forecast periods specified by argument horizon." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){nrow(x)}))) == horizon)
    stopifnot("Argument exogenous has to be a matrix." 
              = all(simplify2array(lapply(exogenous_forecast, function(x){is.matrix(x) & is.numeric(x)}))))
    stopifnot("Argument exogenous cannot include missing values." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){any(is.na(x))}))) == FALSE)
  }
  
  if ( is.null(conditional_forecast) ) {
    conditional_forecast = list()
    for (c in 1:C) conditional_forecast[[c]] = matrix(NA, horizon, N)
  } else {
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data." 
              = is.list(conditional_forecast) & length(conditional_forecast) == C
    )
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data."
              = all(c_names == names(conditional_forecast))
    )
    stopifnot("Argument conditional_forecast must be a list with matrices with numeric values."
              = all(sapply(conditional_forecast, function(x) is.matrix(x) & is.numeric(x)))
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of rows equal to the value of argument horizon."
              = unique(sapply(conditional_forecast, function(x) nrow(x) )) == horizon
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of columns equal to the number of columns in the used data."
              = unique(sapply(conditional_forecast, function(x) ncol(x) )) == N
    )
  }
  
  type      = object$last_draw$data_matrices$type
  LB        = rep(-Inf, N)
  UB        = rep(Inf, N)
  rates_id  = which(type == "rate")
  if (length(rates_id) > 0) {
    LB[rates_id] = 0
    UB[rates_id] = 100
  }
  
  # perform forecasting
  fff           = .Call(`_bpvars_forecast_bvarPANEL`, 
                        posterior_A_c_cpp, 
                        posterior_Sigma_c_cpp, 
                        Y_c, 
                        conditional_forecast, 
                        exogenous_forecast, 
                        horizon,
                        LB,
                        UB,
                        TRUE,
                        p
  )
  
  forecasts       = list()
  Ymean           = .Call(`_bpvars_mean_field`, Y_c )
  
  for (c in 1:C) {
    fore            = list()
    fore_tmp        = aperm(fff$forecasts_cpp[c,1][[1]], c(2,1,3))
    fore$forecasts  = fore_tmp
    
    fmean_tmp       = aperm(fff$forecast_mean_cpp[c,1][[1]], c(2,1,3))
    fore$forecast_mean = fmean_tmp
    
    cov_tmp        = array(NA, c(N, N, horizon, S))
    for (s in 1:S) {
      cov_tmp[,,,s] = fff$forecast_cov_cpp[c,s][[1]]
    }
    fore$forecast_cov = cov_tmp
    
    fore$Y          = t(Ymean[[c]])
    class(fore)     = "Forecasts"
    forecasts[[c]]  = fore
  }
  names(forecasts)  = c_names
  class(forecasts)  = "ForecastsPANEL"
  
  return(forecasts)
}





#' @title Forecasting using Hierarchical Vector Autoregressions for Dynamic Panel Data
#'
#' @description Samples from the joint predictive density of the dependent 
#' variables for all countries at forecast horizons 
#' from 1 to \code{horizon} specified as an argument of the function. 
#' Also implements conditional forecasting based on the provided projections
#' for some of the variables.
#' 
#' @details 
#' The package provides a range of options regarding the forecasting procedure.
#' They are dependent on the model and forecast specifications and include 
#' Bayesian forecasting many periods ahead, conditional forecasting, and 
#' forecasting for models with exogenous variables.
#' 
#' \strong{One-period-ahead predictive density.}
#' The model assumptions provided in the documentation for \code{\link{bpvars}} 
#' determine the country-specific one-period ahead conditional predictive density 
#' for the unknown vector \eqn{\mathbf{y}_{c.t+1}} given the data available at 
#' time \eqn{t} and the parameters of the model. It is multivariate normal with
#' the mean \eqn{\mathbf{A}_c' \mathbf{x}_{c.t+1}} and the covariance matrix 
#' \eqn{\mathbf{\Sigma}_c}
#' \deqn{p(\mathbf{y}_{c.t+1} | \mathbf{x}_{c.t+1}, \mathbf{A}_c, \mathbf{\Sigma}_c) = N_N(\mathbf{A}_c' \mathbf{x}_{c.t+1}, \mathbf{\Sigma}_c)}
#' where \eqn{\mathbf{x}_{c.t+1}} includes the lagged
#' values of \eqn{\mathbf{y}_{c.t+1}}, the constant term, and, potentially,
#' exogenous variables if they were specified by the user. 
#' 
#' \strong{Bayesian predictive density.}
#' The one-period ahead predictive density is used to sample from the joint 
#' predictive density of the unknown future values. This predictive density is
#' defined as a joint density of \eqn{\mathbf{y}_{c.t+h}} at horizons 
#' \eqn{h = 1,\dots,H}, where \eqn{H} corresponds to the value of argument 
#' \code{horizon}, given the data available at time \eqn{t}:
#' \deqn{p( \mathbf{y}_{c.T_c + H}, \dots, \mathbf{y}_{c.T_c + 1} | \mathbf{Y}_c, \mathbf{X}_c) = 
#' \int p(\mathbf{y}_{c.T_c + H}, \dots, \mathbf{y}_{c.T_c + 1} | \mathbf{Y}_c, \mathbf{X}_c, \mathbf{A}_c, \boldsymbol\Sigma_c)
#' p( \mathbf{A}_c, \boldsymbol\Sigma_c | \mathbf{Y}_c, \mathbf{X}_c) d(\mathbf{A}_c, \boldsymbol\Sigma_c)}
#' Therefore, the Bayesian forecast does not depend on the parameter values as
#' the parameters are integrated out with respect to their posterior distribution.
#' Consequently, Bayesian forecasts incorporate the uncertainty with respect to
#' estimation. Sampling from the density is facilitated using the draws from the 
#' posterior density and sequential sampling from the one-period ahead 
#' predictive density.
#' 
#' \strong{Conditional forecasting} of some of the variables given the future 
#' values of the remaining variables is implemented following 
#' Waggoner and Zha (1999) and is based on the conditional normal density given
#' the future projections of some of the variables created basing on the 
#' one-period ahead predictive density.
#' 
#' \strong{Exogenous variables.}
#' Forecasting with models for which specification argument 
#' \code{exogenous_variables} was specified required providing the future values
#' of these exogenous variables in the argument \code{exogenous_forecast} of the
#' \code{\link{forecast.PosteriorBVARPANEL}} function.
#' 
#' \strong{Truncated forecasts for variables of type 'rate'.}
#' The package provides the option to truncate the forecasts for variables of 
#' for which the corresponding element of argument \code{type} of the function 
#' \code{specify_bvarPANEL$new()} is set to \code{"rate"}. The one-period-ahead
#' predictive normal density for such variables is truncated to values from 
#' interval  \eqn{[0,100]}. 
#' 
#' @method forecast PosteriorBVARs
#' 
#' @param object posterior estimation outcome - an object of class 
#' \code{PosteriorBVARs} obtained by running the \code{estimate} function.
#' @param horizon a positive integer, specifying the forecasting horizon.
#' @param exogenous_forecast not used here ATM; included for compatibility with 
#' generic \code{forecast}.
#' @param conditional_forecast a list of length \code{C} containing 
#' \code{horizon x N} matrices with forecasted values for selected variables. 
#' These matrices should only contain \code{numeric} or \code{NA} values. The 
#' entries with \code{NA} values correspond to the values that are forecasted 
#' conditionally on the realisations provided as \code{numeric} values.
#' @param ... not used
#' 
#' @return A list of class \code{ForecastsPANEL} with \code{C} elements containing 
#' the draws from the country-specific predictive density and data in a form of 
#' object class \code{Forecasts} that includes:
#' 
#' \describe{
#'  \item{forecasts}{an \code{NxhorizonxS} array with the draws from the 
#'  country-specific predictive density}
#'  \item{forecast_mean}{an \code{NxhorizonxS} array with the mean of the 
#'  country-specific predictive density}
#'  \item{forecast_cov}{an \code{NxNxhorizonxS} array with the covariance of the 
#'  country-specific predictive density}
#'  \item{Y}{a \code{T_cxN} matrix with the country-specific data}
#' }
#'
#' @references
#' Waggoner, D. F., & Zha, T. (1999) 
#' Conditional forecasts in dynamic multivariate models, 
#' \emph{Review of Economics and Statistics}, \bold{81}(4), 639-651,
#' \doi{10.1162/003465399558508}.
#'
#' @seealso \code{\link{specify_bvars}}, \code{\link{estimate.PosteriorBVARs}}, 
#' \code{\link{summary.ForecastsPANEL}}, \code{\link{plot.ForecastsPANEL}}
#'
#' @author Tomasz Woźniak \email{wozniak.tom@pm.me}
#' 
#' @examples
#' # specify the model
#' specification = specify_bvars$new( 
#'   ilo_dynamic_panel[1:5], 
#'   exogenous = ilo_exogenous_variables[1:5]
#' )
#' burn_in       = estimate(specification, 5)             # run the burn-in; use say S = 10000
#' posterior     = estimate(burn_in, 5)                   # estimate the model; use say S = 10000
#' 
#' # forecast 5 years ahead
#' predictive    = forecast(posterior, 5, exogenous_forecast = ilo_exogenous_forecasts[1:5])
#' 
#' @export
forecast.PosteriorBVARs = function(
    object, 
    horizon = 1, 
    exogenous_forecast = NULL,
    conditional_forecast = NULL,
    ...
) {
  
  stopifnot(
    "Argument horizon must be a positive integer number." = 
      horizon > 0 & horizon %% 1 == 0
  )
  
  posterior_A_c_cpp     = object$posterior$A_c_cpp
  posterior_Sigma_c_cpp = object$posterior$Sigma_c_cpp
  
  Y_c             = object$posterior$Y
  N               = dim(posterior_A_c_cpp[1,1][[1]])[2]
  K               = dim(posterior_A_c_cpp[1,1][[1]])[1]
  C               = dim(posterior_A_c_cpp[1,1][[1]])[3]
  S               = dim(posterior_A_c_cpp)[1]
  p               = object$last_draw$p
  c_names         = names(object$last_draw$data_matrices$Y)
  
  d               = K - N * p - 1
  if (d == 0 ) {
    # this will not be used for forecasting, but needs to be provided
    exogenous_forecast = list()
    for (c in 1:C) exogenous_forecast[[c]] = matrix(NA, horizon, 1)
  } else {
    stopifnot("Forecasted values of exogenous variables are missing." 
              = (d > 0) & !is.null(exogenous_forecast))
    stopifnot("The matrix of exogenous_forecast does not have a correct number of columns." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){ncol(x)}))) == d)
    stopifnot("Provide exogenous_forecast for all forecast periods specified by argument horizon." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){nrow(x)}))) == horizon)
    stopifnot("Argument exogenous has to be a matrix." 
              = all(simplify2array(lapply(exogenous_forecast, function(x){is.matrix(x) & is.numeric(x)}))))
    stopifnot("Argument exogenous cannot include missing values." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){any(is.na(x))}))) == FALSE)
  }
  
  if ( is.null(conditional_forecast) ) {
    conditional_forecast = list()
    for (c in 1:C) conditional_forecast[[c]] = matrix(NA, horizon, N)
  } else {
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data." 
              = is.list(conditional_forecast) & length(conditional_forecast) == C
    )
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data."
              = all(c_names == names(conditional_forecast))
    )
    stopifnot("Argument conditional_forecast must be a list with matrices with numeric values."
              = all(sapply(conditional_forecast, function(x) is.matrix(x) & is.numeric(x)))
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of rows equal to the value of argument horizon."
              = unique(sapply(conditional_forecast, function(x) nrow(x) )) == horizon
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of columns equal to the number of columns in the used data."
              = unique(sapply(conditional_forecast, function(x) ncol(x) )) == N
    )
  }
  
  type      = object$last_draw$data_matrices$type
  LB        = rep(-Inf, N)
  UB        = rep(Inf, N)
  rates_id  = which(type == "rate")
  if (length(rates_id) > 0) {
    LB[rates_id] = 0
    UB[rates_id] = 100
  }
  
  # perform forecasting
  fff           = .Call(`_bpvars_forecast_bvarPANEL`, 
                        posterior_A_c_cpp, 
                        posterior_Sigma_c_cpp, 
                        Y_c, 
                        conditional_forecast, 
                        exogenous_forecast, 
                        horizon,
                        LB,
                        UB,
                        TRUE,
                        p
  )
  
  forecasts       = list()
  Ymean           = .Call(`_bpvars_mean_field`, Y_c )
  
  for (c in 1:C) {
    fore            = list()
    fore_tmp        = aperm(fff$forecasts_cpp[c,1][[1]], c(2,1,3))
    fore$forecasts  = fore_tmp
    
    fmean_tmp       = aperm(fff$forecast_mean_cpp[c,1][[1]], c(2,1,3))
    fore$forecast_mean = fmean_tmp
    
    cov_tmp        = array(NA, c(N, N, horizon, S))
    for (s in 1:S) {
      cov_tmp[,,,s] = fff$forecast_cov_cpp[c,s][[1]]
    }
    fore$forecast_cov = cov_tmp
    
    fore$Y          = t(Ymean[[c]])
    class(fore)     = "Forecasts"
    forecasts[[c]]  = fore
  }
  names(forecasts)  = c_names
  class(forecasts)  = "ForecastsPANEL"
  
  return(forecasts)
}




#' @inherit forecast.PosteriorBVARPANEL
#' @method forecast PosteriorBVARGROUPPRIORPANEL
#' 
#' @param object posterior estimation outcome - an object of class 
#' \code{PosteriorBVARGROUPPRIORPANEL} obtained by running the \code{estimate} function.
#' 
#' @seealso \code{\link{specify_bvarGroupPriorPANEL}}, \code{\link{estimate.PosteriorBVARGROUPPRIORPANEL}}, 
#' \code{\link{summary.ForecastsPANEL}}, \code{\link{plot.ForecastsPANEL}}
#'
#' @examples
#' # specify the model
#' specification = specify_bvarGroupPriorPANEL$new(
#'                   ilo_dynamic_panel[1:5],
#'                   group_allocation = country_grouping_incomegroup[1:5]
#'                 )
#' burn_in       = estimate(specification, 5)             # run the burn-in; use say S = 10000
#' posterior     = estimate(burn_in, 5)                   # estimate the model; use say S = 10000
#' 
#' # forecast 5 years ahead
#' predictive    = forecast(posterior, horizon = 5)
#' 
#' @export
forecast.PosteriorBVARGROUPPRIORPANEL = function(
    object, 
    horizon = 1, 
    exogenous_forecast = NULL,
    conditional_forecast = NULL,
    ...
) {
  
  stopifnot(
    "Argument horizon must be a positive integer number." = 
      horizon > 0 & horizon %% 1 == 0
  )
  
  posterior_A_c_cpp     = object$posterior$A_c_cpp
  posterior_Sigma_c_cpp = object$posterior$Sigma_c_cpp
  
  Y_c             = object$posterior$Y
  N               = dim(posterior_A_c_cpp[1,1][[1]])[2]
  K               = dim(posterior_A_c_cpp[1,1][[1]])[1]
  C               = dim(posterior_A_c_cpp[1,1][[1]])[3]
  S               = dim(posterior_A_c_cpp)[1]
  p               = object$last_draw$p
  c_names         = names(object$last_draw$data_matrices$Y)
  
  d               = K - N * p - 1
  if (d == 0 ) {
    # this will not be used for forecasting, but needs to be provided
    exogenous_forecast = list()
    for (c in 1:C) exogenous_forecast[[c]] = matrix(NA, horizon, 1)
  } else {
    stopifnot("Forecasted values of exogenous variables are missing." 
              = (d > 0) & !is.null(exogenous_forecast))
    stopifnot("The matrix of exogenous_forecast does not have a correct number of columns." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){ncol(x)}))) == d)
    stopifnot("Provide exogenous_forecast for all forecast periods specified by argument horizon." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){nrow(x)}))) == horizon)
    stopifnot("Argument exogenous has to be a matrix." 
              = all(simplify2array(lapply(exogenous_forecast, function(x){is.matrix(x) & is.numeric(x)}))))
    stopifnot("Argument exogenous cannot include missing values." 
              = unique(simplify2array(lapply(exogenous_forecast, function(x){any(is.na(x))}))) == FALSE)
  }
  
  if ( is.null(conditional_forecast) ) {
    conditional_forecast = list()
    for (c in 1:C) conditional_forecast[[c]] = matrix(NA, horizon, N)
  } else {
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data." 
              = is.list(conditional_forecast) & length(conditional_forecast) == C
    )
    stopifnot("Argument conditional_forecast must be a list with the same countries 
              as in the provided data."
              = all(c_names == names(conditional_forecast))
    )
    stopifnot("Argument conditional_forecast must be a list with matrices with numeric values."
              = all(sapply(conditional_forecast, function(x) is.matrix(x) & is.numeric(x)))
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of rows equal to the value of argument horizon."
              = unique(sapply(conditional_forecast, function(x) nrow(x) )) == horizon
    )
    stopifnot("All the matrices provided in argument conditional_forecast must have 
              the same number of columns equal to the number of columns in the used data."
              = unique(sapply(conditional_forecast, function(x) ncol(x) )) == N
    )
  }
  
  type      = object$last_draw$data_matrices$type
  LB        = rep(-Inf, N)
  UB        = rep(Inf, N)
  rates_id  = which(type == "rate")
  if (length(rates_id) > 0) {
    LB[rates_id] = 0
    UB[rates_id] = 100
  }
  
  # perform forecasting
  fff           = .Call(`_bpvars_forecast_bvarPANEL`, 
                        posterior_A_c_cpp, 
                        posterior_Sigma_c_cpp, 
                        Y_c, 
                        conditional_forecast, 
                        exogenous_forecast, 
                        horizon,
                        LB,
                        UB,
                        TRUE,
                        p
  )
  
  forecasts       = list()
  Ymean           = .Call(`_bpvars_mean_field`, Y_c )
  
  for (c in 1:C) {
    fore            = list()
    fore_tmp        = aperm(fff$forecasts_cpp[c,1][[1]], c(2,1,3))
    fore$forecasts  = fore_tmp
    
    fmean_tmp       = aperm(fff$forecast_mean_cpp[c,1][[1]], c(2,1,3))
    fore$forecast_mean = fmean_tmp
    
    cov_tmp        = array(NA, c(N, N, horizon, S))
    for (s in 1:S) {
      cov_tmp[,,,s] = fff$forecast_cov_cpp[c,s][[1]]
    }
    fore$forecast_cov = cov_tmp
    
    fore$Y          = t(Ymean[[c]])
    class(fore)     = "Forecasts"
    forecasts[[c]]  = fore
  }
  names(forecasts)  = c_names
  class(forecasts)  = "ForecastsPANEL"
  
  return(forecasts)
}
