% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_lumi.R
\name{compute_lumi}
\alias{compute_lumi}
\title{Compute land-use mix indicators on a spatial grid}
\usage{
compute_lumi(
  code_muni,
  year = 2022,
  polygon_type = c("hex", "user"),
  polygon = NULL,
  crs_output = NULL,
  h3_resolution = 9,
  verbose = TRUE,
  backend = c("duckdb", "r")
)
}
\arguments{
\item{code_muni}{Integer. Seven-digit IBGE municipality code.}

\item{year}{Integer. The CNEFE data year. Currently only 2022 is supported.
Defaults to 2022.}

\item{polygon_type}{Character. Type of polygon aggregation: \code{"hex"} (default)
uses an H3 hexagonal grid; \code{"user"} uses polygons provided via the \code{polygon}
parameter.}

\item{polygon}{An \code{\link[sf:sf]{sf::sf}} object with polygon geometries. Required when
\code{polygon_type = "user"}. A warning is issued reporting the percentage of
CNEFE points covered by the polygon area. If no CNEFE points fall within
the polygon, an error is raised.}

\item{crs_output}{The CRS for the output object. Only used when
\code{polygon_type = "user"}. Default is \code{NULL}, which uses the original CRS of
the \code{polygon} argument. Can be an EPSG code (e.g., 4326, 31983) or any CRS
object accepted by \code{\link[sf:st_transform]{sf::st_transform()}}.}

\item{h3_resolution}{Integer. H3 grid resolution (default: 9). Only used when
\code{polygon_type = "hex"}.}

\item{verbose}{Logical; if \code{TRUE}, prints messages and timing information.}

\item{backend}{Character. \code{"duckdb"} (default) uses DuckDB + H3 extension
reading directly from the cached ZIP. \code{"r"} computes H3 in R using h3jsr.}
}
\value{
An \code{\link[sf:sf]{sf::sf}} object containing:
\describe{
\item{When \code{polygon_type = "hex"}:}{
\itemize{
\item \code{id_hex}: H3 cell identifier
\item \code{p_res}, \code{ei}, \code{hhi}, \code{bal}, \code{ice}, \code{hhi_adp}, \code{bgbi}: land-use
mix indicators
\item \code{geometry}: hexagon geometry (CRS 4326)
}
}
\item{When \code{polygon_type = "user"}:}{
\itemize{
\item Original columns from \code{polygon}
\item \code{p_res}, \code{ei}, \code{hhi}, \code{bal}, \code{ice}, \code{hhi_adp}, \code{bgbi}: land-use
mix indicators
\item \code{geometry}: polygon geometry (in the original or \code{crs_output} CRS)
}
}
}
}
\description{
\code{compute_lumi()} reads CNEFE records for a given municipality,
assigns each address point to spatial units (either H3 hexagonal cells or
user-provided polygons), and computes the residential proportion (\code{p_res}) and land-use mix
indices, such as the Entropy Index (\code{ei}), the Herfindahl-Hirschman Index (\code{hhi}),
the Balance Index (\code{bal}), the Index of Concentration at Extremes (\code{ice}), the adapted HHI (\code{hhi_adp}),
and the Bidirectional Global-centered Index (\code{bgbi}), following the methodology proposed in
Pedreira Jr. et al. (2025).
}
\examples{
\donttest{
# Compute land-use mix indices on H3 hexagons
lumi <- compute_lumi(code_muni = 2929057)

# Compute land-use mix indices on user-provided polygons (neighborhoods of Lauro de Freitas-BA)
# Using geobr to download neighborhood boundaries
library(geobr)
nei_ldf <- subset(
  read_neighborhood(year = 2022),
  code_muni == 2919207
)
lumi_poly <- compute_lumi(
  code_muni = 2919207,
  polygon_type = "user",
  polygon = nei_ldf
)
}

}
\references{
Pedreira Jr., J. U.; Louro, T. V.; Assis, L. B. M.; Brito, P. L.
Measuring land use mix with address-level census data (2025).
\emph{engrXiv}. https://engrxiv.org/preprint/view/5975

Booth, A.; Crouter, A. C. (Eds.). (2001).
\emph{Does It Take a Village? Community Effects on Children, Adolescents, and Families}.
Psychology Press.

Song, Y.; Merlin, L.; Rodriguez, D. (2013).
Comparing measures of urban land use mix.
\emph{Computers, Environment and Urban Systems}, 42, 1--13.
https://doi.org/10.1016/j.compenvurbsys.2013.08.001
}
