% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/init_resil.R
\name{init_resil}
\alias{init_resil}
\title{Calculate the initial resilience of a community after disturbance}
\usage{
init_resil(B)
}
\arguments{
\item{B}{a matrix, containing the interactions between the species
or functional groups in the community.
Can be calculated with \code{\link{extractB}} from the
fitted MARSS object.}
}
\value{
A single numeric value, the initial resilience. The larger its value,
the more stable the system.
}
\description{
\code{init_resil} calculates initial resilience
\eqn{R_0}
as the initial rate of return to equilibrium (Downing et al. 2020).
}
\details{
\deqn{
R_0 = -\log \!\left(
        \sqrt{\lambda_{\mathrm{dom}}\!\left( B^{\mathsf{T}} B \right)}
      \right)
}
where
\eqn{\lambda_{\mathrm{dom}}}
is the dominant eigenvalue.
}
\examples{
library(MARSS)

\donttest{
  # smaller dataset for example:
  # 3 functional groups and two insecticide concentrations besides control
  data_df <- subset(
    aquacomm_fgps,
    treat \%in\% c(0.0, 0.9, 6) &
      time >= 1 & time <= 28,
    select = c(time, treat, repli, herb, carn, detr)
  )

  # estimate z-score transformation and replace zeros with NA
  data_df[, c("herb", "carn", "detr")] <- lapply(data_df[, c("herb", "carn", "detr")],
                                                 MARSS::zscore)
  data_df[, c("herb", "carn", "detr")] <- lapply(data_df[, c("herb", "carn", "detr")],
                                                 function(x) replace(x, x == 0, NA))

  # reshape data from wide to long format
  data_z_ldf <- reshape(
    data_df,
    varying = list(c("herb", "carn", "detr")),
    v.names = "abund_z",
    timevar = "fgp",
    times = c("herb", "carn", "detr"),
    direction = "long",
    idvar = c("time", "treat", "repli")
  )

  data_z_ldf <- data_z_ldf[order(data_z_ldf$time, data_z_ldf$treat, data_z_ldf$fgp), ]

  # summarize mean and sd
  data_z_summldf <- aggregate(abund_z ~ time + treat + fgp, data_z_ldf, function(x)
    c(mean = mean(x, na.rm = TRUE), sd = sd(x, na.rm = TRUE)))
  data_z_summldf <- do.call(data.frame, data_z_summldf)
  names(data_z_summldf)[4:5] <- c("abundz_mu", "abundz_sd")

  # split dataframe per functional groups
  # into list to apply the MARSS model more easily
  split_data_z <- split(data_z_summldf[, c("time", "fgp", "abundz_mu")], data_z_summldf$treat)

  reshape_to_wide <- function(df) {
    df_wide <- reshape(df,
                       idvar = "fgp",
                       timevar = "time",
                       direction = "wide")
    rownames(df_wide) <- df_wide$fgp
    df_wide <- df_wide[, -1]  # Remove the 'fgp' column
    as.matrix(df_wide)
  }

  data_z_summls <- lapply(split_data_z, reshape_to_wide)

  # fit MARSS models
  data.marssls <- list(
    MARSS(
      data_z_summls[[1]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    ),
    MARSS(
      data_z_summls[[2]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    ),
    MARSS(
      data_z_summls[[3]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    )
  )

  # identify experiments
  names(data.marssls) <- paste0("Conc. = ", c("0", "0.9", "44"), " micro g/L")

  # extract community matrices (B)
  data.Bls <- data.marssls |>
    lapply(extractB,
           states_names = c("Herbivores", "Carnivores", "Detrivores"))

  # calculate initial resilience for each of the B matrices
  purrr::map(data.Bls, init_resil)

}
}
\references{
Downing, A. L., Jackson, C., Plunkett, C., Lockhart, J. A., Schlater, S. M., & Leibold, M. A. (2020). Temporal stability vs. Community matrix measures of stability and the role of weak interactions. Ecology Letters, 23(10), 1468–1478. \doi{10.1111/ele.13538}
}
\seealso{
\code{\link[=extractB]{extractB()}}
}
