use crate::assert;
use crate::internal_prelude::*;
use linalg::householder::{self};
/// information about the resulting $QR$ factorization.
#[derive(Copy, Clone, Debug)]
pub struct QrInfo {
	/// estimated rank of the matrix.
	pub rank: usize,
}
fn qr_in_place_unblocked<T: ComplexField>(
	A: MatMut<'_, T>,
	H: RowMut<'_, T>,
	row_start: usize,
	col_start: usize,
) -> usize {
	let mut A = A;
	let mut H = H;
	let (m, n) = A.shape();
	let size = H.ncols();
	let mut col = col_start;
	let mut row = row_start;
	while row < Ord::min(size, m) && col < n {
		let norm = A.rb().col(col).get(..row).norm_l2();
		let (mut A00, A01, A10, A11) =
			A.rb_mut().split_at_mut(row + 1, col + 1);
		let (mut A10l, A10r) = A10.split_at_col_mut(col);
		let mut A10r = A10r.col_mut(0);
		let A01 = A01.row_mut(row);
		let A00 = &mut A00[(row, col)];
		let (info, v) = if row == col {
			let info =
				householder::make_householder_in_place(A00, A10r.rb_mut());
			(info, A10r.rb())
		} else {
			let info = householder::make_householder_out_of_place(
				A00,
				A10l.rb_mut().col_mut(row),
				A10r.rb(),
			);
			let nrows = A10r.nrows();
			A10r.rb_mut()
				.get_mut(..Ord::min(nrows, col - row))
				.fill(zero());
			(info, A10l.rb().col(row))
		};
		let norm = &info.norm.hypot(norm);
		let eps = &eps::<T::Real>();
		let leeway = &from_f64::<T::Real>((m - row) as f64 * 16.0);
		let threshold = eps * leeway * norm;
		if info.norm > threshold {
			let tau_inv = &info.tau.recip();
			H[row] = info.tau.to_cplx();
			for (head, tail) in
				core::iter::zip(A01.iter_mut(), A11.col_iter_mut())
			{
				let dot = &*head
					+ linalg::matmul::dot::inner_prod(
						v.transpose(),
						Conj::Yes,
						tail.rb(),
						Conj::No,
					);
				let k = &-dot.mul_real(tau_inv);
				*head += k;
				z!(tail, v).for_each(|uz!(dst, src)| {
					*dst += k * src;
				});
			}
			row += 1;
		}
		col += 1;
	}
	row
}
/// the recommended block size to use for a $QR$ decomposition of a matrix with
/// the given shape.
#[inline]
pub fn recommended_block_size<T: ComplexField>(
	nrows: usize,
	ncols: usize,
) -> usize {
	let prod = nrows * ncols;
	let size = nrows.min(ncols);
	(if prod > 8192 * 8192 {
		256
	} else if prod > 2048 * 2048 {
		128
	} else if prod > 1024 * 1024 {
		64
	} else if prod > 512 * 512 {
		48
	} else if prod > 128 * 128 {
		32
	} else if prod > 32 * 32 {
		8
	} else if prod > 16 * 16 {
		4
	} else {
		1
	})
	.min(size)
	.max(1)
}
/// $QR$ factorization tuning parameters.
#[derive(Debug, Copy, Clone)]
pub struct QrParams {
	/// threshold at which blocking algorithms should be disabled
	pub blocking_threshold: usize,
	/// threshold at which the parallelism should be disabled
	pub par_threshold: usize,
	#[doc(hidden)]
	pub non_exhaustive: NonExhaustive,
}
impl<T: ComplexField> Auto<T> for QrParams {
	#[inline]
	fn auto() -> Self {
		Self {
			blocking_threshold: 48 * 48,
			par_threshold: 192 * 256,
			non_exhaustive: NonExhaustive(()),
		}
	}
}
fn qr_in_place_blocked<T: ComplexField>(
	A: MatMut<'_, T>,
	H: MatMut<'_, T>,
	row_start: usize,
	col_start: usize,
	par: Par,
	stack: &mut MemStack,
	params: Spec<QrParams, T>,
) -> usize {
	let params = params.config;
	let (m, n) = A.shape();
	let size = Ord::min(m, n);
	let block_size = H.nrows();
	assert!(block_size > 0);
	if block_size == 1 {
		return qr_in_place_unblocked(A, H.row_mut(0), row_start, col_start);
	}
	let sub_block_size = if m * n < params.blocking_threshold {
		1
	} else {
		block_size / 2
	};
	let mut A = A;
	let mut H = H;
	let mut col = col_start;
	let mut row = row_start;
	while row < size && col < n {
		let block_size = Ord::min(block_size, Ord::min(size - row, n - col));
		let sub_block_size = Ord::min(block_size, sub_block_size);
		let mut A = A.rb_mut();
		let mut H = H.rb_mut();
		let start = row;
		let mut offset = 0;
		while offset < block_size && col < n {
			let block_size = Ord::min(n - col, block_size - offset);
			let sub_block_size = Ord::min(block_size, sub_block_size);
			let new_row = qr_in_place_blocked(
				A.rb_mut().subcols_mut(0, col + block_size),
				H.rb_mut().subrows_mut(offset, sub_block_size),
				row,
				col,
				par,
				stack,
				params.into(),
			);
			let local = new_row - row;
			if local > 0 {
				let mut k = 0;
				while k < local {
					let sub_block_size = Ord::min(sub_block_size, local - k);
					if k > 0 {
						let mut H = H
							.rb_mut()
							.get_mut(offset.., ..)
							.subcols_mut(row + k, sub_block_size);
						let (H0, H1) = H.rb_mut().split_at_row_mut(k);
						let H0 = H0.rb().subrows(0, sub_block_size);
						let H1 = H1.subrows_mut(0, sub_block_size);
						{ H1 }.copy_from_triangular_upper(H0);
					}
					k += sub_block_size;
				}
				householder::upgrade_householder_factor(
					H.rb_mut()
						.get_mut(offset..offset + local, row..row + local),
					A.rb().get(row.., row..row + local),
					local,
					sub_block_size,
					par,
				);
				if offset > 0 {
					use crate::linalg::matmul::triangular::{
						self, BlockStructure,
					};
					let mut H = H.rb_mut().get_mut(
						..offset + local,
						start..start + offset + local,
					);
					let A = A.rb().get(start.., start..start + offset + local);
					let (A0, A1) = A.split_at_row(offset + local);
					triangular::matmul(
						H.rb_mut(),
						BlockStructure::UnitTriangularUpper,
						Accum::Replace,
						A0.adjoint(),
						BlockStructure::UnitTriangularUpper,
						A0,
						BlockStructure::UnitTriangularLower,
						one(),
						par,
					);
					triangular::matmul(
						H.rb_mut(),
						BlockStructure::UnitTriangularUpper,
						Accum::Add,
						A1.adjoint(),
						BlockStructure::Rectangular,
						A1,
						BlockStructure::Rectangular,
						one(),
						par,
					);
				}
			}
			let (Q0, A1) = A
				.rb_mut()
				.get_mut(row.., ..)
				.split_at_col_mut(col + block_size);
			let Q0 = Q0.rb().get(.., row..row + local);
			let H = H.rb().get(offset..offset + local, row..row + local);
			if A1.ncols() > 0 {
				householder::apply_block_householder_transpose_on_the_left_in_place_with_conj(Q0.rb(), H.rb(), Conj::Yes, A1, par, stack);
			}
			offset += local;
			row += local;
			col += block_size;
		}
	}
	row
}
#[track_caller]
pub fn qr_in_place<T: ComplexField>(
	A: MatMut<'_, T>,
	Q_coeff: MatMut<'_, T>,
	par: Par,
	stack: &mut MemStack,
	params: Spec<QrParams, T>,
) -> QrInfo {
	let block_size = Q_coeff.nrows();
	assert!(all(
		block_size > 0,
		Q_coeff.ncols() == Ord::min(A.nrows(), A.ncols()),
	));
	#[cfg(feature = "perf-warn")]
	if A.row_stride().unsigned_abs() != 1 && crate::__perf_warn!(QR_WARN) {
		if A.col_stride().unsigned_abs() == 1 {
			log::warn!(
				target : "faer_perf",
				"QR prefers column-major matrix. Found row-major matrix."
			);
		} else {
			log::warn!(
				target : "faer_perf",
				"QR prefers column-major matrix. Found matrix with generic strides."
			);
		}
	}
	let mut Q_coeff = Q_coeff;
	let rank =
		qr_in_place_blocked(A, Q_coeff.rb_mut(), 0, 0, par, stack, params);
	Q_coeff.rb_mut().get_mut(.., rank..).fill(zero());
	let mut col = rank / block_size * block_size;
	let n = Q_coeff.ncols();
	while col < n {
		let block_size = Ord::min(block_size, n - col);
		let start = Ord::max(rank, col);
		Q_coeff
			.rb_mut()
			.get_mut(start - col.., start..col + block_size)
			.diagonal_mut()
			.fill(infinity());
		col += block_size;
	}
	QrInfo { rank }
}
/// computes the layout of required workspace for performing a qr
/// decomposition with no pivoting
#[inline]
pub fn qr_in_place_scratch<T: ComplexField>(
	nrows: usize,
	ncols: usize,
	block_size: usize,
	par: Par,
	params: Spec<QrParams, T>,
) -> StackReq {
	let _ = par;
	let _ = nrows;
	let _ = &params;
	temp_mat_scratch::<T>(block_size, ncols)
}
#[cfg(test)]
mod tests {
	use super::*;
	use crate::linalg::solvers::DenseSolveCore;
	use crate::stats::prelude::*;
	use crate::utils::approx::*;
	use crate::{Mat, Row, assert, c64};
	use dyn_stack::MemBuffer;
	#[test]
	fn test_qr() {
		let rng = &mut StdRng::seed_from_u64(0);
		for rank in [1, 2, 3, 4, 5, 100, usize::MAX] {
			for par in [Par::Seq, Par::rayon(8)] {
				for n in [2, 4, 8, 16, 24, 32, 127, 128, 257] {
					let rank = Ord::min(n, rank);
					let approx_eq = CwiseMat(ApproxEq {
						abs_tol: 1e-10,
						rel_tol: 1e-10,
					});
					let A0 = CwiseMatDistribution {
						nrows: n,
						ncols: rank,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A1 = CwiseMatDistribution {
						nrows: rank,
						ncols: n,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A = &A0 * &A1;
					let A = A.as_ref();
					let mut QR = A.cloned();
					let mut H = Row::zeros(n);
					let mut params: QrParams = auto!(c64);
					params.blocking_threshold = usize::MAX;
					let params = params.into();
					let computed_rank = qr_in_place(
						QR.as_mut(),
						H.as_mat_mut(),
						Par::Seq,
						MemStack::new(&mut MemBuffer::new(
							qr_in_place_scratch::<c64>(
								n,
								n,
								1,
								Par::Seq,
								params,
							),
						)),
						params,
					);
					assert!(computed_rank.rank == rank);
					let mut Q = Mat::<c64>::zeros(n, n);
					let mut R = QR.as_ref().cloned();
					for j in 0..n {
						Q[(j, j)] = c64::ONE;
					}
					householder::apply_block_householder_sequence_on_the_left_in_place_with_conj(
						QR.as_ref(),
						H.as_mat(),
						Conj::No,
						Q.as_mut(),
						Par::Seq,
						MemStack::new(&mut MemBuffer::new(
							householder::apply_block_householder_sequence_transpose_on_the_left_in_place_scratch::<c64>(n, 1, n),
						)),
					);
					for j in 0..n {
						for i in j + 1..n {
							R[(i, j)] = c64::ZERO;
						}
					}
					assert!(Q * R ~ A);
				}
				for n in [2, 3, 4, 8, 16, 24, 32, 128, 255, 256, 257, 512] {
					let bs = 15;
					let rank = Ord::min(n, rank);
					let approx_eq = CwiseMat(ApproxEq {
						abs_tol: 1e-10,
						rel_tol: 1e-10,
					});
					let A0 = CwiseMatDistribution {
						nrows: n,
						ncols: rank,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A1 = CwiseMatDistribution {
						nrows: rank,
						ncols: n,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A = &A0 * &A1;
					let A = A.as_ref();
					let mut QR = A.cloned();
					let mut H = Mat::zeros(bs, n);
					let computed_rank = qr_in_place(
						QR.as_mut(),
						H.as_mut(),
						par,
						MemStack::new(&mut MemBuffer::new(
							qr_in_place_scratch::<c64>(
								n,
								n,
								bs,
								par,
								default(),
							),
						)),
						default(),
					);
					assert!(computed_rank.rank == rank);
					let mut Q = Mat::<c64>::zeros(n, n);
					let mut R = QR.as_ref().cloned();
					for j in 0..n {
						Q[(j, j)] = c64::ONE;
					}
					householder::apply_block_householder_sequence_on_the_left_in_place_with_conj(
						QR.as_ref(),
						H.as_ref(),
						Conj::No,
						Q.as_mut(),
						Par::Seq,
						MemStack::new(&mut MemBuffer::new(
							householder::apply_block_householder_sequence_on_the_left_in_place_scratch::<c64>(n, bs, n),
						)),
					);
					for j in 0..n {
						for i in j + 1..n {
							R[(i, j)] = c64::ZERO;
						}
					}
					assert!(Q * R ~ A);
				}
				let n = 20;
				for m in [2, 3, 4, 8, 16, 24, 32, 128, 255, 256, 257, 512] {
					let size = Ord::min(m, n);
					let bs = 15;
					let rank = Ord::min(size, rank);
					let approx_eq = CwiseMat(ApproxEq {
						abs_tol: 1e-10,
						rel_tol: 1e-10,
					});
					let A0 = CwiseMatDistribution {
						nrows: m,
						ncols: rank,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A1 = CwiseMatDistribution {
						nrows: rank,
						ncols: n,
						dist: ComplexDistribution::new(
							StandardNormal,
							StandardNormal,
						),
					}
					.rand::<Mat<c64>>(rng);
					let A = &A0 * &A1;
					let A = A.as_ref();
					let mut QR = A.cloned();
					let mut H = Mat::zeros(bs, size);
					let computed_rank = qr_in_place(
						QR.as_mut(),
						H.as_mut(),
						par,
						MemStack::new(&mut MemBuffer::new(
							qr_in_place_scratch::<c64>(
								m,
								n,
								bs,
								par,
								default(),
							),
						)),
						default(),
					);
					assert!(computed_rank.rank == rank);
					let mut Q = Mat::<c64, _, _>::zeros(m, m);
					let mut R = QR.as_ref().cloned();
					for j in 0..m {
						Q[(j, j)] = c64::ONE;
					}
					householder::apply_block_householder_sequence_on_the_left_in_place_with_conj(
						QR.as_ref().subcols(0, size),
						H.as_ref(),
						Conj::No,
						Q.as_mut(),
						Par::Seq,
						MemStack::new(&mut MemBuffer::new(
							householder::apply_block_householder_sequence_on_the_left_in_place_scratch::<c64>(m, bs, m),
						)),
					);
					for j in 0..n {
						for i in j + 1..m {
							R[(i, j)] = c64::ZERO;
						}
					}
					assert!(Q * R ~ A);
				}
			}
		}
	}
	#[test]
	fn test_rank_deficient() {
		let i = c64::new(0.0, 1.0);
		let A = mat![
			[
				-42.63301571125646 + 12.332799004481444 * i,
				22.14792539952694 + 34.88980973191775 * i,
				17.08833097340157 - 35.73642756638833 * i,
				16.46359410465419 - 42.92839086211859 * i,
				-51.764730444742824 + 81.65328924776695 * i,
				16.26315772194988 + 101.5902862344418 * i,
				32.77410298966414 - 29.760957827435675 * i,
				54.439294321277174 - 19.079738442315318 * i,
				-13.472034925473565 - 13.943573427112455 * i,
				-37.92130013603088 - 5.7926798723257775 * i,
				11.194976295897504 - 31.928468357395335 * i,
				22.006669976718612 + 3.3819587127690625 * i,
				-6.674280620224554 + 3.387564723853501 * i,
				-27.293486814495225 + 15.784839382140262 * i,
				-70.1443517961362 + 45.51634877173214 * i,
				8.181229285208616 + 11.577617946683347 * i,
				91.36331153456992 + 5.978813469385237 * i,
				9.984370037964844 - 46.821962088062065 * i,
				-7.961599724739131 + 24.482176731342236 * i,
				-2.3944781784861675 + 59.88090585477201 * i,
				-30.59313953515538 + 26.04745441455745 * i,
				-23.2138559332029 + 71.42630235367764 * i,
				22.210550339117574 - 8.41265667708069 * i,
				-11.395416853480953 - 21.12459074171425 * i,
				53.214674012253 - 115.94154857087852 * i,
				-73.57661533181208 - 26.388545645959155 * i,
				-7.090193099053337 + 25.72478190092928 * i,
				69.88086414991596 - 61.87734406117171 * i,
				1.7746488531197873 + 22.338809391772543 * i,
				53.17879024337897 + 3.0434791244725403 * i,
				17.11701506186965 + 8.310347464005673 * i,
				-1.4446985641791343 - 8.302085605445917 * i,
				31.01042486753896 - 7.032023465838279 * i,
				50.529023639015065 - 30.432756878940538 * i,
				31.20832359612237 - 18.42527233678696 * i,
				-11.219592741413372 + 7.966456895330857 * i,
				-29.972014970412957 + 14.566825708553516 * i,
				0.29786829050329544 - 60.43731329910441 * i,
				68.20619173666897 + 61.4177082555907 * i,
				38.18522457770308 + 39.84442487790024 * i,
			],
			[
				12.392666547455747 + 8.598469521105585 * i,
				-4.989137989279129 - 33.00917260871823 * i,
				-36.3778560847675 + 33.965754531979854 * i,
				-14.186892703037756 + 7.8005444248189235 * i,
				40.14774343578466 - 37.073977200615204 * i,
				20.515112381381726 - 39.67019203209179 * i,
				-11.247367898035563 - 24.867001222923307 * i,
				-49.7400862432402 + 41.41428057951588 * i,
				-3.7213275390216154 + 51.57768254492541 * i,
				-6.345081010580628 + 53.304858550561896 * i,
				-11.529781103419339 + 2.902201112665864 * i,
				-19.346649644440838 + 7.461276014085186 * i,
				-28.718472256689964 - 2.8043513041184553 * i,
				-5.691563912437118 + 48.35693831223085 * i,
				-34.299545543675215 + 6.545395352640526 * i,
				7.438830278671389 - 10.739267797900883 * i,
				24.11439247436401 - 29.701199086689808 * i,
				-54.93729078780896 + 17.091416018852897 * i,
				15.68845749746022 - 2.188338155362544 * i,
				60.97624713742826 + 11.795504838770498 * i,
				53.39361626612977 + 11.614849968126038 * i,
				47.112022139988945 - 47.422312498326974 * i,
				-31.731750386335847 - 13.013784865796737 * i,
				-12.242942262135827 - 23.39632386635281 * i,
				-20.99632660387538 - 50.155831920264006 * i,
				32.853343930391404 + 39.33730823584699 * i,
				25.191258444072332 + 23.926919559757664 * i,
				-34.18728556598965 + 52.5998197331729 * i,
				64.80196305869202 - 10.491798137124405 * i,
				-3.7973985569930306 - 34.25408107334158 * i,
				31.90902882064414 - 30.56021961088784 * i,
				-7.840305921418871 + 51.263365311660365 * i,
				5.999217219970496 + 60.657097999788675 * i,
				-56.00257012089648 + 32.307128304063674 * i,
				1.6182839905382513 - 30.670206400422288 * i,
				-29.228061063708484 + 4.618529965360111 * i,
				-1.841921635134554 - 9.941332210308186 * i,
				-28.83749752522963 + 23.48822343787033 * i,
				31.33927120630301 - 71.58546530411547 * i,
				8.181832440934743 + 82.64255143443799 * i,
			],
			[
				-12.799030330740077 + 66.72947765658043 * i,
				-10.231934039979912 + 12.493297096248554 * i,
				19.285109623888076 + 16.56539210325027 * i,
				-35.97113317387823 + 12.77732345394736 * i,
				-42.753078264350506 + 31.109302328632026 * i,
				-92.5918702738831 + 13.20717885481352 * i,
				-5.444744981663884 + 48.16106562407031 * i,
				48.84587551022895 + 26.251595375317834 * i,
				32.933710253207224 + 14.718356097405149 * i,
				9.648128610518484 - 3.9457537091127026 * i,
				19.03552788630428 - 19.94207328846148 * i,
				17.638202950481865 + 18.33026376219135 * i,
				2.5078899478693097 - 45.70267635206393 * i,
				65.38384334644981 - 25.199272871591987 * i,
				-63.467061790245424 + 106.98211601738333 * i,
				-55.62393136607952 - 45.51555918892136 * i,
				19.15778634967591 - 33.028235082995124 * i,
				-16.336567724656685 - 10.44522106845141 * i,
				6.666994028110054 + 32.92591049296345 * i,
				-6.047530207659875 - 27.439036726043298 * i,
				39.52962940543614 - 42.64379506697883 * i,
				-27.573007769719624 - 9.75081496679099 * i,
				-7.684664374667875 + 23.339026825258244 * i,
				22.58119786377813 - 4.454197784068818 * i,
				45.29146872103806 - 103.15857753791218 * i,
				-31.25659917861597 + 13.70164575929067 * i,
				13.660916345380933 - 19.46889545971939 * i,
				-21.96183310377741 - 23.31280166325685 * i,
				4.5653695184317655 + 10.616453681351558 * i,
				-0.25434597828302685 + 47.167740733214515 * i,
				33.82461168808063 - 20.417606401735725 * i,
				21.40415980541069 - 27.931611547777152 * i,
				49.94594590735055 - 49.56309902061 * i,
				27.886633708526702 + 29.70147478184972 * i,
				-18.16905152235941 - 10.046271334081538 * i,
				-13.32135487551704 - 60.68677560840839 * i,
				-15.237503754455295 - 6.588689364973661 * i,
				-15.481784996249857 - 46.99021467938253 * i,
				-16.136673443184375 + 13.902154631685766 * i,
				25.194569063162533 + 34.5621640946647 * i,
			],
			[
				65.48222712345446 - 4.138391495079304 * i,
				8.306462492672111 - 15.140617871481396 * i,
				-28.25587078248205 + 50.32663902684051 * i,
				-42.955896042037864 + 41.46713382497833 * i,
				75.13916830569464 - 64.90394781293571 * i,
				3.4250892035273566 - 55.122759708615945 * i,
				-38.67812775272067 + 21.094268967787286 * i,
				-39.40657764104256 - 3.733222413764196 * i,
				0.9695375254205487 + 3.876176033829374 * i,
				45.749414055844625 + 18.586694744497834 * i,
				11.935303844620492 + 39.477261236067676 * i,
				-55.36026338025768 - 19.755621035097477 * i,
				12.205216270291857 + 18.307651510095745 * i,
				-17.102914926687376 - 5.8964800477324815 * i,
				73.92643576109086 - 26.358635886040645 * i,
				18.62547989733079 + 21.445184756989846 * i,
				-56.327314013975666 - 24.534245355235583 * i,
				1.8624471836432406 + 30.085648450846378 * i,
				15.302535749118924 + 7.674212804643901 * i,
				25.8637235046063 + 0.9361225038480256 * i,
				15.603583412146435 - 1.6612068829250664 * i,
				18.47432197641083 - 63.34015823189532 * i,
				5.816489465444917 - 23.75750418328684 * i,
				12.548375804510385 + 14.79286301101477 * i,
				-66.90207518361544 + 63.910915160309486 * i,
				83.37918838614108 + 40.05271581472445 * i,
				-37.36994071626608 + 6.558073050359511 * i,
				-75.67731562330931 + 70.34100920967161 * i,
				17.123194674315357 + 2.6912987746509813 * i,
				-56.13906743564637 - 9.747808550411886 * i,
				-32.0719278876096 + 0.035416915023617435 * i,
				7.453294267488758 + 31.840840443445664 * i,
				-27.326428632546204 + 16.404546707806446 * i,
				-85.79400760768401 + 51.86531366733967 * i,
				-47.38136649846963 + 2.067569048796164 * i,
				22.65699895650813 - 26.500540841293287 * i,
				35.74882215661379 + 0.4279576089606252 * i,
				-76.88759162135085 + 61.729308548926745 * i,
				-37.44977437439012 - 56.87002751956335 * i,
				-35.08304533233703 - 4.462986912180606 * i,
			],
			[
				31.583724748998776 - 51.249777560521466 * i,
				22.65843906393765 + 17.313137285238554 * i,
				10.382748046036994 - 42.735528014733234 * i,
				20.724646012317805 + 20.900719508178582 * i,
				16.376566756962994 + 53.89212191787085 * i,
				10.066813288632034 + 52.523949880976545 * i,
				43.579591284487336 + 31.252883531054323 * i,
				28.184699679323813 - 82.13512631377462 * i,
				25.440457311415628 + 31.849880177745945 * i,
				-24.151628096605307 - 23.32888438479003 * i,
				-73.6350009853953 - 81.1777776721621 * i,
				28.364800957876895 - 3.6250014074372885 * i,
				38.396317055131306 - 69.546957049437 * i,
				-1.4874584247683202 + 6.860653145887976 * i,
				99.22080146828978 + 17.543861995516416 * i,
				-10.46250932797625 + 22.241347373245674 * i,
				-23.62190472546816 - 70.59350835314189 * i,
				26.34172971210808 - 60.22547306196421 * i,
				37.90974516173529 - 7.695380437440619 * i,
				-35.43744520165066 + 42.51167883371055 * i,
				-18.775026327486785 - 16.956301595807304 * i,
				21.840451013195377 - 7.788376505976714 * i,
				33.09952105067807 + 43.20960076304959 * i,
				19.34930634141674 + 21.351881549098977 * i,
				-143.72559656621607 - 48.39026005448909 * i,
				-84.861170978867 + 2.7837442780206416 * i,
				6.07739013993995 - 52.6344486015943 * i,
				-21.33187479671998 - 35.95542921378066 * i,
				0.08757059072188866 - 38.46847267117025 * i,
				53.77485342607943 + 49.47511865583735 * i,
				-76.81015945487538 - 18.43833465675783 * i,
				-31.308131660283053 - 39.82604917803266 * i,
				-79.42584088986062 - 36.85873541440512 * i,
				39.063233310761746 - 49.75991465666547 * i,
				-23.590897533452015 + 24.69942294209421 * i,
				-55.00539490222253 + 2.186956953950137 * i,
				-26.638771613402398 - 65.28585892133748 * i,
				11.968149530543663 + 15.74074558115147 * i,
				46.924712251106854 + 90.83370071059912 * i,
				76.4747123007834 - 29.289927777420857 * i,
			],
			[
				-25.578557076310645 - 26.766359674243205 * i,
				-31.644839606419964 - 5.617473287649716 * i,
				9.823483731434163 + 43.248069608313386 * i,
				2.672375889300838 - 4.755961211633132 * i,
				-76.9263312976145 - 50.17566661665678 * i,
				-64.95093839016988 - 45.4578250269475 * i,
				-35.560543630021414 - 3.5888284627331757 * i,
				-41.06467435210163 + 91.17075079127582 * i,
				10.85492097775365 + 31.427121748171828 * i,
				41.19076647688331 + 14.935860258180696 * i,
				45.83767858930871 + 65.30292213636108 * i,
				5.475162771136581 + 23.807507454288483 * i,
				29.287510839124877 + 41.159672576346225 * i,
				16.280362143178394 - 33.78524201973386 * i,
				-75.50032717807619 + 13.18905520206016 * i,
				7.565949918458841 - 6.56539538753897 * i,
				-2.4465013777225932 + 20.456717990118733 * i,
				14.381363859892803 + 58.38589995305506 * i,
				-29.05208950533445 + 4.706022825770864 * i,
				-0.9002125402046204 - 62.12447336684386 * i,
				19.656245943958925 - 12.869662053948904 * i,
				-42.0636887885214 - 39.24399163670813 * i,
				-0.011803621899311167 + 16.688867591699918 * i,
				-12.492893033463261 - 17.937736765361308 * i,
				64.21546351437391 + 51.605080288718526 * i,
				36.91267213569334 - 73.13098005416595 * i,
				-25.525209659150693 + 23.5352740057052 * i,
				57.65163679154867 + 48.64691617765472 * i,
				-42.496849189681654 - 5.291338162190961 * i,
				-38.954293054708295 - 5.521067322012144 * i,
				50.35676002688245 + 36.51708209327393 * i,
				38.10344718654048 + 16.797297865041322 * i,
				83.49399438639901 + 64.83208254781137 * i,
				20.74889885651865 + 64.37977126550692 * i,
				6.995143235134677 + 22.13996728228271 * i,
				63.3232030983194 - 35.57621973993962 * i,
				-9.93099758706921 + 28.827857180270613 * i,
				-9.9372392383756 + 40.36848263407507 * i,
				-81.73994066939929 - 32.540538268088895 * i,
				-6.972702160385735 + 7.2810082661594615 * i,
			],
			[
				-39.533111408076024 + 23.69499447776633 * i,
				-22.91893147481639 + 13.912288879056263 * i,
				31.970112354378752 - 6.274870212732608 * i,
				36.02649811357895 - 29.369289609658956 * i,
				-57.37126467977515 + 39.44543207716319 * i,
				-33.21392750556933 + 2.31325455824782 * i,
				5.504384182791682 + 1.1314383007183153 * i,
				-12.761658291436959 - 7.030143070779076 * i,
				25.42560890198791 + 34.2782992105918 * i,
				31.85069695776061 - 2.611504997126957 * i,
				-1.0778905536494676 + 11.186615280440869 * i,
				7.002191044964846 + 25.783443322475314 * i,
				-0.43478568633533143 - 2.03481196958748 * i,
				47.93268545760567 - 46.05766272795361 * i,
				-86.88661654441438 + 13.166255933762898 * i,
				-11.387147239610272 - 17.288479337212046 * i,
				-34.515012611515104 - 14.177064833349931 * i,
				-6.285753446891917 - 11.72228107335983 * i,
				17.88548716213686 - 4.042341176701882 * i,
				10.515130527678576 - 85.02295246871657 * i,
				34.573508949249714 - 51.81711167668266 * i,
				-11.24922483777982 + 23.114342099470775 * i,
				-16.62291909867281 + 34.402750937701654 * i,
				29.697481944292818 - 17.26924002785852 * i,
				4.287641692176678 + 34.65126459619482 * i,
				8.642563024300204 - 48.67140042344289 * i,
				0.4752675358050844 - 28.83717974726673 * i,
				40.69537730504539 - 32.475682389285055 * i,
				-45.26077064695138 - 37.87842043650326 * i,
				-23.333473340656603 - 1.090546566235858 * i,
				22.524011597323288 + 4.205992256758487 * i,
				24.334900880966856 - 72.83814875934986 * i,
				31.0040417811348 + 52.751349394370656 * i,
				46.69722487096804 + 2.2830930235470532 * i,
				19.13996268796151 + 45.686937737528105 * i,
				-3.155429138228641 - 41.98572124273077 * i,
				-0.27439111585914766 - 0.0424908693981898 * i,
				5.4676976604011465 - 15.819100391461975 * i,
				2.374657295716762 - 10.025647287456497 * i,
				37.693625151559786 - 42.41144011172045 * i,
			],
			[
				-21.55368545784853 - 11.965093929456799 * i,
				15.269836381018422 - 6.723746991502296 * i,
				-30.929835536710243 + 23.625022083338905 * i,
				-13.531291907458922 - 52.08708061196921 * i,
				-44.92575740261259 - 30.00914161138877 * i,
				24.59902956675443 - 16.30770435777285 * i,
				-47.80025879855576 - 41.46279563919512 * i,
				11.20398486871565 + 36.3881233959801 * i,
				-47.22624254734831 - 11.141717272423364 * i,
				-23.171658265997962 - 8.038332725429228 * i,
				45.70764728757604 + 29.482096062539163 * i,
				7.544132969347288 + 2.3976982476295063 * i,
				25.930035282855414 + 59.65598345400858 * i,
				-51.433293377550726 - 5.854456745366065 * i,
				-25.730730323328658 - 69.86711542678464 * i,
				23.928134030762088 + 11.653568766281879 * i,
				37.05319179515189 + 17.7987872750021 * i,
				-10.979415188021491 - 1.4727761821351173 * i,
				-37.19375468872414 + 0.6767807462333195 * i,
				29.507732428817008 + 18.52436798045187 * i,
				-10.112277301766158 + 37.50012696393238 * i,
				-14.934816780618064 + 16.622070267250677 * i,
				41.00689183009243 - 42.87360584605926 * i,
				-7.496381140067598 + 16.285315632526927 * i,
				-7.561809097739257 + 34.0552036851923 * i,
				35.54745065665762 - 11.6406777042546 * i,
				-51.1577320652325 + 13.70231514956923 * i,
				33.1223690319064 - 11.290266444430863 * i,
				-32.90001970416135 + 33.76462261168461 * i,
				6.053853814446258 + 0.9147809286238626 * i,
				16.187628643199545 + 54.050152022727275 * i,
				24.24447446880718 + 27.403214810457058 * i,
				30.665341430648112 + 42.51731745981449 * i,
				-6.620407584002567 - 20.44372835756372 * i,
				10.23165104189317 + 9.928497327685793 * i,
				121.70943435733494 + 18.294569782547413 * i,
				-3.6847206148913685 + 69.79985799810493 * i,
				-5.054102781790107 - 6.3692984966866995 * i,
				-47.61454098877892 - 59.293642895756484 * i,
				-34.900856482546594 - 51.790923765750144 * i,
			],
			[
				17.25576978214997 + 33.96148302919222 * i,
				44.714117433784864 - 5.411255296952598 * i,
				-46.863417595503336 + 5.508210705561666 * i,
				-34.18132328642872 + 5.272749724437205 * i,
				46.4419425747629 - 53.32689961225293 * i,
				38.29445042230218 + 23.197929666400228 * i,
				17.21281092244584 + 8.230188131904669 * i,
				-7.217789039096928 + 29.041218034259046 * i,
				15.73517400166007 + 1.9407830190829465 * i,
				7.47632611613335 + 7.978325374479539 * i,
				-24.988360236253527 + 20.123721221995247 * i,
				-38.2075730880623 - 7.333836942616784 * i,
				-21.858663091236537 - 38.90229242928695 * i,
				-9.059545472857895 + 43.50862080875834 * i,
				39.98959128240756 + 56.92806332756845 * i,
				-18.600231145661766 + 2.6396216335135136 * i,
				-1.4608102424286642 - 35.717697857977726 * i,
				-4.150810006497679 + 32.086732333943694 * i,
				38.91902454862037 + 4.468176459346665 * i,
				-1.1697858359666284 - 64.99977103743046 * i,
				0.4893420729536899 - 1.0099182663732904 * i,
				-8.446213900022329 - 25.984869471148087 * i,
				-15.59524811926535 - 6.057014366076132 * i,
				-26.543712262455216 - 24.748644114504856 * i,
				-18.12802694989497 - 68.21035571463506 * i,
				-16.224582597247643 + 8.015061611429338 * i,
				14.159319660845412 + 2.54205111699531 * i,
				-27.67826887730265 + 14.907573609844768 * i,
				45.471642789278334 - 15.86728255811461 * i,
				17.232632641938014 - 13.467429805676602 * i,
				-36.256180213437304 - 15.860064222576765 * i,
				1.0021755792482239 + 41.239542333909554 * i,
				-9.256238161631071 - 60.492571285285656 * i,
				-15.63297732439154 + 11.609718253052726 * i,
				-9.448486745366756 - 71.70278041871568 * i,
				-39.39640690815541 + 80.78519727338436 * i,
				-39.395869356072204 - 31.15543950766675 * i,
				-26.864700236517663 + 29.262478568227674 * i,
				32.815596529404914 - 34.11708879171351 * i,
				63.42201800783117 + 40.94151078790488 * i,
			],
			[
				-37.85424922481582 - 18.632083635692226 * i,
				8.30192797319016 + 77.51772701863564 * i,
				-38.55632574319898 - 54.06715956988576 * i,
				31.247394161859802 - 88.94283002300054 * i,
				11.369144694581848 + 73.7852859262728 * i,
				43.166934241082814 + 6.0752659654121715 * i,
				33.72393018072812 - 39.2055653391711 * i,
				-54.64017042959031 + 8.195171091011446 * i,
				-23.51135860051815 + 9.802965968574085 * i,
				-1.0787050942309264 + 7.730773526337842 * i,
				-0.1954232786143706 + 17.69212538790228 * i,
				3.56906505863984 - 13.724978295570452 * i,
				-12.070894188488708 + 11.565186160031828 * i,
				-66.44997934464766 - 48.32425130895769 * i,
				-72.29920368488105 - 43.653088379987054 * i,
				30.14758969155585 - 30.772317284413276 * i,
				67.81080621413294 + 33.15388129620932 * i,
				4.909043167647951 - 58.061131485356384 * i,
				-25.76355904795031 + 13.034149366895122 * i,
				12.715904812920968 + 67.60581655112854 * i,
				-5.6167982150059 + 68.44469029883031 * i,
				24.31964208796947 + 77.67068147975843 * i,
				23.656225127895787 - 41.94456731192289 * i,
				17.492984169162654 - 24.571122134909402 * i,
				54.80499072287952 - 62.073916111466644 * i,
				-40.846550889156816 - 1.9589474792947916 * i,
				-25.0737715301371 - 1.9674768751847722 * i,
				36.00624296393538 - 49.58243114695291 * i,
				-7.678194485500839 + 6.258226835328728 * i,
				19.135758311254307 + 37.51858856067837 * i,
				-11.795536865338804 + 77.23125200441869 * i,
				-11.450170627980023 - 16.64776327917454 * i,
				-6.99564453031995 + 53.86331982831497 * i,
				56.08323835230311 - 54.03674113601569 * i,
				85.08202088984388 + 30.210597189390157 * i,
				9.491599772377413 - 1.1313051456639514 * i,
				28.465770812149742 - 14.522398049695596 * i,
				-37.056379799428775 - 35.63554255779242 * i,
				86.60950287192956 + 6.553565452138766 * i,
				-25.432851041005407 + 43.3044718614866 * i,
			],
			[
				38.31410384312522 - 21.401900990808706 * i,
				13.819070705307922 + 10.751535690829314 * i,
				-6.34071019755625 - 60.4107514871459 * i,
				1.0358618673699311 + 28.579887281978007 * i,
				26.61381527141313 - 20.40908457666393 * i,
				15.379552452611335 - 38.10152094685664 * i,
				-2.8848738006862504 + 25.37282331258244 * i,
				-58.444474255501476 + 22.1614500700669 * i,
				32.71610777097065 - 7.800021473935473 * i,
				52.74668124709601 - 35.096670997852755 * i,
				-15.224283378548222 - 0.3716116323928986 * i,
				-9.507398139383266 + 1.2854765551085165 * i,
				28.846067346072033 - 23.556902272759512 * i,
				-5.359722010456778 - 23.102588929338012 * i,
				70.82975944282254 + 6.305168317276269 * i,
				-6.8153005132962505 - 32.111785425358114 * i,
				-64.92515970955154 - 1.8932624557584674 * i,
				6.80973203113561 + 11.103695465268103 * i,
				26.877413356688578 - 24.88491047641267 * i,
				-6.3727499924673765 - 33.54877708349753 * i,
				-17.09305048648136 - 51.61851764252174 * i,
				-0.6958815178287416 - 13.011389209324562 * i,
				-4.7920081526425236 + 0.08560497128022604 * i,
				-7.11538373141579 + 33.180232214236895 * i,
				-7.057677675147627 + 33.704749876766826 * i,
				13.394069460401283 + 23.499775650134666 * i,
				33.032957263602064 - 17.162411901076496 * i,
				-9.723684915207429 + 22.90261459373055 * i,
				8.417780554636167 - 45.476415590540924 * i,
				-12.925373617724695 + 25.701470498566195 * i,
				-77.62893801412532 + 8.771172991492747 * i,
				-12.577190512415115 + 11.770869636170222 * i,
				-28.631356375314837 - 44.33826543718645 * i,
				8.667376603495633 + 41.49404929739423 * i,
				-2.4385088484789605 + 0.04378952650684198 * i,
				-81.8696956832087 + 41.58634268689738 * i,
				38.43811817341395 - 65.59361739828425 * i,
				12.923498206429501 + 10.578834645788252 * i,
				-17.40928902913147 + 1.882715575965321 * i,
				-12.197281737560397 + 24.644707351619186 * i,
			],
			[
				15.624787749477978 - 3.08175959535572 * i,
				31.64127531457332 - 16.881615782042036 * i,
				1.890207866353876 + 50.32088799159419 * i,
				-65.4977125824758 + 24.989786335448628 * i,
				34.81214444655414 - 59.30792733893145 * i,
				-10.433501968142323 - 22.998943439836978 * i,
				10.502853068617924 + 33.94711412226598 * i,
				15.593203335082762 + 76.6520637908932 * i,
				-13.395991136070506 + 8.687052915142047 * i,
				2.081731201094403 + 3.8559227478572495 * i,
				30.583016268547937 + 1.2879653859015718 * i,
				-25.29562951899294 + 8.41557681790519 * i,
				11.479123421013526 - 22.8866683592249 * i,
				-52.19831242424364 + 68.83294717361125 * i,
				32.39840985061852 - 7.468503718345374 * i,
				7.345658271150569 - 44.65348101340274 * i,
				18.32361340083014 - 20.175472912397026 * i,
				-13.366276401727387 + 39.4889913842033 * i,
				10.173839514362811 - 2.5344239200743885 * i,
				12.966807751013704 - 7.825732938608937 * i,
				19.990977936447862 + 26.96608297565048 * i,
				-3.172560104018274 - 18.614006882422416 * i,
				-29.777636820282737 - 60.630406210085766 * i,
				-28.614421681306737 - 13.463181327439386 * i,
				32.29564701389236 + 0.346177776532528 * i,
				10.929769288479605 + 15.84579430884051 * i,
				32.72281107515033 + 56.370005822465686 * i,
				-10.16653333610929 + 23.88123265466092 * i,
				52.95907792627234 + 51.23002827773357 * i,
				4.54104618436611 + 31.14307823441345 * i,
				29.339772939929347 + 3.842122341156994 * i,
				8.881209533228002 + 49.588235065052714 * i,
				48.72530396726207 - 61.79170083110623 * i,
				-26.7340209080433 + 26.480558437595334 * i,
				7.460845492562186 - 51.590210517672396 * i,
				-21.91784534689704 - 1.4648992581772973 * i,
				-1.4427493124100184 - 2.391725334754419 * i,
				18.898438167960094 + 69.61121648531747 * i,
				-29.868503282256945 - 41.330967392528365 * i,
				-1.283833365665796 + 62.50858072750488 * i,
			],
			[
				-43.64318909682555 + 4.8941608869612026 * i,
				-70.79640827922367 - 27.126700858136928 * i,
				76.14660539482074 - 26.556417050291213 * i,
				117.00275205054717 + 2.9988074366013677 * i,
				-89.95680320725006 - 4.92349460578572 * i,
				-38.53982332900939 + 8.819147741003874 * i,
				14.298897092634263 - 49.37224176808918 * i,
				1.5009142040404981 - 47.92325842975333 * i,
				-37.45050424154303 + 17.497304199946427 * i,
				-24.77739734280604 - 2.0680037278574446 * i,
				-4.890945632657038 - 8.642690530198706 * i,
				-2.6778042225101046 + 23.853545899984905 * i,
				-31.563681368220728 + 25.890824976859307 * i,
				82.97881151057727 - 52.495343732731484 * i,
				-78.07838315278896 - 64.13652405966148 * i,
				-3.764734044790657 - 0.806912565428334 * i,
				12.512790435857085 + 13.909287822962725 * i,
				14.757682613305434 - 4.131788085272191 * i,
				-9.579808839126052 + 14.799799358965785 * i,
				43.5107073892724 - 30.390794582492035 * i,
				31.021081848789947 + 23.123293543932505 * i,
				-61.510584339706725 + 15.314013470919633 * i,
				-2.632985102996713 + 58.398184920897876 * i,
				14.27833068051109 + 23.642372443191427 * i,
				1.9208925479959191 - 13.193333103522152 * i,
				19.654474686923294 - 8.863411913646086 * i,
				25.85257728446011 - 14.298151407009328 * i,
				44.17530413879953 + 8.852761485918613 * i,
				7.018896223789966 - 55.11837834082728 * i,
				-25.900038844840672 - 50.43966246407622 * i,
				18.751399694251397 + 18.4732825989075 * i,
				57.036848610807255 - 46.83577480112478 * i,
				-35.895257059978086 + 81.59738725969054 * i,
				50.05063227862899 + 31.43379854673976 * i,
				-49.709545002221375 + 56.91275670701566 * i,
				12.55122082164452 - 13.662157067565253 * i,
				11.322520086642877 + 11.185007198024138 * i,
				87.49311687729501 - 78.2510335700629 * i,
				3.0749400775656492 + 19.789535402317004 * i,
				25.405727401450527 - 43.35496574398705 * i,
			],
			[
				-21.9746382913192 + 57.260908020582626 * i,
				-17.309691012150665 + 12.033347232917805 * i,
				9.302187048813522 - 22.032737667484625 * i,
				41.27703570954847 - 15.269768039718656 * i,
				79.16259342881648 + 59.305130154682104 * i,
				12.030021233034802 + 30.275408551393852 * i,
				-29.49329022135327 - 23.150293214003028 * i,
				-1.5613387562800085 - 51.12885551287232 * i,
				32.28709805588023 - 8.267893194263635 * i,
				1.5326974594392624 + 25.485008886532345 * i,
				-31.531115590201352 + 1.1048156153813977 * i,
				-24.210884431168658 - 26.96850294936297 * i,
				-18.507559106972863 + 6.59168048372706 * i,
				45.59302134574375 - 18.673184981872915 * i,
				-4.728850072156561 + 52.81851607653809 * i,
				17.148937673984662 - 9.700143230108893 * i,
				-15.591427264035246 - 41.91135055423448 * i,
				31.872068745285944 - 11.453762703390211 * i,
				-26.23389655731808 + 13.788251384758752 * i,
				25.980219033276455 + 3.2178359208652694 * i,
				36.133781871039254 + 22.599669655770768 * i,
				25.65424771957772 - 21.303861385193812 * i,
				27.270843064780966 + 23.13581207323918 * i,
				-8.973459119113244 - 16.74435852466283 * i,
				-90.44462219143102 - 2.152481729896664 * i,
				-11.53284081399481 + 53.40695657482124 * i,
				15.206297927827634 - 17.628152336425824 * i,
				-21.02399786508812 - 57.249701116442225 * i,
				2.649307094830732 + 9.467735117892076 * i,
				-49.503862365300506 - 47.168394057030426 * i,
				-12.015603358243723 - 33.396844343088134 * i,
				18.39414823450135 - 41.585835797705315 * i,
				14.317689771468562 + 44.484576593516245 * i,
				-14.138077633424562 - 34.55412842873703 * i,
				-12.395602349365284 + 23.11224302339442 * i,
				3.00986844474005 - 15.720821426996626 * i,
				-29.014232795811523 + 13.029926354248124 * i,
				-16.609396608986565 - 33.929196400487555 * i,
				63.41696149772055 - 30.016370201881642 * i,
				63.685918806049585 - 78.81077025082564 * i,
			],
			[
				13.608444500419012 - 45.76495274688483 * i,
				-41.26154320891504 + 20.89351260456067 * i,
				28.744659927469826 - 30.276490227970584 * i,
				109.17450705211161 + 9.741052214387139 * i,
				-35.68371217685993 + 15.972356879562483 * i,
				-7.739434458645235 + 12.114479360097704 * i,
				12.244438081154442 - 27.66015252699981 * i,
				-47.036947278419916 - 32.19599788864667 * i,
				-20.993596205767965 - 28.23131609812873 * i,
				-27.866299080701477 - 8.732591004585032 * i,
				19.035967711486364 - 23.566426343753236 * i,
				-13.800952405284185 + 23.604638602885103 * i,
				-11.851645055485605 + 59.76733977060161 * i,
				0.46715283013466546 - 55.21581637106732 * i,
				-18.627696606462784 - 41.20922935409757 * i,
				17.310599071128593 - 0.33985114325957966 * i,
				6.814687434465926 + 39.68857868501219 * i,
				-15.757902299602193 - 64.95282935600267 * i,
				-5.570137850793948 - 11.85553741249852 * i,
				1.8061881999332599 + 56.31240269296606 * i,
				-3.55421490597584 + 0.10876458487277318 * i,
				-15.979450250131492 + 42.224411011579406 * i,
				-5.910038199762397 + 22.08375782983679 * i,
				24.591599582592863 + 27.08053472967371 * i,
				-17.235687223648295 - 10.544708467361545 * i,
				26.61567293928264 - 3.2643382310833005 * i,
				-13.892010132738001 + 12.911036413993642 * i,
				-1.6488123513608883 + 7.2326458557921995 * i,
				-23.242323343015727 + 1.366245700405031 * i,
				3.2023456756353212 + 16.85741765034464 * i,
				0.18005791157862205 + 61.568716163143876 * i,
				-7.7079706318874415 - 26.717641791924414 * i,
				-45.76931258932926 + 100.09816470138165 * i,
				21.955675175243623 + 10.607290156394264 * i,
				-11.469155865689004 + 82.44285377857035 * i,
				0.9312183950837323 - 14.131285742410636 * i,
				7.7328823679160195 + 24.139750519658513 * i,
				17.996981303712538 - 35.6488449160984 * i,
				29.55197580215438 - 3.805551540452804 * i,
				-12.058038313063589 + 1.448777541654494 * i,
			],
			[
				-4.229357059790402 - 72.40610078608012 * i,
				-34.88549134866386 - 39.6136074648432 * i,
				6.354505606078746 + 18.875461952177943 * i,
				6.310516500448216 + 77.96689028141947 * i,
				-57.11285308247869 - 36.203426612061264 * i,
				-16.30397020760644 - 36.15922683969603 * i,
				42.597814712927416 - 6.78957208895531 * i,
				-37.02481698863755 + 8.29718307926592 * i,
				-54.02015453074131 - 39.23995967145683 * i,
				-32.470830217559175 - 18.60328089542739 * i,
				26.764199002210887 + 9.935207147962728 * i,
				-4.858447743916884 + 27.923154611924218 * i,
				0.5608781893273687 + 16.601784372307336 * i,
				-10.244611861977994 + 26.000291436374347 * i,
				-37.73169629044152 - 125.98789619034028 * i,
				39.8949805530587 + 11.79707876126129 * i,
				6.8322001393378144 + 77.9909047044081 * i,
				-10.27191835518672 - 1.4454225348595884 * i,
				-10.489048129230667 - 45.4901792449904 * i,
				8.558616738254614 + 66.75117558788398 * i,
				-37.26003027395352 + 19.07117143213256 * i,
				15.21852443350083 + 1.4325203833028297 * i,
				-25.02174935064903 - 5.56159246552204 * i,
				12.372999867622504 + 2.042365614210084 * i,
				84.42998328429722 + 51.77527913042954 * i,
				34.55016677403128 - 4.263461496054845 * i,
				-19.6280078741828 + 39.75957851722526 * i,
				46.43923175167566 + 25.26809102236923 * i,
				-55.18303726343304 + 32.463364037020824 * i,
				8.283408611578992 - 8.75754976575652 * i,
				9.686374414822353 + 75.33196693028094 * i,
				-13.991635106126498 + 22.237863979224713 * i,
				-54.0481383072162 + 84.23408782656517 * i,
				-5.392914855133282 + 49.790006193704585 * i,
				-25.650381181372346 + 44.033757579525414 * i,
				14.068355218563383 - 0.14503501847040923 * i,
				-2.57211645758089 + 3.9505825965873598 * i,
				16.673010195517644 + 14.311615477701505 * i,
				14.250300059681733 + 6.570667755098676 * i,
				-42.74087002958286 + 53.00462825810864 * i,
			],
			[
				38.55008095330993 + 3.422082855243943 * i,
				37.84894311690596 - 23.67212938712936 * i,
				-23.716259966269035 + 29.520706481304103 * i,
				-61.852141425522944 + 11.839586995418017 * i,
				3.486480260015128 - 37.57825925370582 * i,
				34.632798245250285 - 31.832474253435407 * i,
				-17.32770338567167 - 3.466774325420765 * i,
				14.374307596494177 + 32.720501808485636 * i,
				20.145574493135342 - 11.492983491145708 * i,
				19.48220044250869 - 14.340239530641043 * i,
				19.720752634698897 + 29.133814011227845 * i,
				10.055146739763757 + 0.9612596320716378 * i,
				2.8181637703194298 + 0.15997885339936602 * i,
				-24.722403752341755 + 32.39008679553746 * i,
				44.12964617731052 + 9.425885288051902 * i,
				7.240175941627442 - 3.369081694912353 * i,
				-39.9010282098079 - 2.880520771661253 * i,
				10.560531969197157 + 4.979482511610939 * i,
				1.6817112577895585 - 0.17540256604016236 * i,
				-34.83074814309978 + 24.088100748063752 * i,
				2.4171317331423747 - 27.823632962510914 * i,
				12.702646954534003 - 43.48295218153633 * i,
				8.42586967428602 - 10.36001030976685 * i,
				-53.12428661451531 + 11.572261395176994 * i,
				17.816425664189346 + 52.156410859096326 * i,
				5.314810377721827 + 1.499206338839869 * i,
				-25.204658127159462 + 9.616917099150735 * i,
				-0.2535100714001146 + 12.951371150572683 * i,
				-6.964781734674965 - 22.606024885451802 * i,
				15.783531986942975 + 0.3608251821098847 * i,
				-3.665721051799892 - 8.201625463800085 * i,
				-24.327412225713992 + 24.990881935322506 * i,
				-14.455104013715886 - 31.32973970226689 * i,
				-55.77297177520783 - 23.699310697874495 * i,
				-12.914190731428036 - 51.12780232633092 * i,
				12.551513287940423 + 33.29795397467205 * i,
				11.83153540032891 + 17.99329381619981 * i,
				-9.569150814440363 + 43.7625574548146 * i,
				-50.75047546395437 + 8.52808075903065 * i,
				-39.263097882048854 - 8.91891935562206 * i,
			],
			[
				0.74107227080097 - 39.69605169916656 * i,
				-21.105886155923088 - 22.354692172575483 * i,
				5.69516820748089 + 47.49315244193902 * i,
				11.848155693941607 + 69.63142984694267 * i,
				-79.992334352585 - 11.897737339735954 * i,
				-94.21865918691569 - 4.856865904039495 * i,
				8.912888798393084 + 56.27371990342825 * i,
				39.0687781422352 + 1.843997612689833 * i,
				21.736221807853255 + 15.976431572481033 * i,
				-13.605125835101951 - 6.7768652767821465 * i,
				39.4691609346836 - 25.91127345661309 * i,
				1.6285971326651634 + 17.434398075699008 * i,
				46.85180798009261 - 18.540360823621654 * i,
				34.573312209814134 + 10.450299455133774 * i,
				-24.876170859727843 + 18.03442170793224 * i,
				-20.122870921753844 + 17.38392267323141 * i,
				10.70990530681648 + 14.283793518394805 * i,
				37.334949207616376 - 9.319713575293719 * i,
				5.900542635256677 + 14.668441637967163 * i,
				-42.460090219154466 + 5.547680302482917 * i,
				0.6874148452607571 - 27.434611763470322 * i,
				-26.1708588598 - 39.66681136299043 * i,
				-4.8347421201724075 + 40.34975431738896 * i,
				25.507773205798287 + 13.405099700824692 * i,
				61.12065964429675 - 27.921024799970496 * i,
				-6.148363706136507 - 85.16090754838733 * i,
				-9.539908970777859 + 4.748628824784735 * i,
				28.060561450651704 + 34.54835310543215 * i,
				-26.27223583252392 + 1.4702289228079266 * i,
				-12.18864528705086 + 34.652024431694855 * i,
				23.63814580699487 - 26.935501102750727 * i,
				13.24583397097598 - 1.6139275255802241 * i,
				-10.701207790595447 + 19.278777231584325 * i,
				13.127004140763432 + 71.78059863868138 * i,
				-42.215599192487545 + 29.781044424010723 * i,
				38.81905365290007 - 67.2980105608856 * i,
				-34.97755510690054 - 23.854562156086985 * i,
				12.34718345702866 + 28.51755491883042 * i,
				-33.83751330040761 + 88.3220396100055 * i,
				9.276957999512465 + 59.340980877805904 * i,
			],
			[
				64.69871892727258 - 10.55934518885931 * i,
				-2.385539574654814 + 56.6338616877193 * i,
				-10.16636397821938 - 0.8397654077265173 * i,
				-44.81694505635235 + 11.247800285358224 * i,
				-14.134343674055799 - 30.451481152436422 * i,
				-44.66260502553216 - 28.251576812010967 * i,
				-18.095626496390892 + 73.55836609481395 * i,
				-7.622819050993485 + 63.726879249818325 * i,
				-17.418927594379795 + 45.626129472146744 * i,
				30.06706025259095 + 2.0123034454551316 * i,
				26.854521181815585 - 6.424969619013531 * i,
				-30.727799938990472 + 23.055949819160986 * i,
				45.948739868288385 - 3.2441109671126753 * i,
				-23.360486910505813 - 8.705734864451486 * i,
				-21.21526941714727 + 41.57712521317042 * i,
				-4.711659134894024 - 53.119522984399396 * i,
				25.77530119051491 - 8.208406053679795 * i,
				22.473171773361983 + 23.297057847781407 * i,
				11.97439476956003 + 10.191278587948801 * i,
				-13.51054049141728 - 34.816211074326375 * i,
				26.206414834379146 - 13.315833273007598 * i,
				-8.218249093720246 + 1.1806516630911261 * i,
				-22.212962701687665 - 12.782611222830532 * i,
				21.755260203006557 - 2.6042129157440606 * i,
				32.074452649121575 - 88.70637795375612 * i,
				23.51588739514236 - 20.263174186447145 * i,
				22.08638078162612 - 1.3590487271081315 * i,
				-15.033726330546646 + 25.355467066703973 * i,
				29.31229536319725 + 14.46221862647856 * i,
				-46.0728001234772 + 80.8964095854964 * i,
				-28.33465300132268 + 13.602772971926125 * i,
				22.56460034032588 + 0.9675541437146933 * i,
				26.8195885253419 - 22.568512489752038 * i,
				5.442022147793708 + 80.44128692953169 * i,
				-10.97237343800677 + 17.536094021607184 * i,
				3.168184673910627 - 20.78563517316433 * i,
				24.511301729733297 - 67.6030568986513 * i,
				-63.67613748493606 + 37.59845008959193 * i,
				-21.639069165162574 - 19.586428472358502 * i,
				10.295062506841418 + 100.17067997847857 * i,
			],
			[
				23.250881427485325 - 67.29323614838148 * i,
				9.92750770148966 - 11.831423364926742 * i,
				14.540087007481375 + 24.142781334784075 * i,
				-2.3207814217325904 + 38.1560581615922 * i,
				-33.657739284937676 - 75.15696036836445 * i,
				-1.6041432313123494 - 9.766415922490978 * i,
				-40.62297447592118 - 8.149236897239206 * i,
				7.80886539863593 + 31.566537092861893 * i,
				-54.29591395226455 - 42.79341096287732 * i,
				-10.296889808510343 - 6.517332892984413 * i,
				49.78347923975511 + 11.966446846780144 * i,
				-20.32452564704883 + 31.98316907161535 * i,
				-32.25662915549705 + 25.37282280191431 * i,
				-50.31927949097101 + 73.73505998713645 * i,
				-6.663281930524249 - 54.806314451998226 * i,
				21.753802510052402 - 15.754076897390576 * i,
				6.367839510107132 + 47.30901717134401 * i,
				-0.4177040938928869 - 1.278845438954761 * i,
				12.741345685722763 - 1.3930383302903806 * i,
				66.8995135967796 + 24.795098474261522 * i,
				7.638308607108485 - 1.793378272540643 * i,
				-89.53787188757589 + 6.1311798312565315 * i,
				-26.084030489451962 - 87.87822154620747 * i,
				-15.408301223404706 + 30.744253951343484 * i,
				33.93105103910251 + 33.3184100975244 * i,
				83.12197217256276 + 13.729811861686073 * i,
				4.829982369231587 + 75.77362584366404 * i,
				22.052780581379437 + 64.87583072877513 * i,
				-2.531656428950022 - 1.0708413435485085 * i,
				18.058316724706458 + 21.744816784762 * i,
				29.54915662250555 + 75.70323123452431 * i,
				0.9042867595537771 + 71.17128141212594 * i,
				32.059941777819695 - 1.6527211823988637 * i,
				-29.211968937731953 + 43.2076847068635 * i,
				-8.939727868023354 - 4.6183507181268135 * i,
				-1.982683145067388 + 48.388189241771315 * i,
				-7.916333072524349 - 17.465021232109592 * i,
				3.9651739569142777 + 11.862567440833864 * i,
				-25.31998272039262 + 18.6405878856228 * i,
				-43.698552191671006 + 88.49620362901246 * i,
			],
			[
				-25.148635632839017 - 46.671880516041014 * i,
				21.620272524940564 + 58.016511711873505 * i,
				-13.287827757421681 - 90.23693838983749 * i,
				71.44528966509196 - 32.75794401974472 * i,
				-52.341182345186525 + 10.034350223954789 * i,
				3.4541031322372646 + 50.2718915131812 * i,
				91.48756692870069 + 8.048429001109497 * i,
				7.589317030547058 + 26.95905628184977 * i,
				4.183404641187337 + 9.369095897385236 * i,
				-32.231398886480434 - 38.29308830781305 * i,
				-42.05364966561765 - 54.52465546122146 * i,
				31.90848672054338 + 18.648640369411055 * i,
				53.97119469433545 - 28.713856790807707 * i,
				-45.55590731221844 - 21.69478705981112 * i,
				0.9868323144423599 - 29.01453704299577 * i,
				-30.956042502373947 - 16.09528767169473 * i,
				37.60197336274782 + 2.1802391100331704 * i,
				22.04902041576556 - 46.13321977377893 * i,
				-6.396035675433595 - 0.8538734823428058 * i,
				-57.19002843478512 + 22.501186662458352 * i,
				-29.40466948348333 - 7.918041073056116 * i,
				11.436040449638881 + 23.166959914327663 * i,
				17.070257884040316 + 56.58577112314836 * i,
				-17.312603911913556 + 10.01119210097383 * i,
				19.122160282958482 - 19.826180147521484 * i,
				-61.663052880689435 - 57.59817804689294 * i,
				22.855470650211146 - 44.87786477178476 * i,
				54.94567942590961 - 25.591470279401676 * i,
				27.210951430387837 - 33.26986167536724 * i,
				77.30551925795852 + 59.843218170153676 * i,
				-38.956956209261755 + 30.549939985829315 * i,
				-34.59337055799069 - 31.472782262895308 * i,
				-41.36340112580628 - 15.993311717791457 * i,
				85.19715534907768 - 42.84037089951667 * i,
				55.79185406300712 + 1.6383780335200981 * i,
				-70.67741562430412 + 47.557554188498294 * i,
				-8.461916260151327 - 52.70993867733971 * i,
				78.14168991493341 - 34.62208622059792 * i,
				-5.407203176949589 + 78.85611004764357 * i,
				29.3620459671619 + 16.829495044486976 * i,
			],
			[
				43.8309900721614 - 32.28830108746635 * i,
				1.4054886639338378 - 16.75440347517599 * i,
				28.46739835999591 + 4.785877013880452 * i,
				-23.701707836498016 + 16.544398810150142 * i,
				9.907494985570853 - 75.09449880123869 * i,
				-10.550245158893517 - 8.822963984511325 * i,
				17.774655256224108 + 31.14684567474677 * i,
				0.38410843188066224 + 21.68825071951349 * i,
				-10.969906152355776 - 3.828704543262461 * i,
				-15.548663546707227 - 26.37097595545353 * i,
				-12.152236210773722 + 9.383230886804043 * i,
				-17.97550826060572 + 29.311870271069978 * i,
				-16.64393505108056 - 10.66664098008583 * i,
				-9.890584437053532 + 8.278761457209956 * i,
				2.34885580040217 - 64.76232494782768 * i,
				7.615229863943853 - 46.10678321901108 * i,
				-15.369884621514188 + 18.23347697019439 * i,
				-26.37653510611463 + 6.402558049376296 * i,
				-10.332763844143813 - 5.942523082096855 * i,
				48.26436525788195 + 8.975777400545498 * i,
				4.962260887713775 + 6.832281763651686 * i,
				-24.84852973118221 - 30.38757795007578 * i,
				-10.112016631828908 - 32.440374236295895 * i,
				-0.9535450299813704 - 23.04469733169421 * i,
				-2.1294159059968436 + 22.515328896761503 * i,
				47.31432391934753 + 22.40741556750401 * i,
				-6.116521731860875 + 1.504399754658408 * i,
				1.0263408323507974 + 26.40647186660876 * i,
				29.621941833791908 - 4.340229947841436 * i,
				-0.3049952988455402 + 47.055043197807315 * i,
				-45.29802217492767 + 48.53351479206309 * i,
				10.19341151998363 - 9.832453988463811 * i,
				11.99442751319512 + 18.698928127474872 * i,
				-3.200648552059306 + 54.93589101832884 * i,
				-13.611839122405307 + 18.23652800343084 * i,
				-10.677899458277658 + 10.693089317751465 * i,
				-11.731578674391983 - 4.142512308041219 * i,
				23.539310224324083 + 39.66302237625199 * i,
				-14.791440787905394 - 56.982039508505025 * i,
				-7.152129172707706 - 1.3240191468107105 * i,
			],
			[
				-29.743030369168398 - 33.62686114495567 * i,
				-13.501120323241139 - 54.04504109164331 * i,
				65.9497031491571 + 20.1959490659042 * i,
				-12.44338458721623 + 48.02040454899752 * i,
				-43.5638284467521 - 31.72389775947994 * i,
				-61.36044709816238 + 8.541775583581767 * i,
				-21.622832977032886 + 15.344645517958043 * i,
				7.91389823206373 - 9.537058597796204 * i,
				-5.867409619195058 + 14.538649378659848 * i,
				-10.303231208285993 - 6.338590914468426 * i,
				24.208480137209477 - 18.21545857284393 * i,
				-15.861600078519205 + 3.281840893581901 * i,
				11.634879159826653 - 3.7623217473663537 * i,
				30.646307467161563 + 43.061308400725316 * i,
				-53.26512382347469 - 39.88939834688358 * i,
				4.418177529500127 + 31.274451940672872 * i,
				27.649278672057505 + 30.81234137382603 * i,
				6.005601687507379 + 25.22775303591638 * i,
				-5.145529833333185 - 3.1224578140768102 * i,
				36.50038526073098 - 13.080934247408749 * i,
				-2.7559892160851405 + 17.65537209364598 * i,
				-40.42183526951577 + 8.386773117258926 * i,
				-30.904094239037406 - 31.63746962027396 * i,
				35.82878556823447 - 19.241794802339502 * i,
				55.49487459675273 - 15.46412397726283 * i,
				-7.605361830298023 - 24.95191165347302 * i,
				23.62778178327948 + 49.060723278913834 * i,
				40.33313748204157 + 39.984177366665094 * i,
				11.637202160554423 + 36.78573584952643 * i,
				-10.234176063346023 - 34.62974139508268 * i,
				19.16074707957131 - 19.406398616894865 * i,
				50.428170657505106 + 18.189232348795976 * i,
				25.26911901858297 + 22.665164413146776 * i,
				-7.364726194981937 + 88.61721208896164 * i,
				-13.028935880010062 + 10.058291062358553 * i,
				28.279561096437206 - 75.34215103069343 * i,
				-24.06959635654036 - 26.44171511355713 * i,
				50.8875023940169 + 17.183248870993285 * i,
				4.164754865667793 + 29.110342715172937 * i,
				-9.901409792694635 + 40.49047460881421 * i,
			],
			[
				-28.786599692820516 + 19.03303886789044 * i,
				-7.232906370810639 - 50.051309975800564 * i,
				-2.894044369437439 + 49.877689261891646 * i,
				-5.096941664239068 - 26.130236394196295 * i,
				-53.574186343995734 + 35.79703999362177 * i,
				-7.114141153509818 + 23.48652694259613 * i,
				-14.233055521500063 - 1.3300805687087234 * i,
				11.461795987952712 - 3.3285247703778396 * i,
				-22.084655471338685 + 1.4649092675726827 * i,
				-25.928222233471978 + 12.997567898590098 * i,
				26.048585834230693 + 14.510058777454162 * i,
				7.583241324009478 + 0.8449861128674314 * i,
				3.173041206453661 + 32.56223000370389 * i,
				11.398422993010456 - 5.767587888063133 * i,
				-35.10514947091053 + 27.758357725875783 * i,
				30.906098112931893 + 65.95804670159836 * i,
				56.97069315357777 + 7.934354105605372 * i,
				-22.931469406895474 + 2.7222508163698933 * i,
				-6.02205222389129 + 9.621605467569978 * i,
				-18.363576579050584 + 2.660887727249772 * i,
				-40.52558022897875 + 16.465927167691646 * i,
				-0.8202287126448482 + 31.59851165307733 * i,
				3.886591259379763 + 6.523383053098314 * i,
				10.05000890992369 - 25.07371351281985 * i,
				39.16293631067013 - 12.580838729115998 * i,
				-14.330754011923183 - 46.12548475984953 * i,
				-32.22815494075138 + 23.53559016795667 * i,
				49.585262922746494 - 12.107887302261567 * i,
				-38.10021778178597 + 30.94456751055798 * i,
				-26.540578350199382 - 57.47824971538415 * i,
				49.396713225830055 - 12.050039153177252 * i,
				21.50181848618686 + 5.304881994480279 * i,
				43.89934368843253 + 46.18226122719289 * i,
				0.4605455580632878 + 17.77087131762816 * i,
				-9.361319933100575 - 21.66546967574018 * i,
				64.71738124979568 - 45.50180284405613 * i,
				-52.21985415156074 + 75.01595493685983 * i,
				-28.676292187324492 - 14.070474297969996 * i,
				21.52145803803683 - 3.5380647746314753 * i,
				2.3236308358803393 - 1.2752750969860969 * i,
			],
			[
				-22.05811146786081 + 10.322103864300573 * i,
				-58.60829654669651 + 15.988998635151976 * i,
				39.983241701820106 - 21.212297603039023 * i,
				15.29614931181285 + 26.85677643373583 * i,
				10.163582852051704 + 81.40115032296558 * i,
				-37.13194012860623 - 10.234612828699966 * i,
				21.324581114621296 + 26.97386695697901 * i,
				18.115593129499246 - 35.80512803638272 * i,
				6.771385007596848 + 44.690334114079434 * i,
				-22.916906360723132 + 28.319903395522196 * i,
				-12.835625128513316 - 47.99548341302511 * i,
				35.964444374896104 - 14.65189603300958 * i,
				22.59120189701204 - 6.05500805053839 * i,
				71.28786205787736 - 6.6772007272599865 * i,
				-48.586192377936314 + 36.69865170428035 * i,
				1.8182753358048842 - 47.39203398923351 * i,
				44.50760328448799 - 14.565771495263974 * i,
				2.813990376938875 - 9.79655042379838 * i,
				-34.08393595968741 - 6.246330080355392 * i,
				-5.506518230794541 + 27.137593329004147 * i,
				4.118356178232968 + 19.622709389214705 * i,
				47.442340433941595 - 3.2298817526522043 * i,
				-10.955087829107214 + 42.28395376031422 * i,
				9.67151680848049 + 1.5785432728099806 * i,
				44.93457958986092 - 67.40637566791199 * i,
				-51.13073451984209 - 7.1649243700716845 * i,
				47.34038658401961 - 23.874868113948 * i,
				-16.264196472191216 - 78.38398973972768 * i,
				11.662046278941894 + 57.395096871028656 * i,
				-34.39948233716019 + 26.11069227212529 * i,
				7.12732219172705 - 63.142406925341206 * i,
				-4.077536856077472 - 27.350347029473443 * i,
				25.848292525761792 - 5.809902479393069 * i,
				15.599470329948872 + 11.519771451394647 * i,
				-3.2122747602321637 + 32.02187361890526 * i,
				3.1821265861368744 - 62.295278523964086 * i,
				10.899120821151953 + 7.833443333241153 * i,
				26.676558087606367 - 19.45701284307566 * i,
				34.07231800659871 + 6.776407524409958 * i,
				22.8478373822427 + 9.702641613220981 * i,
			],
			[
				39.821808107810206 + 40.25417861483771 * i,
				4.719091632314985 - 6.028737399163788 * i,
				-39.459098832526195 + 93.29897870135079 * i,
				-52.08995179313288 + 14.015366869663808 * i,
				76.63262286220214 - 71.66941042066142 * i,
				-9.680960230696712 - 41.42554753198476 * i,
				-18.60217967579888 + 53.815080183568455 * i,
				-32.59963331485443 + 50.38306563966976 * i,
				-24.184468571501483 - 15.62831990753217 * i,
				6.953108357112768 + 7.791306044165481 * i,
				-0.147641147154971 + 23.652985441914552 * i,
				-36.904071483665874 - 17.65761540232404 * i,
				23.3352026967832 - 33.75587513221486 * i,
				-43.90477639663948 + 13.393952512933055 * i,
				60.771172380321545 - 26.582057242731906 * i,
				45.240360719005196 + 4.570404999127582 * i,
				-22.228561214105934 + 2.191925232706069 * i,
				-20.587063265349997 + 35.882766615457285 * i,
				-2.8879026706118367 - 12.241984127998066 * i,
				-12.74023788680644 - 16.897954935736976 * i,
				-10.943596180886605 + 19.94065543078213 * i,
				12.303455994475613 - 78.21266552746654 * i,
				24.917149503867734 - 47.403388655113986 * i,
				12.100960711189241 - 18.360708221490682 * i,
				22.839678140872294 + 102.4886536912054 * i,
				41.067108999959785 + 14.96652304921897 * i,
				-28.507973627226782 + 26.601444777767178 * i,
				-54.68669865068979 + 40.31121318461764 * i,
				12.969269946732325 + 86.37969583372083 * i,
				-50.29995422956982 + 21.276028582394474 * i,
				-18.371085778819776 - 13.65883465526843 * i,
				-11.712400020261654 + 47.68886141297147 * i,
				32.09512113017917 - 17.116661519978088 * i,
				-64.74044848605959 + 10.787368985663019 * i,
				6.908646184768928 - 18.73073440538082 * i,
				3.6416547825046024 - 11.562462020211633 * i,
				2.203765024256633 + 16.407248769676077 * i,
				-44.32467396945714 + 77.73143148716287 * i,
				-37.276968783689824 - 55.928949686080514 * i,
				-63.82182132142299 - 2.813689034683429 * i,
			],
			[
				-8.807632732456298 + 34.06584218661163 * i,
				-54.032792939596675 - 3.906886938884483 * i,
				-27.992074356877055 + 34.3223221306515 * i,
				-63.34503939132198 - 63.62654380381349 * i,
				25.191238958743526 + 65.06751534530355 * i,
				-34.2088080510605 - 17.45112775547159 * i,
				-74.17465664237972 + 23.52414651711055 * i,
				-10.834646927951098 - 17.48670656326017 * i,
				-11.358295963164153 - 5.448468696158841 * i,
				-62.924814421053696 + 47.44690103935986 * i,
				-2.6276782967284333 + 21.059168829157702 * i,
				12.338306134554546 - 47.8174193515096 * i,
				28.771019903184822 + 29.284940656478298 * i,
				-19.218460741855708 - 39.41688169121509 * i,
				-52.987074685769954 + 19.695782435896774 * i,
				20.376350589577275 + 16.834715691535404 * i,
				50.054302839157415 + 25.95270089428555 * i,
				-49.37030200418019 + 52.60857650544787 * i,
				-44.322619223647195 + 11.944844331748644 * i,
				20.22064897474118 + 5.9618389309210755 * i,
				-30.475278635556958 + 50.63469526547405 * i,
				17.85509955834341 - 5.347173580551271 * i,
				69.53921703179228 - 29.03358594163069 * i,
				55.328731798037566 - 27.822710240527314 * i,
				14.407092703517407 + 10.652425133074004 * i,
				-4.85445638247646 + 2.6705875310467064 * i,
				-64.2214615134073 + 21.64486489546306 * i,
				-24.991807360173144 - 36.374786589054274 * i,
				-78.5179605835208 + 92.9013247091658 * i,
				-56.643873665344444 - 27.04720237276733 * i,
				14.455968070561255 - 10.76402671513467 * i,
				-9.216446716267868 + 1.392177628889499 * i,
				94.79943251271632 + 67.96879599403152 * i,
				-36.97786153528395 - 0.16413839084363246 * i,
				5.511467636676315 + 45.89805245827924 * i,
				116.11410414746886 - 64.03551118589803 * i,
				1.4705945155181066 + 50.63132289252928 * i,
				-74.18320398134583 - 55.55009048710702 * i,
				2.2927410771323373 - 41.83928808286546 * i,
				-72.85050697001863 - 29.32614666940412 * i,
			],
			[
				-44.55765539684779 - 45.0345790094857 * i,
				-34.96064692345531 + 83.0582040106521 * i,
				34.67921938632702 - 75.11985084950085 * i,
				87.71906357877862 - 39.62776525091138 * i,
				-75.46658888938273 + 111.2451945359199 * i,
				-17.448759886948324 + 58.1080672177614 * i,
				65.3352358111079 + 7.401427659318642 * i,
				5.139318460948829 - 32.91065164391199 * i,
				7.213336317080234 + 22.677080710236723 * i,
				-11.820327048087023 - 0.7963373866151358 * i,
				-17.679622132411282 - 24.226662127429037 * i,
				60.82211307736901 + 6.889255275886455 * i,
				30.968237442506876 + 18.668559333344263 * i,
				11.901168646172376 - 79.05402462061276 * i,
				-31.789031137553593 + 34.13531971183797 * i,
				-7.318829486543848 + 0.06150297743789679 * i,
				44.84884084020503 + 25.11057244849738 * i,
				33.95488070080373 - 28.365149846209974 * i,
				-17.24530488414021 - 15.841914317439205 * i,
				-78.04408363684973 - 8.893897964599986 * i,
				-59.765046584873645 - 17.6559135494973 * i,
				-11.787283554335659 + 68.39921692619455 * i,
				35.546462096116144 + 73.83391291719602 * i,
				7.896536540073413 - 3.983401859132023 * i,
				25.413215406637704 - 16.68976653089966 * i,
				-69.95669478818249 - 89.75355581485896 * i,
				1.2921726856197324 - 42.12872890961883 * i,
				58.354356586468924 - 69.57302479495513 * i,
				-52.46540790137006 + 14.118357062431834 * i,
				6.331074904207981 + 51.206548090112136 * i,
				-11.230507287039458 + 6.373502127487981 * i,
				-36.061219871966166 - 56.93586831069358 * i,
				37.43942156691017 + 4.210900015037073 * i,
				112.66543238805511 - 33.798368255509665 * i,
				62.867253609296355 + 67.11540989752197 * i,
				-10.65906745812776 - 9.223066132773369 * i,
				-9.370474217183414 + 4.668422527895203 * i,
				43.429576629434386 - 38.93785785400394 * i,
				9.375817628274024 + 58.83538000703049 * i,
				32.34543509336801 - 34.073740751308 * i,
			],
			[
				-6.984590491324269 + 8.758776654722965 * i,
				-22.06220484592035 - 37.04934323603416 * i,
				32.77787009920452 + 56.094804584732096 * i,
				6.653941342362195 - 36.29579185865835 * i,
				12.061999369899965 - 45.67380426134889 * i,
				-42.82928895007572 - 11.911134055394683 * i,
				-45.26686475491034 - 41.338309946878546 * i,
				-45.751575395582364 + 0.06991921922335109 * i,
				-45.846263731398395 - 4.23675902579032 * i,
				-3.0206613425541065 + 36.2900847599396 * i,
				23.140239569906576 + 18.61636914038317 * i,
				3.4528400272032798 + 14.589158996486603 * i,
				-34.864814660958984 + 42.6202753387153 * i,
				16.10009694038547 - 24.28373184037993 * i,
				-37.95890787831246 + 16.444062973045618 * i,
				-31.28009106924273 + 21.868410692187652 * i,
				33.05918408266979 + 23.61909268191107 * i,
				-52.58184322744853 + 26.285666808693747 * i,
				-13.806895299883493 + 24.148204233079355 * i,
				59.63160324860385 - 20.047439561486776 * i,
				23.870606905013076 + 9.764778526598192 * i,
				-39.45730130812866 - 19.386882894442714 * i,
				-6.256220759424934 - 22.05096731977251 * i,
				8.872139007560314 - 20.99739885117623 * i,
				15.280975314800912 - 18.570260572058572 * i,
				25.707304945767703 - 4.175172923408425 * i,
				-19.40750218854285 + 32.79854077268282 * i,
				-33.437318506558285 + 46.18860683533995 * i,
				-3.3744250625918255 + 26.272332756046076 * i,
				1.5252501775473206 - 21.75632161998466 * i,
				59.12116099179899 + 3.7382652768328057 * i,
				1.9659642635401298 + 33.29505063157714 * i,
				73.07967560641104 + 46.5115394438975 * i,
				-19.48671313641543 + 16.673542547263796 * i,
				24.825757285744857 + 1.1957801485381907 * i,
				38.10617356599211 - 49.948597837591265 * i,
				-3.2645527899626003 + 52.8455152758648 * i,
				-12.429107003562507 - 13.921472857559506 * i,
				-23.62098568298314 - 60.84150287541546 * i,
				-36.98666280017338 - 9.03800318607365 * i,
			],
			[
				21.22944416913873 - 54.3343174373922 * i,
				-24.92210624258982 + 9.860655891685553 * i,
				19.56478820896814 - 15.932193039324432 * i,
				13.233317592128117 + 53.32377328782508 * i,
				-70.38967847631676 - 40.20220814808256 * i,
				-3.460431822197563 + 0.24182847906450888 * i,
				41.87338612635795 + 30.56519301984305 * i,
				23.092098116119715 + 51.10409662775635 * i,
				-39.16562189593728 - 3.301260381078208 * i,
				-12.263254120916073 - 11.552704389434979 * i,
				21.71763305306949 - 10.690744989889396 * i,
				15.453241624126207 + 24.229902013928225 * i,
				16.91157623713446 - 10.37662051078803 * i,
				-16.846346543255457 + 29.06047491302828 * i,
				-14.939639826136101 - 51.14073209374253 * i,
				21.314054353922504 - 30.512024624470676 * i,
				19.85791461542749 + 45.112965896899894 * i,
				20.09381583520269 + 15.486667005419406 * i,
				-21.504194047322105 - 9.926874558124126 * i,
				-21.838780256485048 + 26.964831760128448 * i,
				-29.087395030666613 - 1.9907905213771642 * i,
				-20.4181001382494 - 3.3740557335838766 * i,
				-38.754337078294014 - 5.044614159945737 * i,
				-28.68953721352939 + 9.72593748923633 * i,
				121.56741494781843 - 3.4142404021150554 * i,
				22.9218987326082 - 42.43510890083742 * i,
				24.914241044344777 + 11.573648848097122 * i,
				38.0954853601416 + 13.782624868130739 * i,
				45.594934321002896 + 15.456968069030722 * i,
				13.41441086116346 + 52.85902426609979 * i,
				-11.128628685792542 + 20.88579775658327 * i,
				-6.14440989391872 + 25.527512634133927 * i,
				-6.056187732642801 - 40.949587735973736 * i,
				22.99544025216779 + 44.39629387722769 * i,
				6.469815634852608 - 5.199061270886688 * i,
				-32.109232533021014 + 23.77788589070608 * i,
				30.387000020414057 + 8.89216059533377 * i,
				37.67826389392566 + 18.740840554094362 * i,
				-54.417282691297615 + 37.97367131190644 * i,
				-40.66471205602454 + 82.69853029260582 * i,
			],
			[
				25.98126112294122 - 47.55171610902498 * i,
				11.861955766073706 + 39.49863632644664 * i,
				-22.37607725670071 - 13.249726562217177 * i,
				-13.150211141704773 - 54.06299467863803 * i,
				-56.69124849008681 + 14.745410991538137 * i,
				6.03845794025591 + 14.10495659551853 * i,
				2.5629668106314796 + 1.9953671268215771 * i,
				53.49308738029006 + 81.67035600936839 * i,
				-28.800824367185523 - 14.873258997140589 * i,
				-16.895753526320068 - 29.75932907513507 * i,
				46.77619922741928 - 4.995193725202007 * i,
				47.36733587326832 + 6.025053736388877 * i,
				24.57249955035229 + 49.651187454414455 * i,
				-84.6842221004784 - 4.44317117863784 * i,
				8.928248479805987 - 17.840350429667712 * i,
				-8.250073953519134 + 10.223728575620228 * i,
				58.39552112564887 + 82.25505810946612 * i,
				8.65650218134958 + 14.744544674996234 * i,
				-40.839098986532505 - 11.518551617438803 * i,
				-68.18020383581052 + 64.38680418881995 * i,
				-79.75364986344064 - 1.9351986460899013 * i,
				7.99747051874337 + 45.77558835276368 * i,
				15.157383721004873 - 0.6896994658369913 * i,
				-9.428917639628452 + 5.000132827952618 * i,
				113.37731964850113 - 34.36155677001043 * i,
				6.89700908872959 - 78.04102966073404 * i,
				-15.723486413505164 - 0.40020214221516426 * i,
				64.15365376729919 + 6.680324019289701 * i,
				-15.797488204670074 + 25.10440995769308 * i,
				24.071164437539466 + 40.542019477309395 * i,
				21.930959053064075 - 5.1764728769480115 * i,
				-18.81830468011784 + 28.942174652390356 * i,
				24.147914678906382 - 18.080993521980673 * i,
				21.955417413132874 - 13.391777295598555 * i,
				45.93695790223202 - 14.152634938499745 * i,
				48.94423221023273 + 62.84270654883151 * i,
				30.923665165110215 + 16.58196425584319 * i,
				-17.831088630216698 + 4.8949832075287745 * i,
				-60.74408176236093 + 37.373504895779746 * i,
				-44.39531772699465 + 55.91350214252248 * i,
			],
			[
				-52.874527419135745 - 95.28523239832946 * i,
				-15.309177101343522 - 1.3022153279628057 * i,
				14.781157533583155 - 47.80690062724453 * i,
				88.01937858412936 + 17.30097937091577 * i,
				-92.78047585626888 - 38.040867564265454 * i,
				-33.425102288024505 - 15.667607808696232 * i,
				18.845525858758634 - 38.80769030346539 * i,
				-3.95649414264696 + 13.947841833135044 * i,
				32.9226418181278 + 2.476372525240251 * i,
				-0.9016466411563542 - 40.456098285327315 * i,
				-16.393621748798907 + 12.02097251125457 * i,
				30.473604782590684 + 18.420196201994656 * i,
				-16.901408405180177 + 16.37352317099061 * i,
				31.657217426853762 - 5.6950056597027015 * i,
				-48.12191770436852 - 72.84711942682745 * i,
				-24.916983163307435 + 6.165357845409233 * i,
				-14.465722031046113 + 49.848001716909565 * i,
				40.218536430389 - 34.26016686260573 * i,
				-23.90247218763028 - 9.754000038327227 * i,
				4.863691932616803 + 2.2873559531149112 * i,
				-3.4474956987606467 - 25.473313950698667 * i,
				-52.78792794666442 - 10.846150675882502 * i,
				-10.150153023536578 + 45.11988981913923 * i,
				-11.740174869338574 + 32.27817900224277 * i,
				2.324871436195626 + 51.290050120781956 * i,
				11.574559207187576 - 42.75837339896126 * i,
				-1.2162757851173467 - 36.03885069823681 * i,
				83.21946770131926 + 27.48470891082792 * i,
				-43.21319532388348 - 97.87621131567056 * i,
				46.09317260258544 - 8.436030761937104 * i,
				-11.810682604374328 + 66.7093995952209 * i,
				11.305719733591056 - 10.165970662086599 * i,
				-41.43979703028907 + 68.30313235036104 * i,
				58.8586394719013 + 3.948784356374265 * i,
				9.191522210983134 + 27.741943070538287 * i,
				3.768534210166795 + 51.078194939918774 * i,
				-42.92661494839656 - 50.73665124797652 * i,
				103.32563012714638 - 21.469534116248465 * i,
				-20.929220831723804 + 63.85738265621242 * i,
				24.089613206218644 - 9.332717885203678 * i,
			],
			[
				22.92415314743156 + 74.12387242368064 * i,
				17.081097391145477 + 46.368428104057784 * i,
				-4.220748837777094 - 26.461644565223594 * i,
				-35.82773649967575 - 82.66282708927406 * i,
				36.18984375445058 + 31.365232399929077 * i,
				34.98864032252479 + 20.155905631203723 * i,
				-36.374582581133495 - 13.777721744769224 * i,
				-46.72322092055569 + 22.92317899962309 * i,
				8.78832347337044 + 33.68171812870858 * i,
				46.816209450003754 + 16.456236108310904 * i,
				-20.657369311495657 + 38.889976356577165 * i,
				-13.499874753790607 - 7.7431557040046926 * i,
				8.791542689610978 - 35.52844995256413 * i,
				-36.352673670933555 - 65.03899048068071 * i,
				-2.2215199038992637 + 56.49743593119183 * i,
				26.585793984991362 - 25.207940882647137 * i,
				-33.90589762752762 - 57.63934190650572 * i,
				-15.254066135839622 + 5.795936981980631 * i,
				11.293236153755913 + 24.643817210107272 * i,
				6.866951865994119 - 39.035746574013025 * i,
				45.97643740473197 - 2.1978343774818256 * i,
				6.255749199767038 + 18.60380461360947 * i,
				37.04922725781462 - 39.013154506413464 * i,
				10.122808019663747 - 28.42146226072643 * i,
				-31.26618784805686 - 0.5202916658965293 * i,
				-14.393559697639075 + 57.64711088441915 * i,
				-15.828554830078943 - 13.337958093277692 * i,
				-41.275407916540715 - 29.66416068183475 * i,
				32.412991423173956 + 5.507319562097379 * i,
				-36.78081047739239 + 15.413147294181432 * i,
				-17.493528227106335 + 6.231452208746276 * i,
				0.04064048770442774 - 37.41872936363492 * i,
				52.1676899967333 - 22.534757437429924 * i,
				6.49869048111367 - 40.58444102635213 * i,
				40.002825653679025 + 5.604977652128869 * i,
				-40.15622573607647 - 28.410882073257895 * i,
				44.038043336251505 + 26.701341757022085 * i,
				-66.08481620636942 - 19.57462901967508 * i,
				1.8737153059686689 - 68.55094998008643 * i,
				-39.331838917040585 - 53.660303089641396 * i,
			],
			[
				28.54059465409493 - 15.583126276747743 * i,
				1.7759627404416278 + 38.53037982005973 * i,
				-24.02608756872337 - 15.932395012234634 * i,
				-24.94223970739978 + 30.953121166835544 * i,
				16.317277723130758 - 45.714069045146985 * i,
				-17.979466797796135 + 2.481294631460429 * i,
				1.5218077601869027 + 20.942859601882503 * i,
				7.280577283017103 + 51.199037884133276 * i,
				14.938266991160532 + 15.708874549554992 * i,
				7.860579617220139 + 2.9236226765639053 * i,
				-8.962364670792006 + 18.016824049989054 * i,
				-7.151343845398912 - 2.334659653197555 * i,
				2.4705794039326463 - 44.92722588010676 * i,
				-31.86648157718311 + 12.142950697894037 * i,
				26.00900597920588 + 8.650039182268927 * i,
				-27.078020889271016 - 38.041622362521764 * i,
				-17.325381049230852 - 27.196746607448986 * i,
				40.765898211291145 + 24.476530806160895 * i,
				3.8915705171683044 + 6.143788431875279 * i,
				0.5751403554872923 - 35.82124381715815 * i,
				24.041739799935755 - 5.310767626740045 * i,
				-35.789572833443856 - 47.732351985842286 * i,
				16.61343046302379 - 28.88212299288648 * i,
				-21.778232255376786 + 9.141055575710617 * i,
				-26.93967430101017 - 14.442367823581526 * i,
				21.85951193448883 + 23.03722802930533 * i,
				5.837224672131121 - 15.018213971158488 * i,
				-29.275817850963307 + 1.3370083887767805 * i,
				24.244653078628332 + 1.130813756050486 * i,
				4.1239598495523495 + 62.811804888814095 * i,
				-39.308144816300285 + 35.137513385010706 * i,
				-8.946388532603628 + 22.043003509493747 * i,
				18.85054854884372 - 56.85315969610568 * i,
				16.74814837023179 - 2.4391247850520994 * i,
				8.404073005577654 - 1.9906108782058638 * i,
				-21.90970336857654 + 49.932915170146586 * i,
				-6.812569636886536 - 43.337689224476094 * i,
				-12.303066940656688 + 11.825115101933068 * i,
				-49.20103768456001 - 9.582339119955012 * i,
				8.387196243571054 + 21.1831163074076 * i,
			],
			[
				-3.5357792329740505 - 59.68573975229219 * i,
				-33.689857730995016 - 21.167154404908473 * i,
				28.78999914868041 - 26.737427466923705 * i,
				4.940660808524484 + 26.201416565674123 * i,
				-32.815902206844356 - 54.12883870570329 * i,
				11.677620618130248 - 4.969978298179143 * i,
				-8.113951943651983 - 11.227112597182867 * i,
				-58.38292660675876 - 1.3521675759727714 * i,
				-22.015101681900546 - 33.83977181318883 * i,
				-0.8582121468676291 + 17.677857193515592 * i,
				25.419472965819864 + 27.270634206680388 * i,
				-1.1132569458357011 + 19.397326467022765 * i,
				-20.421566639050326 + 45.7770089484437 * i,
				-9.620269905042363 - 8.277679797637916 * i,
				-4.503064541880093 - 71.76998104933313 * i,
				13.802990994450571 + 29.905330275622987 * i,
				9.562367782684298 + 72.91891869365816 * i,
				9.540501569564555 - 13.353248145730447 * i,
				-9.05237483855904 - 0.7620307802338147 * i,
				6.056779427905582 + 49.987055176556225 * i,
				1.308784203920979 - 9.923126809419927 * i,
				-31.25326802702411 + 14.666553680211294 * i,
				-3.207241440946447 - 29.423182624575972 * i,
				-29.748875334810293 + 2.825657273686332 * i,
				33.19084928418238 + 71.06574697884496 * i,
				24.78971120822375 + 29.970937556629394 * i,
				-3.7932786112865244 + 38.53913010983749 * i,
				17.023228600508958 + 56.6209337513442 * i,
				-6.08192931865689 + 14.61316317662812 * i,
				-5.955440518096395 - 18.935771870010804 * i,
				2.5612964064619703 + 33.8244774690264 * i,
				-32.556290585405144 + 21.506361594137566 * i,
				14.820942098901869 + 45.58449591511646 * i,
				-27.68410396394588 + 39.974098461794114 * i,
				26.952877076173806 + 26.78162810581243 * i,
				-5.914850678645536 - 41.11207628264059 * i,
				10.806160041126589 + 33.939575597521475 * i,
				39.00881240507683 - 15.354420541942888 * i,
				-14.788015581615019 - 6.733117857743682 * i,
				-104.73200952359508 + 4.760226907642539 * i,
			],
			[
				-20.625298155041605 - 53.90520825470385 * i,
				5.721446110529712 - 28.82977150267328 * i,
				-15.935854174346616 + 31.94655559262557 * i,
				20.4760121741445 + 15.18006663377787 * i,
				-60.09645762836544 - 34.52559784016313 * i,
				4.23997074480755 - 15.406912442301042 * i,
				-28.247040265099777 - 11.601228790919748 * i,
				16.096222196567098 + 18.549582147050558 * i,
				-55.613976801973564 - 14.744242493530436 * i,
				-42.54653500387033 - 31.5021886166372 * i,
				52.31011203656709 - 7.2358740514589215 * i,
				-4.723348074230955 + 30.034067180062124 * i,
				-9.613476143154589 + 17.63309684149519 * i,
				-35.9369274795813 + 49.1942271048583 * i,
				-22.32254319501979 - 87.07978377751778 * i,
				28.247267284948975 + 15.277484163852684 * i,
				43.134296253327975 + 51.62384038997544 * i,
				34.610559029592814 - 38.303919078558835 * i,
				-13.770022944497294 - 7.01092215936464 * i,
				-1.8404060248046505 + 30.53072497690294 * i,
				-4.906007786376691 + 36.03697603359204 * i,
				-22.66496596912188 + 16.308751151918713 * i,
				-11.115103306359913 - 33.31717409896794 * i,
				-4.520309194531642 + 9.019072256718601 * i,
				4.455293474546147 + 9.243802138023911 * i,
				23.90355888931096 - 27.6522830058743 * i,
				-17.82894057868981 + 33.849837662819645 * i,
				44.68542018611526 + 9.110605459250134 * i,
				-29.543845341747414 + 13.111002392922103 * i,
				13.840396586718924 + 12.35402627047171 * i,
				26.794386348095678 + 70.63416734688849 * i,
				15.074049042392602 + 19.359815096285374 * i,
				-21.557006267719366 + 45.725191383906484 * i,
				13.487015891660022 - 20.970823134457618 * i,
				8.050670198357679 + 8.397680785119256 * i,
				75.17668335843189 + 18.571575706502863 * i,
				-30.45229030561198 - 1.120639908934308 * i,
				28.87549897116034 + 13.761983131567376 * i,
				-27.14326649685647 + 24.174483473493467 * i,
				-22.586097249288837 + 18.078972655673425 * i,
			],
			[
				47.50283273204178 - 3.70805350742404 * i,
				-24.598874230177557 - 36.469372473026226 * i,
				28.850377963086032 - 40.61770062768889 * i,
				26.76864314134042 + 18.83302463057862 * i,
				-6.151833315464785 - 52.91241373036084 * i,
				3.1432055792416804 + 10.841955900197405 * i,
				-6.406408657168441 - 33.060084392915755 * i,
				-25.83444723091945 - 44.898566459244854 * i,
				-10.88921626723129 - 53.702101282113105 * i,
				13.405932015884588 - 19.347552820790497 * i,
				-8.65520116055879 + 41.024644628258756 * i,
				-33.3510071645548 + 16.635945972498547 * i,
				-59.964308560452324 + 13.264842551457209 * i,
				42.68307800054768 - 41.36076275896913 * i,
				32.37878561504654 - 52.82470066752239 * i,
				-34.862952041906226 + 2.439293627093491 * i,
				-77.67446884082547 + 44.86852545410283 * i,
				8.176716630083938 + 19.882304571509195 * i,
				13.183839988882909 - 18.663980584356167 * i,
				33.78261923692318 - 30.060046224502784 * i,
				-5.3987389877799785 - 34.540099914514556 * i,
				-62.07098057876117 + 20.07789992489296 * i,
				-2.9173561833358974 - 11.960033701117226 * i,
				20.1933953443138 + 24.94183913416434 * i,
				-3.734124602680538 + 0.8954197099442103 * i,
				63.99764377393302 + 50.01830484796605 * i,
				2.4723795485542723 - 33.72369018173109 * i,
				0.5648448216209943 + 24.777765796048556 * i,
				-27.080842039671708 - 58.73373791310421 * i,
				-19.639536155388292 - 24.313305709283924 * i,
				-67.76594839285414 + 29.44608942676743 * i,
				28.4732042785825 - 11.146330164253001 * i,
				-66.4131577816626 + 23.424142532041458 * i,
				2.7088452167114365 + 60.84228152058945 * i,
				-65.23740734485266 + 53.93829509194309 * i,
				4.617015815496583 + 68.72671548949586 * i,
				-9.261860378512047 + 9.012473377842051 * i,
				-1.1275979852754516 - 60.99403464705627 * i,
				35.00056310144652 - 45.93018972790337 * i,
				8.93184393102788 - 44.45168024954286 * i,
			],
			[
				-49.475782458153475 - 65.11497519242316 * i,
				0.34476087057848304 - 40.697190377479586 * i,
				-4.749002409350027 + 18.336119560242068 * i,
				64.33158721867096 + 8.688971165637042 * i,
				-64.15572517408128 - 25.077945382101248 * i,
				-11.242474715263597 - 6.324164296616345 * i,
				-1.2830615415942503 - 55.384724103861515 * i,
				11.919293641278273 + 35.558090342805976 * i,
				-2.7931599045245363 - 7.56952187471137 * i,
				-20.20975107356213 - 25.856229565928828 * i,
				23.78589024321935 + 26.27163636720839 * i,
				-14.930501503203828 + 13.048924018025295 * i,
				-6.593715354547982 + 16.70738918209055 * i,
				-18.167178204595615 + 28.837139020997625 * i,
				-22.255080174501927 - 70.42920831693552 * i,
				31.79998968893625 + 33.40314005989359 * i,
				-9.885617458603136 + 39.92581106432365 * i,
				3.9357899720737635 + 11.158507889562259 * i,
				-9.628639386536491 - 32.76431382264346 * i,
				8.455757826529648 - 7.25042754891502 * i,
				-31.520418261068862 + 18.527867269863656 * i,
				-37.51958438545674 + 1.6217506646729873 * i,
				-2.621808755744766 + 11.066857281099825 * i,
				6.435395074915679 + 12.863410862029326 * i,
				0.04812780649378823 + 38.37397318767748 * i,
				19.898566108594594 - 31.482222841301564 * i,
				-13.259522191807974 + 21.92903980229777 * i,
				52.983361419702184 + 23.39663783897597 * i,
				-37.71589567540537 - 12.21749182752729 * i,
				-2.8325545916933095 - 46.04776202903533 * i,
				35.58901545175842 + 48.834128612765454 * i,
				24.700906530532606 + 25.795673546789807 * i,
				3.6342317709994916 + 82.88195215170174 * i,
				19.681911205527072 + 27.97494645290615 * i,
				-19.664330085560074 + 12.684046752274845 * i,
				36.861132410314674 + 42.677658624002135 * i,
				-58.05946749347715 + 7.214704399870957 * i,
				69.0123503774615 + 22.706363438928694 * i,
				4.686660232615612 - 12.5935816027585 * i,
				38.667141144924265 - 12.009938931705012 * i,
			],
			[
				-47.00083280576598 + 16.782929222825743 * i,
				12.920411639103008 - 15.9659695844271 * i,
				39.08727105530375 + 20.302012903034665 * i,
				-22.859480908304903 + 38.228725329067515 * i,
				38.38345546871251 + 37.12296245851195 * i,
				-17.992425580723705 + 27.850974525354843 * i,
				31.67866670028748 + 27.734434160147572 * i,
				3.9252246672607214 - 56.774674728646715 * i,
				-6.315382808171634 + 16.525845454124422 * i,
				-24.43058686040512 - 4.461407740538789 * i,
				-3.57867150966473 - 35.07382065285992 * i,
				-18.153184992830234 - 39.03951153694257 * i,
				9.847544051975774 - 59.747545674507506 * i,
				-5.945692287183533 + 8.901931934859874 * i,
				18.405372105613978 - 27.146721261332658 * i,
				22.111092610147935 + 4.068642334929352 * i,
				10.59067485570774 - 37.54044817059929 * i,
				-21.75930153867958 - 28.58439643642297 * i,
				27.293538026026795 + 29.142907123947793 * i,
				32.937507779677105 + 31.296234272123144 * i,
				16.82945671446251 + 57.494928842399744 * i,
				9.998531696713755 + 8.937019237078147 * i,
				3.6900110003817588 - 19.946334164253813 * i,
				31.81751058964828 - 17.523855282590524 * i,
				-35.00579667163724 - 18.911544912807905 * i,
				-69.42025493875913 + 50.802565904826054 * i,
				-15.037094373933357 + 53.654841922534565 * i,
				-57.099649974198925 - 15.485484899110482 * i,
				27.629877248871765 + 74.25600569790603 * i,
				36.44080187137095 - 13.77881004526489 * i,
				32.160663436425295 - 2.8194480590750697 * i,
				2.7981057506767204 - 16.93634666718385 * i,
				-32.45046591426954 - 26.171444732887775 * i,
				-24.847231203023647 - 29.258605048814726 * i,
				-6.319404404544123 - 2.281167456609282 * i,
				-36.02128783200105 - 111.43911639082827 * i,
				29.30011108127967 - 1.5443002800002361 * i,
				31.839523857007663 + 36.48299595968461 * i,
				49.55612329475002 + 37.377575720416786 * i,
				-18.394267214983646 - 7.5213289777809855 * i,
			],
			[
				-23.89302799484389 - 27.33369921864705 * i,
				63.75744688521593 + 80.47838089377595 * i,
				-9.880516279863045 - 69.82880393251293 * i,
				12.617438151531331 - 81.59036920732844 * i,
				75.22851360061087 - 68.5342513878486 * i,
				3.1959868865274643 + 9.447067305674068 * i,
				6.530219789455842 - 22.210877624664718 * i,
				-85.67963356541671 + 61.249794872340544 * i,
				15.45083442452436 + 27.42895326837423 * i,
				44.439744570309344 - 13.388114632159784 * i,
				-47.24425770546404 + 4.336988892772282 * i,
				-13.572627465562057 + 10.242215987546075 * i,
				21.429890206013226 + 0.4601963483031497 * i,
				-78.1685885993539 - 7.837283876436743 * i,
				-23.67753893950725 + 34.50966413103162 * i,
				-38.044096013828025 - 48.19463713404893 * i,
				7.394033202648899 - 37.09742071727337 * i,
				-50.118342968937384 - 2.706904759307932 * i,
				30.428691326487794 + 23.680317019341572 * i,
				89.3226303630612 - 82.9389631837063 * i,
				40.98808891971925 - 20.78192876901558 * i,
				-22.48133486229881 - 15.582220493418632 * i,
				-12.098716985169176 - 22.574219943320657 * i,
				-15.208361131625963 - 21.21658629868955 * i,
				-46.99334021359758 + 20.400530103566027 * i,
				9.63285398422127 - 6.29509265728106 * i,
				16.68408302107369 + 5.861951438898771 * i,
				-24.186902251956077 + 30.91816410929131 * i,
				40.257256721730634 - 42.4237668414176 * i,
				56.35898177011376 + 51.980968379825 * i,
				-9.644232156507229 + 58.74363555548047 * i,
				-43.69670473195748 + 14.580467183102366 * i,
				91.84604888933613 - 21.99799652310147 * i,
				27.284923408230412 - 22.536354674872143 * i,
				107.56756101918927 - 41.57759504397463 * i,
				-133.120602306975 + 47.60878690404279 * i,
				6.845192891953218 - 101.49492085033685 * i,
				17.32386848366346 + 23.582536636775238 * i,
				-4.993637898930341 - 47.90636814203103 * i,
				22.824463269065827 + 33.60394422592093 * i,
			],
			[
				73.92475160159236 - 34.382793787237645 * i,
				-37.380165366617064 - 19.480014892807446 * i,
				37.19449380946361 + 59.84509529573279 * i,
				-61.83606473404757 + 87.16184759014888 * i,
				43.126737780281445 - 59.35293420961668 * i,
				-73.16011172627562 - 53.46905042029522 * i,
				-34.64405531952475 + 74.88023142614011 * i,
				-40.17611438669611 - 4.2659681791589605 * i,
				-0.9121096633561407 + 30.270040618314013 * i,
				14.82220784143908 + 17.34285776801522 * i,
				26.300197917588605 + 15.094540971766378 * i,
				-54.1063159815167 - 9.99505872644125 * i,
				3.329700260928348 + 3.0075760023164833 * i,
				15.466596129364703 - 16.59763756674704 * i,
				32.15146103832194 - 7.659983032009776 * i,
				23.484486039200817 - 15.463457629078789 * i,
				-6.800869360938748 - 6.546006020069257 * i,
				5.6728975675345765 + 42.65510955537334 * i,
				15.224966595334536 + 19.386975474164316 * i,
				35.77621297895939 + 15.455784464370879 * i,
				39.40504440932927 + 23.713522576113803 * i,
				-9.640877718733421 - 73.82336383577098 * i,
				-8.087535846813758 - 11.886237699932956 * i,
				26.233236826620924 - 26.87025604292637 * i,
				-41.91718045386536 - 51.10515825008556 * i,
				32.01643670592722 + 36.95012503985117 * i,
				-23.750209386229145 + 41.23061930942605 * i,
				-65.74916814725641 + 81.93524093200061 * i,
				12.589578741158437 + 32.61799278607167 * i,
				-65.82794504590721 + 30.155667598551958 * i,
				-34.5948853623033 - 0.3140583003140023 * i,
				21.659889037975425 + 14.084862207099498 * i,
				-12.0875095599263 + 38.21129173489684 * i,
				-66.59875859583829 + 101.69549733531846 * i,
				-68.93420143391944 + 36.811032112241925 * i,
				25.540262015727997 - 100.05864069211977 * i,
				28.130010292191763 - 41.45740560984442 * i,
				-49.71856228456848 + 89.94313528051737 * i,
				-11.423001797274068 - 25.665353429073615 * i,
				-23.315299537768954 + 70.21028670252188 * i,
			],
			[
				31.635880501191977 - 25.366686812232345 * i,
				-52.86744718172963 + 32.19343455375508 * i,
				4.325224480061706 + 18.898597864583955 * i,
				-57.0066245596664 + 28.99362591456827 * i,
				-0.0887048627104079 - 26.364007249741498 * i,
				-53.88622886338635 - 54.52064205089022 * i,
				-17.47316635053481 + 74.71815155924324 * i,
				-46.92012959279384 + 32.08967030875141 * i,
				-3.366270771129223 + 53.962809920416035 * i,
				26.423710133971667 + 47.1584477241712 * i,
				15.971177118464936 + 33.12713940064133 * i,
				-26.21538861333144 + 1.297301464562075 * i,
				23.207673815201133 - 25.845368898394355 * i,
				-18.021551162667834 - 2.2751940849154373 * i,
				-72.28373989142938 - 12.61944051360728 * i,
				17.765091994115707 - 39.42295660794443 * i,
				4.201051691146333 + 21.137808723662204 * i,
				51.09114188693394 + 50.00475544815056 * i,
				-10.438633582776475 + 22.9167648354902 * i,
				-9.46537004075763 - 44.24972633493477 * i,
				42.70745091010903 + 17.207348389216435 * i,
				-18.180391487872768 - 32.72251042241648 * i,
				-17.352010263815195 - 50.05405300689405 * i,
				13.743687499810054 - 30.939764551244963 * i,
				79.93624127836321 + 33.061611828884 * i,
				30.154200892298604 - 31.01661486605416 * i,
				-3.7406028062807053 + 26.416651514167043 * i,
				-12.967789192045682 + 35.46038356748625 * i,
				15.827596815827707 + 23.87612541781953 * i,
				-68.62582570670739 + 53.68537205788773 * i,
				-12.170722855122214 + 28.865281032871327 * i,
				1.8988478314101336 - 5.953421234022542 * i,
				58.14741630557843 + 7.463482332893131 * i,
				-9.224635029857808 + 70.06505030007048 * i,
				27.34895483822358 + 37.10974665547179 * i,
				8.18206986546865 - 96.40809551799184 * i,
				46.23604366412268 - 51.873652722634915 * i,
				-49.65227456965019 + 33.3509221972479 * i,
				-50.123827002746346 + 19.548825160300343 * i,
				-85.52131879936695 + 80.23163082797168 * i,
			],
			[
				-22.7307694616743 + 1.178336546274224 * i,
				-24.684945148130943 - 4.927262161901041 * i,
				43.414009886083036 - 0.10250093029083018 * i,
				14.317748913292974 + 16.99394961914641 * i,
				-37.054963883249314 + 19.24620624535254 * i,
				-26.275673281510706 + 5.737002248117392 * i,
				-20.76035611104382 + 34.13900538682334 * i,
				25.995499586170922 - 44.047168770530504 * i,
				2.969611341252957 - 2.7237143340227803 * i,
				15.304599760164507 + 5.997288230706545 * i,
				3.4854400626095483 - 52.19530354191819 * i,
				-7.599069294096807 - 1.432421464035741 * i,
				57.199789012501924 + 11.62061458813831 * i,
				45.97788150304169 + 18.884753956635805 * i,
				6.593698757819993 - 7.8794981949474305 * i,
				2.3061274488899812 + 18.63743016694349 * i,
				14.494111621109028 - 24.3265667944331 * i,
				29.934965208174674 + 6.584540675180058 * i,
				19.539830479311355 + 4.063261207289258 * i,
				17.28022775125529 - 19.694631271735375 * i,
				3.4395284887458217 - 13.202603209255116 * i,
				-10.10006598979372 + 21.048016304580955 * i,
				-5.01849711657502 + 17.688148497686527 * i,
				30.359006645951133 + 33.06093864035989 * i,
				-13.596894222285542 + 3.756206049302179 * i,
				-11.465846347583458 - 38.14882467012106 * i,
				45.15077780887542 + 4.026267310919806 * i,
				57.185214461553166 + 0.04654714666412474 * i,
				-11.844506386368067 - 15.743046663642744 * i,
				-29.205265972214114 - 23.7210996954461 * i,
				-26.088707092331944 - 58.96415275604358 * i,
				53.68020936001696 - 11.28829681793998 * i,
				-6.3841750599303 - 13.432413489843501 * i,
				19.657710325904844 + 44.47892679501565 * i,
				-32.7673760290358 + 25.042662016714818 * i,
				11.878553914562062 - 38.471373953044846 * i,
				27.68512098601679 - 23.060621581136953 * i,
				22.491548591845614 - 30.16672821030201 * i,
				-8.291590829841937 + 58.367958038512185 * i,
				37.09812718714799 + 11.246062902243807 * i,
			],
			[
				-11.93511219230679 - 51.42802670522292 * i,
				-0.1455427330553789 - 6.532753280506029 * i,
				15.813155585487493 - 4.239490150750015 * i,
				-21.697775317090368 + 23.67450438743177 * i,
				-3.855138150192886 - 35.75835147183366 * i,
				-10.375856173446692 - 14.15822754355196 * i,
				-3.137983616127329 - 14.776850820167851 * i,
				8.302059666902757 + 78.14614118507143 * i,
				23.12754853587998 + 46.555241058565684 * i,
				23.703640170288775 + 1.5419317964465291 * i,
				7.253946750945033 + 14.375662531759787 * i,
				19.030316788183008 + 2.1640145186776207 * i,
				-7.825644146144185 - 12.699557050473771 * i,
				-40.24468917578696 + 35.64909163811849 * i,
				3.946220847004895 - 29.540970257674815 * i,
				7.959125319807855 - 19.72086707657634 * i,
				-13.042491195571873 - 2.4171930686308087 * i,
				-1.38674593153398 + 69.77672905062774 * i,
				-12.2433914045846 - 13.834639865282469 * i,
				34.784363095611326 + 3.14833719679266 * i,
				-4.887944191382715 + 13.361337758444947 * i,
				-24.848719921850346 - 39.683189747057085 * i,
				4.29209972404135 - 18.697767921184376 * i,
				-30.511443553232198 - 8.665994134830889 * i,
				-19.225805363349373 + 25.54294140004402 * i,
				33.98318928748378 + 21.14338012889067 * i,
				10.907399400912283 + 34.93715304988244 * i,
				10.34451578341308 + 84.31904406177027 * i,
				59.02283587240533 - 15.660654655712719 * i,
				25.496160757252255 + 2.3020075694237576 * i,
				27.189969359739464 + 9.773651857149236 * i,
				10.912754570217686 + 60.94251469709755 * i,
				44.59208639281714 - 24.73447719871571 * i,
				2.3215255068664646 + 44.38378853339704 * i,
				5.885442925848231 - 25.886483167822142 * i,
				-47.74576782448476 + 24.75837860454751 * i,
				32.772840905912915 - 24.96950172010171 * i,
				31.061905074071348 + 50.435535531214384 * i,
				-76.30139590418084 - 8.192421503860341 * i,
				-4.130423422294769 + 45.11781609158833 * i,
			],
			[
				1.2659540051087106 + 24.925957061642663 * i,
				51.67299256569147 + 31.35640478236334 * i,
				-40.92277555121095 - 44.60743653810829 * i,
				-20.76995835158708 - 16.96390767196047 * i,
				4.666290262799748 + 70.81716461317441 * i,
				12.509026813018735 + 22.776518526874398 * i,
				32.55350959618903 - 18.53257642954918 * i,
				17.99675264650017 + 28.154594706464735 * i,
				19.174341500137068 - 18.24737569602327 * i,
				18.483415084640228 - 52.24129963731052 * i,
				17.94542015929626 - 9.663922354591065 * i,
				38.19055273995432 - 12.038765490348027 * i,
				24.18998530477352 - 34.29249662943943 * i,
				-49.318617358211114 - 53.45657624592248 * i,
				34.72338643154685 + 55.691040040122004 * i,
				-35.37914193843888 + 2.2678678807557873 * i,
				-10.77102716822256 + 3.5949850931830483 * i,
				2.7883638324586855 - 22.662273083891975 * i,
				-8.326580670190385 - 9.627564961250041 * i,
				-77.49998812050882 + 24.803408467361322 * i,
				-43.77239559742128 - 27.334931930432596 * i,
				52.946965368224895 + 34.90916099609235 * i,
				18.80775324265844 + 16.516751372865684 * i,
				-8.92233214902727 - 0.47278732785008515 * i,
				25.366784030685523 - 59.47331403401701 * i,
				-60.48214764772351 + 11.674682621457848 * i,
				-20.6268077276235 - 39.49445887055417 * i,
				-21.562360441482213 - 61.60726423676908 * i,
				-10.384371646030765 + 22.56798719139077 * i,
				48.472653249597485 + 42.28116201977733 * i,
				-0.6170586705847114 + 2.836751485940809 * i,
				-44.450419318981886 - 23.236385289675944 * i,
				-27.039553898042477 - 67.67893387694141 * i,
				45.974405651836555 - 71.89096023397339 * i,
				47.696854809145265 - 24.785799368098594 * i,
				-34.163370954643796 + 5.711953632319318 * i,
				17.61114945717516 + 1.316460121948822 * i,
				-11.444853447207658 - 6.41097418109635 * i,
				-10.467460831561642 + 11.327177196870863 * i,
				-13.344042887188255 - 22.30440753638801 * i,
			],
			[
				85.97720128165054 - 38.86278183681418 * i,
				-10.723739729177758 - 5.1710139857040645 * i,
				22.495199813419845 - 5.7579177434700455 * i,
				18.136521914190045 + 45.462303898900345 * i,
				68.06820136331814 - 88.38331304290473 * i,
				8.32691843253582 - 13.294450044095193 * i,
				31.398625574221 - 7.466694589916248 * i,
				-59.02168115463138 - 32.27183615155828 * i,
				15.826205983010937 - 11.73510125612512 * i,
				20.447263893089882 + 4.587699762952425 * i,
				-22.296208838146924 + 20.065728212062623 * i,
				-39.96214309655082 + 32.14276453308639 * i,
				-26.166880809746544 + 22.56128936127049 * i,
				-14.76316523891046 - 30.68223375177982 * i,
				65.98544048495964 - 24.924168276224968 * i,
				-4.401283579071901 + 1.6041633837359228 * i,
				-68.09817652753912 - 14.228427640032514 * i,
				-2.6001810436749224 + 2.1146971559881393 * i,
				26.266061297836416 - 10.57507079993212 * i,
				6.565881195702502 - 7.106516169911984 * i,
				28.071604844680145 - 32.29364167279505 * i,
				-2.2213112428313195 - 68.01927400864787 * i,
				-1.572585586378473 + 13.460014564169583 * i,
				-34.63517733330248 - 6.308410738869802 * i,
				-102.48914309117762 + 75.37475617856775 * i,
				66.65008648800519 + 34.78420519135656 * i,
				-18.158221333532936 - 15.085566391942784 * i,
				-68.54844776793767 + 59.533656639284246 * i,
				27.491397744695327 - 43.91799871224707 * i,
				-29.64101755879112 + 41.410102063171074 * i,
				-63.411746300985044 + 29.461656117692385 * i,
				-56.08134549929534 - 20.034722328398395 * i,
				-12.047356536440724 + 32.89601567808943 * i,
				-24.759890506362687 + 34.167902803829136 * i,
				-19.54291342026823 + 27.69384501949406 * i,
				-74.10009331113861 - 2.4667687481359337 * i,
				-0.14441156204461691 - 9.545445892007514 * i,
				11.229317133881786 + 46.74382507654973 * i,
				-34.97151888903262 - 70.68890615324082 * i,
				-16.12934324048154 - 45.74688249440054 * i,
			],
			[
				30.804735741054145 - 11.199328773106451 * i,
				-35.003385075199816 + 6.715704002711241 * i,
				-12.886293753266898 + 13.299542430504687 * i,
				20.116514083615 - 2.287386910939958 * i,
				-0.2798630922994505 - 31.388917776533482 * i,
				0.12034510313446845 - 29.700984397121182 * i,
				-43.86912915839048 + 18.395328749603507 * i,
				-54.98214173549668 + 22.117490480910647 * i,
				-15.973623194467756 + 5.336268254891095 * i,
				19.698080593742517 + 16.109786596357978 * i,
				10.244929192222315 + 1.8943703287391713 * i,
				5.454925292816455 + 23.277489839136766 * i,
				17.60115559915136 + 50.61297132483075 * i,
				2.295755757137215 - 50.66666153198261 * i,
				46.30932117885766 + 24.233195607625603 * i,
				13.133539688605595 + 5.423823360698719 * i,
				-9.022476846609571 - 19.786859108608336 * i,
				-31.298937671400942 + 19.932494162876367 * i,
				23.584473840075823 - 10.197251459492307 * i,
				23.875959401771468 - 38.16514791192146 * i,
				14.693290531980907 - 37.375144097922814 * i,
				2.6264822869745075 - 28.13246331104203 * i,
				10.556035746349998 + 29.060970364413805 * i,
				-6.686300964057438 + 7.713525265149798 * i,
				-90.8662333002558 + 26.399616331294876 * i,
				61.43315127722573 - 2.524751014526231 * i,
				-3.3791908548838627 - 6.034038592611111 * i,
				-15.729037607952023 + 51.353669278412575 * i,
				-22.31999234263307 - 17.768596507037874 * i,
				-43.027605507966015 + 23.38871720792342 * i,
				-5.65536096967876 - 10.12584337613433 * i,
				-11.839763907498902 + 12.906866267084588 * i,
				31.8525896151574 + 26.109924156075852 * i,
				6.2958778689325925 + 21.14163036999662 * i,
				-9.926099275215595 + 32.18683685249255 * i,
				-6.365000302237739 + 4.751278766398077 * i,
				17.847270842209305 + 21.659042820521577 * i,
				-40.82747777009605 + 25.05838789081676 * i,
				-50.038702001285145 - 61.93869182610665 * i,
				1.9407025522713015 - 2.972802658344452 * i,
			],
			[
				12.357339688635317 - 42.79361129666253 * i,
				-8.030875563180928 - 3.0442033746774513 * i,
				64.25215855921768 + 32.62854928377088 * i,
				17.79357595572185 + 48.069960612782836 * i,
				-6.390609052411008 - 17.17480731722077 * i,
				-13.438990335199065 + 21.360844995688396 * i,
				35.44698159562311 + 13.966353076053025 * i,
				17.9217481626591 - 65.57913569987555 * i,
				-7.472428603108426 + 20.834801788892158 * i,
				15.39703072268405 + 45.48533690845923 * i,
				-22.91590676337172 + 7.187551544067642 * i,
				5.122001767258425 - 6.022277447271833 * i,
				-11.95522611503447 - 0.0028972146206029947 * i,
				57.80877697744499 - 2.602012924983228 * i,
				40.84278892200723 - 6.43552791110551 * i,
				-7.672246072879425 - 8.449196914049011 * i,
				-7.6426288510148765 - 51.29172142012612 * i,
				16.98526481388359 + 3.932761836298077 * i,
				-12.365211744983789 + 19.98689512226227 * i,
				39.33882089230997 + 13.847863630700573 * i,
				14.087055508215393 + 11.132616854576298 * i,
				-52.98650955438529 - 17.94978992036879 * i,
				-2.117577235189998 - 14.03356463966623 * i,
				-11.647319905025045 + 7.79034400078848 * i,
				-63.92230421449135 + 4.035046926411809 * i,
				6.046986040422382 + 4.6273631962396 * i,
				-8.081447399843313 - 7.969215067253543 * i,
				-11.58194144748868 + 7.237224693333573 * i,
				37.11193888915619 + 0.9541188040251001 * i,
				27.64528489760208 + 21.39930018751687 * i,
				-36.19537617683979 + 13.045069057993317 * i,
				34.373288573553246 + 9.315037068204177 * i,
				-5.2663742493670185 - 29.10174518576407 * i,
				10.887358366339498 + 31.43598479785139 * i,
				-26.679771651132995 + 31.40733527625038 * i,
				29.563159099938495 - 48.68127681055062 * i,
				-16.025307490245247 + 64.17689521042493 * i,
				18.488183106164847 + 30.77682384525022 * i,
				-7.834667804234201 - 13.91148438820306 * i,
				47.08142548604982 - 39.88152303043765 * i,
			],
			[
				69.77353695085276 - 10.585475843154903 * i,
				6.802979090135349 - 4.6977277151446515 * i,
				10.602515830389024 + 2.159911800301984 * i,
				-49.14279845630789 + 32.71012289295372 * i,
				12.213941268918404 - 35.95903505193857 * i,
				32.83264256880405 - 16.255526747481696 * i,
				-25.517117720624576 + 13.281771003796507 * i,
				-31.04916079361146 - 5.892564883564816 * i,
				-27.401757447813964 - 9.831600327570044 * i,
				20.626294833245957 - 23.729729604566444 * i,
				28.855173715877584 + 0.7457016191739996 * i,
				3.609794653896671 + 5.4180885902807745 * i,
				22.895217818912293 + 23.70821028417319 * i,
				-32.36583230377772 - 8.963069149489655 * i,
				40.58274702748913 - 5.908138804742663 * i,
				23.905723999125165 + 20.002275518200918 * i,
				-25.491851802233214 + 33.83390157499875 * i,
				0.9127204422597259 + 29.942057615260524 * i,
				5.282936165933961 - 24.8269858891195 * i,
				-29.22829444427549 - 8.980033659011099 * i,
				-33.856345360836734 - 24.602811491570954 * i,
				23.84805703994175 - 13.5686741289731 * i,
				-5.219606838667474 - 13.752388875083627 * i,
				-14.430768957225439 - 13.236168706138216 * i,
				25.66158624031911 + 22.996567960584244 * i,
				22.4200274789971 - 23.3200975774882 * i,
				-24.244106590697193 + 17.38393672158996 * i,
				-24.038045096308892 + 25.99978246582785 * i,
				0.5752782206896825 + 31.34185828708602 * i,
				-13.379956512176719 + 29.34009751233218 * i,
				-41.06076888627553 - 23.204278202180127 * i,
				-33.82752027964192 + 15.028584268085012 * i,
				-13.273566712065438 - 35.239278811812014 * i,
				-31.90567513795584 + 15.153892342481399 * i,
				5.668020357780218 - 0.3921454400403661 * i,
				-2.224605811885919 - 2.3046951327781926 * i,
				56.70019333374036 + 1.9168847215488896 * i,
				-31.8270872897136 + 63.47832905614035 * i,
				-60.81014126679963 - 16.488180664810194 * i,
				-74.30467919849414 + 0.8182238841415101 * i,
			],
			[
				-13.430433699074058 + 20.172971027312656 * i,
				8.70606066866543 + 26.97818576788461 * i,
				42.4241354960167 - 22.733717185205034 * i,
				-18.75246545854543 - 14.643581475763114 * i,
				30.739568225539884 + 12.07404730347416 * i,
				7.720143675961168 + 37.01130886965173 * i,
				32.04208214782549 + 48.038773742700656 * i,
				-25.041147251115063 - 11.148656766389028 * i,
				28.286926880151846 + 57.16706533224283 * i,
				24.720804108870226 + 16.19891224006804 * i,
				-32.11746628602964 - 0.1530410582412145 * i,
				-30.548791640129224 - 32.808044712071194 * i,
				-2.8891646013022836 - 49.711873596040846 * i,
				-1.344001863893311 - 2.9441696751881565 * i,
				2.673971783526156 - 1.8420433188900347 * i,
				34.30074986652198 - 37.821162895287785 * i,
				9.136364891622634 - 29.12976796377569 * i,
				-2.8867025659938435 + 29.726920350786074 * i,
				10.318597605994881 + 12.818251048806488 * i,
				12.4433813947862 - 26.325288517438395 * i,
				-1.3988146180205447 + 39.12930883416033 * i,
				-22.787498108773732 + 26.811537389550054 * i,
				-4.229766828203949 - 31.582710207510786 * i,
				22.351045635324624 - 44.87830256505845 * i,
				25.501297860295622 - 40.20590301153781 * i,
				-42.03485953355871 + 11.099008746578448 * i,
				12.837113892069752 + 18.09749694654151 * i,
				-3.262513410983635 + 30.20503206115756 * i,
				90.93647833197653 + 17.129694079387143 * i,
				8.550582352989935 + 20.69274709782485 * i,
				-60.51641501991767 - 29.82418926313315 * i,
				42.436508198269884 + 13.324064785140706 * i,
				-3.785996389719692 - 66.44628580481115 * i,
				6.191777605979446 + 46.9611553550631 * i,
				11.810565367270954 - 14.892051190413397 * i,
				-63.80760693010095 - 31.556786652483908 * i,
				31.46345553307431 - 13.049539601632368 * i,
				18.667583730013035 + 16.83546117871706 * i,
				41.04455104616386 - 0.5058629619662152 * i,
				7.215074349958879 + 48.335034351126104 * i,
			],
			[
				53.86888294230599 - 30.269218533893763 * i,
				-22.60542935934879 + 6.108840830802649 * i,
				62.778055406604665 - 6.072073887686173 * i,
				5.572331052595729 + 43.968730860537775 * i,
				-2.3167871622500797 - 9.252595250667355 * i,
				-39.320734189643076 + 49.710148408893176 * i,
				51.458285228191244 + 20.32996937988292 * i,
				8.092959491277867 - 18.445811974325352 * i,
				22.56551921010106 - 0.8730724358220527 * i,
				21.943661876911705 + 8.94301330704478 * i,
				-32.81200781364006 + 1.279705268017807 * i,
				11.539988026385622 + 12.26107092592368 * i,
				-30.472037443562797 - 25.538613341257406 * i,
				14.012724994995978 - 29.735245291621524 * i,
				45.51193750205286 + 16.43127093143883 * i,
				-30.690783676536057 + 16.28407952549682 * i,
				-46.86918755984691 - 18.6885711526251 * i,
				15.748830297473726 + 30.77122741050016 * i,
				15.57937593190966 - 7.075113361179646 * i,
				-19.349066532333417 - 6.922611643416019 * i,
				0.19975550825708321 - 41.73400059621287 * i,
				-28.77180902334503 - 13.674072561199242 * i,
				-13.334063664887992 + 10.416977030873014 * i,
				-1.8787155803644353 - 27.600713068491736 * i,
				-28.053549693140283 + 0.45088971158887503 * i,
				10.047889233104385 - 0.19315008445062176 * i,
				6.588089900012378 - 24.5659376995196 * i,
				5.416268797700042 + 28.068255418640774 * i,
				13.6725253134192 - 30.40447087345369 * i,
				14.94789420914033 + 29.57827206823262 * i,
				-31.89568448364048 - 9.665893324136405 * i,
				-8.930948673104139 - 27.62564211583696 * i,
				-10.535735981292532 - 28.932645521236466 * i,
				34.578876068415056 + 15.859104880902962 * i,
				-6.7519524645227165 + 25.380237691018202 * i,
				-65.77273105010207 - 16.1726618005629 * i,
				-8.884686749542121 - 4.53338008016061 * i,
				-8.941049832111094 + 15.823050350204914 * i,
				-15.439650922030381 + 26.31737403889332 * i,
				27.85406760269613 + 0.8433384658903716 * i,
			],
			[
				20.832029235439734 + 26.95394064767038 * i,
				-16.400532828695702 + 40.76068137064118 * i,
				-20.3762412990431 + 16.7239385107724 * i,
				19.328413673876327 - 61.28608206590842 * i,
				45.96058695668779 + 16.155240513373784 * i,
				13.710412595693072 - 19.48411134973752 * i,
				-34.066536041118894 - 15.770062256206842 * i,
				-46.8420215424655 - 39.03356636680786 * i,
				-19.480650001424408 + 5.505433616299598 * i,
				10.713421816103377 + 60.30387541586941 * i,
				-31.36677550435138 + 29.074556083333796 * i,
				-14.430257608224437 - 16.462598403083717 * i,
				22.246786746159614 + 23.430607188649706 * i,
				7.72118182656628 - 35.70251845389575 * i,
				-15.977367504374826 + 39.50085615616264 * i,
				15.180691688751196 - 16.80790360218699 * i,
				-1.0860621081625368 - 28.429737515088814 * i,
				-26.413135729092783 + 20.224141929410408 * i,
				-10.869179736870485 + 22.85819276349531 * i,
				45.47563659539801 - 24.89850307420174 * i,
				12.810632491343398 + 12.15447626072323 * i,
				-0.5447185934059462 - 27.71939781848519 * i,
				38.50406413551025 - 5.143842113390489 * i,
				26.317598880680457 + 7.336121158233738 * i,
				-5.94565721309209 + 31.041143424885945 * i,
				34.662460848476385 - 17.938123618305234 * i,
				-27.359999000580608 - 11.319744853409915 * i,
				-40.08508776495662 - 13.467925157270614 * i,
				-11.024779910842561 + 1.4449956877013506 * i,
				-48.64902745757692 + 2.4260988277217184 * i,
				-22.84785025590069 - 2.163623937891998 * i,
				-11.063674528206866 + 7.564286757628977 * i,
				52.895252846892355 + 33.475801564797365 * i,
				-16.952768256128635 + 0.37058751998078066 * i,
				10.068118743774576 + 35.85258433527894 * i,
				29.518598517282292 + 5.452012814772586 * i,
				26.8930085383461 + 31.069011724087645 * i,
				-72.57578140422834 - 36.73743801610483 * i,
				-0.7410449602371791 - 48.375330706839954 * i,
				-15.791992768532978 - 25.937632165842054 * i,
			],
			[
				17.070266539204937 - 33.826193113112105 * i,
				17.20256302927534 - 4.369823694298873 * i,
				-0.6256453995751272 + 29.040000368609924 * i,
				-38.45091310443095 - 12.417149771969164 * i,
				37.26687454387479 - 100.15485843281785 * i,
				-5.296562238154991 - 44.7565850973765 * i,
				-12.870426421092247 + 36.81093204527298 * i,
				-62.96342043205158 + 34.45676451494757 * i,
				-25.65881564831932 + 7.612941073780711 * i,
				28.09220025414359 + 15.584138758739257 * i,
				-6.26580460956464 + 45.921037474776 * i,
				-22.092552748419394 - 13.448172190919607 * i,
				12.454877318713585 + 19.62016128287658 * i,
				-27.47492076058691 - 3.3122854863668465 * i,
				32.36278210611842 - 84.24590251732624 * i,
				-9.657352179107093 - 15.520736639512904 * i,
				-33.62497158008675 + 10.877884095249286 * i,
				-50.372050923962924 + 38.06600038138231 * i,
				2.973601939430125 + 2.3474216974971718 * i,
				68.07303750819072 - 40.082736816253984 * i,
				3.049013002400798 - 5.770785816519554 * i,
				-25.591941227806245 - 23.994573979299048 * i,
				-13.548247021811974 - 60.6852425437873 * i,
				23.109668161291488 - 2.634982788187358 * i,
				20.910087786549013 + 112.12267078497081 * i,
				80.67373963816945 - 7.762790915286587 * i,
				-24.73754515720994 + 17.774368270969205 * i,
				-26.194857361465836 + 68.57878921317972 * i,
				10.04495703858099 + 22.975011084717625 * i,
				13.239048521941442 + 12.02294335248564 * i,
				-12.60693152200075 + 34.93872607037377 * i,
				13.17596719539344 + 47.71361724738763 * i,
				16.44485262755767 - 1.9942001369894733 * i,
				-61.973684465899076 + 41.99146307033121 * i,
				24.248103009912054 + 1.0207014475546146 * i,
				28.366187204841708 - 2.3171496575496704 * i,
				31.71959264147496 + 16.08855551349015 * i,
				-27.169807542209497 + 52.82111404714613 * i,
				-42.122282803850105 - 58.74781618493256 * i,
				-57.752280509432815 + 7.284212426969521 * i,
			],
			[
				-66.3279873275788 - 1.8991101943448143 * i,
				-24.490932556914757 + 37.118468211243886 * i,
				21.968925376946302 - 80.6048684014292 * i,
				57.28858918930405 - 25.484388947336825 * i,
				-27.66804642228334 + 107.7469776097434 * i,
				4.402264307312294 + 25.782017626776906 * i,
				28.93486890545188 - 6.5609859413404905 * i,
				-30.6349305085148 - 68.15550283497893 * i,
				29.018058990865512 + 16.541768415686022 * i,
				9.958899351824854 + 14.14354336518872 * i,
				-41.80557345955293 - 18.988902129024755 * i,
				21.254272446543325 - 23.08066655632483 * i,
				43.787133287316976 + 3.3077076511158623 * i,
				47.382308715830504 - 47.22324645356548 * i,
				-61.42179464158967 + 22.352434848341687 * i,
				49.68552676618009 + 19.806643719250232 * i,
				8.463665922416247 - 19.65507576151528 * i,
				-8.161775102357481 - 48.44041602242205 * i,
				-0.442570170128052 - 3.509124120360127 * i,
				-15.21717347775154 + 28.72830416618848 * i,
				-17.72148362340998 - 9.545556666831004 * i,
				35.80545731431546 + 43.75078725501401 * i,
				9.836997266490382 + 62.37665355633524 * i,
				26.7182215442692 + 4.092710622121146 * i,
				36.24275005894708 + 14.426583515276178 * i,
				-77.63961505005923 - 15.691949845042899 * i,
				11.080935829899268 - 33.09104058487998 * i,
				22.714057447362585 - 58.66909665002718 * i,
				-26.15289475696094 - 6.675114679344361 * i,
				-25.822401058042228 - 18.73365480241455 * i,
				-8.861436777714047 - 31.142694863705145 * i,
				-26.61698841463098 - 46.467885967743584 * i,
				1.1041568648386573 + 27.701123287840375 * i,
				17.00533329820324 - 6.169074730841629 * i,
				12.649830185075473 + 19.094825728962086 * i,
				-53.37786662093051 - 66.19814865977862 * i,
				-1.0607302921737443 - 8.940666879562759 * i,
				20.766419847622103 - 61.32621287836802 * i,
				90.97587538958096 + 71.11550915683125 * i,
				0.3187718267528513 - 4.201985602708994 * i,
			],
			[
				-19.805737025549604 - 5.208325447465597 * i,
				0.31431023298279115 + 17.57345110016665 * i,
				15.020414053405375 - 86.35787387314272 * i,
				59.81742217646246 - 25.958950162795645 * i,
				17.587625645985966 + 71.86080926767636 * i,
				25.948977884423073 + 51.0910505173687 * i,
				48.01559054707763 - 14.013171878173651 * i,
				-3.9093268551344806 - 58.85875429111641 * i,
				70.29161034968645 - 19.140258895939393 * i,
				-0.23242401746214014 - 20.61335621189218 * i,
				-45.344409165623546 - 12.73975918522649 * i,
				30.143309611652896 - 12.403507998655542 * i,
				2.018435302712554 + 4.2943779345065565 * i,
				19.19164382065849 - 52.61372197238887 * i,
				21.31977336560153 + 30.62545743667919 * i,
				-22.07158228641994 + 16.096672306733716 * i,
				-39.556746544208806 + 4.919357724770791 * i,
				22.985847789588192 - 44.25658216742957 * i,
				-2.499695217855148 - 13.815691373510905 * i,
				-47.09641188580742 - 0.14840756365668284 * i,
				-40.26968088721738 - 45.351232743936464 * i,
				5.797620583173099 + 16.282530721538993 * i,
				41.186060525058274 + 50.51805077943745 * i,
				-25.45223948736222 + 5.364713592566488 * i,
				-28.99884386962945 + 56.836353906080596 * i,
				-52.61586183608749 - 22.54149181157848 * i,
				3.5239078141733557 - 48.48696291270217 * i,
				31.768485349562958 - 63.72358117981572 * i,
				-38.67648658735493 - 37.16382824394989 * i,
				8.074229469293044 - 10.165821677951172 * i,
				-46.77881634259232 - 25.467342472454227 * i,
				-45.06160526599975 - 62.51865173385722 * i,
				-0.14965656161296437 + 14.177337481496965 * i,
				35.320033420204965 - 34.58020916527549 * i,
				28.046291005290044 + 22.252129419795857 * i,
				-48.73561459557655 + 9.832985776240193 * i,
				-34.141958567181156 - 6.5555250347912555 * i,
				52.311651966774576 - 33.87402178415848 * i,
				49.12454559367675 + 45.42335753651207 * i,
				29.122740303731824 - 91.76271421653193 * i,
			],
			[
				5.414098724080446 + 41.68796294731413 * i,
				28.545640779488302 - 22.287201492190494 * i,
				-71.04315500598483 - 29.153819716274626 * i,
				15.884070712360653 - 32.96708829840537 * i,
				77.03508291676381 + 59.45907505123171 * i,
				49.27873452310216 - 28.592784628956032 * i,
				-4.380689880942878 - 69.01393125785751 * i,
				-10.178158148457818 - 43.992093633417944 * i,
				-1.4898792950213249 - 9.970809902545732 * i,
				-24.017209940344458 + 16.535113487480146 * i,
				-14.5382532711063 + 21.924446899318482 * i,
				-10.467254289162941 - 5.229069360605479 * i,
				-35.49479256387173 + 16.337585714873228 * i,
				15.755488964285377 - 3.072172142843897 * i,
				-31.882125581190856 + 78.06332587030752 * i,
				-45.91070926831213 - 24.854303031458244 * i,
				-13.639973074322498 - 17.084964865092296 * i,
				-25.7400232041542 - 5.950695313478548 * i,
				-12.755608927118175 + 16.43769619418022 * i,
				15.950831566321781 - 14.283286929491105 * i,
				18.574690503961442 + 11.327761974629276 * i,
				84.05116215723527 + 1.2911958349593817 * i,
				-29.88764363155028 + 38.16939921088958 * i,
				-15.622838042928578 - 1.9214585344671065 * i,
				-29.89279979498075 - 59.5960505075242 * i,
				13.287486194140186 + 23.617407546074332 * i,
				-5.25545334277067 - 33.5523419481676 * i,
				-75.51646183716343 - 77.6702434872773 * i,
				-11.387888424839304 - 40.17668964486957 * i,
				0.04083343117059535 - 50.03659958761561 * i,
				47.745240111757866 + 21.87267587908528 * i,
				-40.05711401079681 - 36.19313403524751 * i,
				-25.458302247900207 + 42.59320203251539 * i,
				-18.32397262510689 - 59.62793216072172 * i,
				-4.871217444530949 - 22.302932083097357 * i,
				17.27230112579271 + 43.915911909014184 * i,
				-14.705889831186724 + 8.118022257365212 * i,
				-52.533275366172376 - 52.498803725611786 * i,
				47.65012693100489 - 59.3563178675229 * i,
				40.5881456750965 - 36.46999730264966 * i,
			],
			[
				-9.265054062054332 - 7.193124567303215 * i,
				-39.644737772612054 - 23.863446783431456 * i,
				99.74725454351395 - 52.45735446823076 * i,
				59.54932476592498 + 85.22196338208444 * i,
				-98.1400200113242 + 41.86685413318247 * i,
				-72.31232840600342 + 14.606925009218173 * i,
				48.950812842374944 + 35.25558342673759 * i,
				111.73495993723245 - 43.06151213744086 * i,
				47.33830033705326 + 54.87769924029544 * i,
				4.752725333386024 - 34.00921543376623 * i,
				-26.5219485544313 - 80.58375713204532 * i,
				40.60299112056124 + 24.883668281597934 * i,
				16.93898383177509 - 38.61847143885106 * i,
				126.26424796567116 + 24.439357205014517 * i,
				-25.83009322314966 + 29.331401364819186 * i,
				-46.35024982346958 - 40.33162027889885 * i,
				-41.10249390592375 - 54.8723046909883 * i,
				90.55336675900139 + 3.604362988281707 * i,
				-14.267622930411441 - 14.932037279240262 * i,
				-71.9691811498256 - 44.54434186277761 * i,
				25.268159468702144 - 47.485502909934425 * i,
				-18.99628190666465 + 8.782502651773278 * i,
				-65.52973644217079 + 98.16615520466057 * i,
				-21.471076718890913 + 47.85731932086043 * i,
				-7.114580559043844 - 36.627789719438624 * i,
				-45.92557021869452 - 41.31693351251518 * i,
				97.6657684683403 - 75.29201391451275 * i,
				59.362052614023575 - 47.64705820600847 * i,
				35.95988704102839 - 89.7771264218949 * i,
				18.175275189571806 + 13.762069742893829 * i,
				-14.997870697792067 - 52.565725984043596 * i,
				67.93501081146412 - 76.8506949628985 * i,
				-78.42909496695391 - 76.59966414544844 * i,
				69.66100880026221 + 23.94621716217453 * i,
				-87.81308544577516 + 0.8106213250390084 * i,
				-36.52408366509986 - 17.3117797545045 * i,
				-22.4879033610406 - 51.22558804295195 * i,
				123.95774254619954 - 42.572088887758746 * i,
				-32.72790885753172 + 125.8123906021491 * i,
				120.36613751970275 - 17.505967586628895 * i,
			],
			[
				-63.4172918043232 + 23.643408585946737 * i,
				-5.637806927164945 + 10.513190037920328 * i,
				-30.665619467577528 - 39.12238136949699 * i,
				28.690557386557373 - 44.37979455900687 * i,
				-27.049642766600073 + 49.13733791869811 * i,
				4.3768241953005 + 6.10388031624754 * i,
				-27.226164204067402 - 7.731805594304305 * i,
				-28.622939315785572 + 5.1255891512712175 * i,
				-11.647398662467225 + 15.705096593503754 * i,
				-3.4171476604665543 - 7.7884183906679425 * i,
				4.575711535060009 - 19.110255977904046 * i,
				28.679301519517765 + 3.2184802433691573 * i,
				40.55403108981514 + 14.747876912577645 * i,
				18.01265278446796 - 30.72882494543458 * i,
				-42.878724576583835 + 36.555775533689044 * i,
				3.92076755352727 + 7.3216194909402645 * i,
				48.26812225309495 - 17.220098355508085 * i,
				0.3508387318601933 - 34.36924820951731 * i,
				-6.225576138483191 + 14.39726521928884 * i,
				6.131087427928653 - 20.877239463180217 * i,
				4.627537219407635 + 10.305688900433196 * i,
				30.118247984530115 + 39.78553462456118 * i,
				5.434731637987738 + 26.198855592814702 * i,
				10.639386140092787 + 4.650062100940081 * i,
				4.510100091966578 - 25.333697646527536 * i,
				-26.906063317569505 - 26.88632976983982 * i,
				13.429777882023863 - 16.827733282021754 * i,
				36.41495617676364 - 44.61975283537263 * i,
				-30.475650955107323 - 1.8355160308553788 * i,
				-7.709366548138862 - 7.7115336317871765 * i,
				36.06655662033849 + 0.34890109839811334 * i,
				11.342389947889274 - 18.61426280838774 * i,
				19.494629090255646 + 6.139694102798526 * i,
				55.83578237186073 - 34.15917651568593 * i,
				28.89310090018619 + 3.0629170565852064 * i,
				12.887892664257109 - 11.912467298674114 * i,
				-11.175083456349014 + 5.781246203882176 * i,
				-19.544835336877874 - 47.77960752248377 * i,
				15.443927395370082 + 14.401841553878505 * i,
				13.66220938411311 - 7.839141580448219 * i,
			],
			[
				-24.554856510313897 - 14.883797274976587 * i,
				-3.633038808323711 + 36.96289741139053 * i,
				15.567827736793632 - 57.02637280489836 * i,
				8.759301132331586 + 4.820714938466011 * i,
				7.273067195121081 - 45.279764863892495 * i,
				-16.227964312749364 - 6.498135865864299 * i,
				-11.20054621063079 - 8.426273125267045 * i,
				-58.705248303116484 + 81.65445509008538 * i,
				25.594931889391546 + 39.35274872626501 * i,
				30.532309611036982 + 3.256923448612241 * i,
				7.129379172792408 + 6.838738947153601 * i,
				-15.190456636557016 - 2.2640782849479066 * i,
				31.910440308615442 - 15.29693075177645 * i,
				-39.124952616618685 - 2.837702414107156 * i,
				-20.999209101849488 - 11.697633954717489 * i,
				7.5884728774133805 - 41.05982235596716 * i,
				-18.57945603720178 - 8.51797072596581 * i,
				3.7213551064906056 + 38.28470046120539 * i,
				-9.591623141043726 - 6.440141500678379 * i,
				-1.6077782794851103 - 35.479927056383836 * i,
				42.07717542912572 + 15.97520671619767 * i,
				-5.062921415966187 + 4.00486534658808 * i,
				-22.060815068513847 - 19.366702158711433 * i,
				-4.61351673044696 - 7.801948348984595 * i,
				28.490431520610592 + 22.881853616671908 * i,
				2.082108560153415 + 23.555511363258546 * i,
				38.875229607076975 + 32.14117251388956 * i,
				-28.368058605991724 + 53.309435390115546 * i,
				75.01512549490411 + 25.092503280592197 * i,
				10.167320629547305 + 24.706753461420604 * i,
				28.25935236432018 + 16.124482278499286 * i,
				5.346452189736027 + 24.95298955330945 * i,
				26.596477084769305 - 40.68761542016499 * i,
				9.489041601049749 + 18.962661059822352 * i,
				39.96535074275272 - 1.2938098402726794 * i,
				-73.16794757765041 - 12.605417296973222 * i,
				47.09778621253232 - 31.7273897555976 * i,
				42.554318879901345 + 12.433245937648222 * i,
				-55.642285416444935 - 20.78705673417153 * i,
				-47.43055926537415 + 36.31857442850313 * i,
			],
			[
				27.170665240079952 - 12.556095234336539 * i,
				-54.796414924358324 + 71.20064916969002 * i,
				4.3904040032280705 - 80.10425016806684 * i,
				82.12082311686282 + 11.615571692954191 * i,
				22.878100587954613 - 1.5123862064746447 * i,
				10.059505130723675 + 21.473756998667454 * i,
				37.051288224050026 + 12.918762360647719 * i,
				-47.55183198492146 - 42.13334204934497 * i,
				33.81571579732643 + 41.79468302461305 * i,
				9.280509483078077 + 14.190236397618918 * i,
				-65.29781393332239 - 8.127088650560047 * i,
				-26.271455102674224 - 6.215384518307843 * i,
				-4.096022331183416 - 20.815918724286462 * i,
				9.220668537791155 - 57.85279232060919 * i,
				-29.001666407455218 - 29.390835910296005 * i,
				21.082981234737193 - 19.1189853351367 * i,
				-30.530065955545115 + 5.49351185024608 * i,
				42.92144374611563 - 13.904528271157389 * i,
				-1.7137581454371116 + 8.110292457513353 * i,
				1.5428073825626427 - 10.45923563767019 * i,
				26.18329088031431 - 11.084491516662247 * i,
				11.791975237320372 - 43.78954881413137 * i,
				21.334090450633063 + 55.28528187203936 * i,
				9.440243098618524 - 15.192376986184797 * i,
				-42.21095756994187 + 18.07856404275178 * i,
				30.177284952664458 + 3.0202096681995556 * i,
				3.8781580412827594 - 55.493058979317254 * i,
				-19.869966363493745 + 28.16208639266803 * i,
				58.51587627377572 - 55.21899485133818 * i,
				-44.44322955925524 + 30.815288659789857 * i,
				-96.1772076600449 - 19.93020592818185 * i,
				-26.991885704297566 - 55.550070956044095 * i,
				-45.44996489801197 + 57.635569015953465 * i,
				22.792999460267964 + 26.169910861595554 * i,
				13.107909898343834 + 70.94708603496012 * i,
				-95.98802754887068 + 3.049231197007856 * i,
				40.91088752351679 - 68.41664164478816 * i,
				15.45530001031669 - 15.004263586634355 * i,
				13.959846303043523 + 11.136256506055993 * i,
				-11.046465707001419 - 1.588928127033379 * i,
			],
			[
				16.694937908759982 + 32.35098891155559 * i,
				59.58450298945323 - 17.588123257737763 * i,
				-57.19287463402591 - 19.693477280585597 * i,
				-38.36807028534648 + 5.150458733922619 * i,
				24.639598018793844 - 10.70597729079492 * i,
				64.73764787481088 + 8.022656628091681 * i,
				41.50101846234429 - 47.872298041759365 * i,
				-28.3368956849107 + 28.94393300992462 * i,
				44.46017744402893 - 4.73642810937406 * i,
				4.846981346201986 - 22.02271660214808 * i,
				-12.166556355382852 + 54.97920913572901 * i,
				-14.852165025828697 - 15.929430829901747 * i,
				-84.08528882512249 - 39.5732081530151 * i,
				-25.853431157994123 + 0.8186941254087117 * i,
				29.839729219522667 + 28.56997070746624 * i,
				-12.404841510025719 + 3.9695958404289513 * i,
				-26.30152166387613 - 10.69108559787945 * i,
				-45.56899641543097 - 13.946238916469355 * i,
				44.09282277217365 - 13.370683425033999 * i,
				-23.67968383791154 + 1.080502487774993 * i,
				-4.242496384247204 - 1.3752871618532936 * i,
				13.792895233618346 + 25.238079081726895 * i,
				-17.134127947347395 - 20.876168516954458 * i,
				-19.06215887657048 - 22.745202476169183 * i,
				3.771419499425022 - 77.83657704454501 * i,
				-24.862418776127598 + 73.45537930637539 * i,
				-23.05465673112218 - 3.1273428245579424 * i,
				-40.9647247314676 + 9.726875028627546 * i,
				24.837833623859037 - 40.935606876343485 * i,
				52.53679709265559 - 21.58775635950127 * i,
				-8.898921069358932 + 30.250469340551717 * i,
				-29.302235595967552 + 26.226655083813192 * i,
				-74.71142149564812 - 39.31509795796596 * i,
				-12.01909477463233 - 37.02748653348231 * i,
				-4.261495354198544 - 71.1757307089134 * i,
				-56.46124362442106 + 90.65196142513504 * i,
				-27.86888053917925 - 7.6810070995144315 * i,
				-14.194155098250015 + 3.911129583745481 * i,
				63.653576144156965 - 43.67448655444909 * i,
				-0.028902572134995086 + 35.39599939277515 * i,
			],
			[
				-28.845101788058855 + 19.25797643943614 * i,
				34.92642143706021 + 20.01596879607112 * i,
				-22.528088608456812 - 84.12744163284049 * i,
				21.009284616039267 - 17.783449125416134 * i,
				-7.781933349187227 + 72.38625093079267 * i,
				10.33633422251464 + 37.1973388621351 * i,
				19.781474517693198 + 8.660447114953058 * i,
				-8.855128705036122 - 41.327129452569274 * i,
				36.20240900791313 - 7.871742563751752 * i,
				14.848431028682661 - 29.51134587533103 * i,
				-28.168184193770678 - 34.000509762169344 * i,
				21.88120179274705 - 23.474615376217287 * i,
				34.76203191575519 - 2.31859640880608 * i,
				23.5078165705256 - 33.46445614970884 * i,
				-9.48766141740279 + 55.17797814056428 * i,
				-36.23765714002439 + 33.78927596576293 * i,
				-1.177646812505202 + 6.963408321941131 * i,
				-11.579281603271411 - 47.828045148921206 * i,
				16.068533753227484 - 1.7391308186464656 * i,
				-24.564560045435382 - 0.6847242649798009 * i,
				-42.52338292157252 - 55.98677691142332 * i,
				36.8435562287044 + 61.19117262748678 * i,
				-3.2604479519230694 + 39.40915738243291 * i,
				31.805559384793042 + 17.389559589589226 * i,
				35.65190965281982 - 24.464533456828335 * i,
				-43.54287335196493 - 17.78592834993293 * i,
				18.562007283763535 - 42.45709136127148 * i,
				22.78372053982021 - 33.49667320619335 * i,
				-21.472101599947514 - 20.988601565137174 * i,
				28.464273424633657 - 31.31610973594841 * i,
				-13.040794751555108 - 48.276384738206644 * i,
				-17.793985590135662 - 28.12870230394705 * i,
				-42.72327372531533 - 17.220626181899483 * i,
				11.834697166104021 + 0.9055628710653778 * i,
				20.163700128886546 - 8.891485690047872 * i,
				-45.72025091136536 - 3.2860179051443126 * i,
				-14.226667434676937 - 46.22507940014523 * i,
				-7.872488897838053 - 52.25761441273042 * i,
				76.96436338609321 + 64.48738388326561 * i,
				14.916696406972425 + 10.203998720829462 * i,
			],
			[
				-27.98023104835636 + 47.71260636884574 * i,
				-6.173969076661976 + 2.9826954568633255 * i,
				-42.035741213743734 - 19.88600494870252 * i,
				-39.75939309507565 - 66.19661193460844 * i,
				25.806983801193297 - 41.780807207724536 * i,
				12.840981610590632 + 7.595343035075587 * i,
				-40.3828546724795 - 22.950170466801765 * i,
				-42.795749059321636 + 33.48237894440735 * i,
				29.37928171083361 + 6.386368996438382 * i,
				-6.106914920354413 + 29.79934715783088 * i,
				-12.289068815045704 + 50.00179456308443 * i,
				-12.60372198937523 + 12.765321597435277 * i,
				-0.41042438046834206 + 10.858892918776732 * i,
				-14.83922683767446 - 0.5874931392624418 * i,
				-45.976567701791325 + 44.51452256470078 * i,
				5.350361252925936 - 7.871073670139097 * i,
				18.830729797561467 - 2.5864553580188243 * i,
				-12.537965954877373 + 26.05301180100858 * i,
				-13.432038552196815 + 25.384018105059653 * i,
				-2.0635012007511193 - 35.1340434715115 * i,
				38.90763400227098 + 11.405415000940977 * i,
				15.775734645695408 - 22.1108616109327 * i,
				25.07657991447217 + 5.359416544159636 * i,
				-23.27556499783445 - 35.633372528500615 * i,
				9.58824544592715 + 39.76426122048282 * i,
				13.100803124767552 + 23.354573791510305 * i,
				-17.479969795265532 - 1.7604542039994584 * i,
				-17.87786848672522 + 21.36451048913001 * i,
				11.200098755481164 - 7.397982752736169 * i,
				-36.75297391150074 - 8.570488941670053 * i,
				19.263049029073052 - 5.838585847264191 * i,
				-31.653827720272794 + 5.7399431094085145 * i,
				86.86899230543679 + 3.1050839811657767 * i,
				-4.359950790296136 - 4.620673313676676 * i,
				29.639063730518476 - 23.11899280867528 * i,
				-23.292567964141192 + 21.517859688796506 * i,
				-25.151994252398687 + 36.35196796958007 * i,
				-9.647998879618468 - 59.79797788238365 * i,
				-35.46974454934007 - 83.10534482290342 * i,
				-50.72760264145238 - 34.3455985465426 * i,
			],
			[
				31.061609454305326 - 29.982574064324893 * i,
				-21.845882840250297 - 39.657540961403285 * i,
				24.1830156232553 + 15.874587863579421 * i,
				8.523302730100326 - 2.8506122645006666 * i,
				-14.073951796987899 - 49.511734320679935 * i,
				-14.301237095263687 - 20.094787825367597 * i,
				-26.969509253315323 - 39.6617573539908 * i,
				-8.53903462515008 - 11.696370441340434 * i,
				-9.998365193816737 - 17.378834620143465 * i,
				-33.957716514297715 + 7.492199110962524 * i,
				-7.143637815162151 + 21.380497465169015 * i,
				4.148270665619265 + 7.843765276705312 * i,
				-1.7514578270855168 + 46.24133702157342 * i,
				-0.07590106341558567 - 1.7905497655463378 * i,
				-34.2729284550756 - 22.71972591486523 * i,
				-14.597580454845925 + 12.745835842762517 * i,
				0.41146362103899037 + 32.13016259921736 * i,
				-27.056906117125152 + 28.739236913099774 * i,
				-29.186462319235368 + 11.445737783303358 * i,
				52.14178702879578 + 5.720325112569697 * i,
				-11.340086196141243 - 8.278916884636203 * i,
				-30.55814948250108 - 64.24876972888782 * i,
				26.326976792042384 + 1.9536759502548549 * i,
				-20.12155501592609 + 9.628804922094261 * i,
				3.442817109177719 + 53.26166320556354 * i,
				49.94824006452059 - 18.91089476253643 * i,
				-36.21911882543711 + 16.322493482051293 * i,
				9.202974380758093 + 15.006038634595484 * i,
				-23.767091741794076 - 18.075591007583263 * i,
				-8.161002955329156 - 38.80252736674977 * i,
				-14.808732447344164 + 24.489486093138993 * i,
				-18.1021055687359 + 8.155217567964641 * i,
				47.99524724267755 + 92.1246534582509 * i,
				-34.64282051412141 + 36.62787554452396 * i,
				-11.243513685868962 + 17.642478518103907 * i,
				40.03979446247281 + 10.464455848259675 * i,
				-3.5107303425582543 + 22.913497572326776 * i,
				27.59106298858709 - 20.055363813689038 * i,
				-53.210164482369315 - 35.74054372688787 * i,
				-38.621319943268546 - 39.26316810446023 * i,
			],
			[
				70.64727518782524 - 19.813928938979018 * i,
				4.510047202811606 - 41.247131792208535 * i,
				-16.215943311219846 + 72.91717556735546 * i,
				-82.61986164376333 + 39.030812054520254 * i,
				66.71552377848673 - 32.41885749369514 * i,
				-25.99871070785241 - 43.79142431325326 * i,
				-41.32221635631794 + 84.86269363218929 * i,
				17.67021953680451 - 4.877443772319822 * i,
				-1.6842449872742833 - 15.497490601950048 * i,
				-15.847438088014144 - 22.627855663788637 * i,
				27.999069747754664 - 19.463122190449383 * i,
				-10.287313872068708 - 13.65105199610798 * i,
				-5.515236333988703 - 4.576820606648724 * i,
				-20.875478086962374 - 0.8085764258052137 * i,
				103.11947491894982 + 14.439503825250014 * i,
				-6.622383019570748 - 7.046189678836512 * i,
				10.339952422149807 + 9.830602025527725 * i,
				-22.73731262825009 + 9.918419119507945 * i,
				3.2376472184436125 + 25.187871320947792 * i,
				13.425098037104192 + 36.95360869137306 * i,
				-11.79593176408795 + 21.975524140837862 * i,
				13.500104744931157 - 36.65537402251556 * i,
				-8.650711476059811 - 35.57439592624377 * i,
				14.74792136867815 - 12.306951331547197 * i,
				-69.92105462926628 - 50.974434705713534 * i,
				22.291297160275374 + 28.87998717879787 * i,
				-37.12770106166326 + 19.24414179298026 * i,
				-59.97293980170082 + 45.89533164049446 * i,
				2.2135246357886995 + 19.706240376447454 * i,
				-5.012910312883321 + 48.478271975831845 * i,
				-27.96144766288748 - 5.4073950983638355 * i,
				7.248699558000453 + 13.525831727331521 * i,
				-15.315082168597662 - 34.53612224301531 * i,
				-25.358848856510605 + 25.985390108276604 * i,
				-35.41363166856526 - 4.456065105876171 * i,
				38.276605990528985 - 54.1651578413959 * i,
				15.19260835620417 + 5.250999879463517 * i,
				-43.09889668807442 + 82.83674858289275 * i,
				-35.58146808553647 - 26.599076384791903 * i,
				-28.787644716899624 + 30.423249671622642 * i,
			],
			[
				-26.616024472284565 - 16.39271706569957 * i,
				-23.60768274685438 + 0.10954284154638572 * i,
				-36.3324069618119 - 15.164318214992875 * i,
				19.528144719280355 - 42.96832031847201 * i,
				-19.665880114151804 - 13.45134308190626 * i,
				22.168405023946477 - 17.618744431707654 * i,
				-3.289623151610897 - 44.813845348256855 * i,
				-53.30338754855228 + 11.5206870662727 * i,
				-25.34531839054138 - 34.279102290595915 * i,
				-18.86428449633742 + 16.698256078367564 * i,
				-8.984186136973449 + 33.83826894049395 * i,
				-0.13019701895882463 + 1.0481174434116483 * i,
				-30.152763177609884 + 14.005009580601438 * i,
				-12.643398732554118 - 22.882399704005707 * i,
				-37.52707603281792 - 39.41273814662171 * i,
				-11.451228842058853 + 2.9654527050602866 * i,
				21.260928401659953 + 34.424715920911524 * i,
				-28.80138331068613 - 29.57171168335239 * i,
				-9.855973350504076 + 2.113599576903468 * i,
				31.389424704406018 + 0.9144861848154733 * i,
				0.3635629124218269 + 14.057599245150186 * i,
				-27.390143326001294 + 37.931107340891394 * i,
				-3.5122406635850325 - 41.365891755651944 * i,
				5.141443596205384 + 20.609989256903 * i,
				12.724380321528542 - 10.256604954770694 * i,
				-4.1986978418806835 + 30.89308254422635 * i,
				-20.196996131844394 - 9.324117018247467 * i,
				4.811255738635941 - 37.38392355210847 * i,
				-45.18228730548884 - 7.539414761530132 * i,
				21.780191513020654 - 13.74894299926207 * i,
				-16.554498368276974 + 86.80159588061107 * i,
				-6.374545047331367 + 8.229687408800192 * i,
				8.338980649240055 + 52.881973081765324 * i,
				29.290763770368834 - 20.98043791576805 * i,
				32.596295821762084 + 20.777228055729083 * i,
				28.12142407080193 + 22.91200177451114 * i,
				-20.92089482541423 + 40.21180025265104 * i,
				3.4974520314650626 - 65.13308592551193 * i,
				46.75542915917377 - 59.8234339493441 * i,
				-25.880766298253974 - 13.01657873028351 * i,
			],
			[
				-35.911236681080936 + 33.05144430354082 * i,
				9.736188448315033 + 89.13797850462689 * i,
				-33.269483796810775 - 84.19938822497977 * i,
				45.590422636659255 - 98.81844725124243 * i,
				42.621975033346004 + 21.776375248513773 * i,
				10.819517842108326 + 61.349321053923234 * i,
				42.54819880016214 - 4.5392198541783495 * i,
				-74.66389985085382 - 79.18200179595712 * i,
				10.16370515621179 + 16.439041183999564 * i,
				-6.985967833533152 + 37.45890114115272 * i,
				-85.96152674754991 + 10.295160438551164 * i,
				-3.916921843335217 - 14.740493624104829 * i,
				34.16562864565531 - 9.238460682164828 * i,
				33.87722406234178 - 76.47695157100483 * i,
				-70.14140594635282 + 64.18039900738282 * i,
				-24.698207675408554 - 12.962532497068636 * i,
				41.09837903165564 - 27.709046535392424 * i,
				-14.403079491942279 - 68.0984946618311 * i,
				-2.7958278754045978 + 50.27832929114754 * i,
				23.519801532512616 - 26.622662072251618 * i,
				18.427948090775573 + 7.255552241037108 * i,
				3.102623604262959 + 9.748274537377426 * i,
				48.98056695708056 + 37.210974403952775 * i,
				32.61155229653296 - 23.896638081068545 * i,
				31.205301661878977 - 39.11520397705354 * i,
				-63.84165390798225 - 38.06182122634158 * i,
				-24.183893729087075 - 70.10835534549456 * i,
				-26.558359310032287 - 65.26297799832406 * i,
				-6.357751433220766 - 7.5881084433959956 * i,
				2.4699228094136174 + 44.398691726332004 * i,
				-61.13651873468991 + 1.3152164425419528 * i,
				-46.40006827518259 - 48.10506942733605 * i,
				29.249845142800204 + 17.236564127472185 * i,
				38.17492204415657 - 20.64835183096131 * i,
				58.584702661255974 + 35.70227539043968 * i,
				-27.552443546518173 - 6.816296278022237 * i,
				-45.98556255229736 - 13.968918808798218 * i,
				-40.10934088413354 - 76.65613490629775 * i,
				111.35268226675284 - 6.600426409464494 * i,
				17.03569293628805 - 41.4781521290674 * i,
			],
			[
				16.214875310535888 - 1.0724267243571148 * i,
				-1.2175018229823422 + 2.7088072238354357 * i,
				25.712837063709046 + 12.056478073843266 * i,
				-13.824876703280893 + 13.68132915693269 * i,
				6.73747183692231 - 31.879603221474866 * i,
				-20.536063196492528 + 30.49659048284542 * i,
				16.184996314777223 + 58.282872757812456 * i,
				31.540062949189462 + 25.088258377361996 * i,
				26.12414827238888 - 23.45788733746717 * i,
				-13.500060573403374 - 26.532713448756454 * i,
				4.438353428346154 - 10.637674000328104 * i,
				-1.3148910518464945 + 3.053668778943745 * i,
				4.643112596184949 - 10.605939204056384 * i,
				-1.3161602098984417 + 12.846876125969828 * i,
				26.1325855349276 + 10.792572692975142 * i,
				11.758899854756162 - 23.60899399473149 * i,
				7.59523311524903 + 33.58634798965073 * i,
				10.62605508702427 - 5.020765936428859 * i,
				-19.409845458041076 - 2.079912377237136 * i,
				-27.987096504664954 + 13.986905371679399 * i,
				-38.67176501700948 - 9.597283438792118 * i,
				-44.74208930382067 - 28.691838983368612 * i,
				20.2847847581713 - 9.852199778225142 * i,
				-23.99842719130922 - 11.984757276141224 * i,
				50.86313088223632 + 18.9147544608918 * i,
				-15.22796802515243 - 20.936603412773977 * i,
				-2.551980907481095 + 13.750656834440633 * i,
				23.64730329698373 + 4.137171749865814 * i,
				12.321853118444608 + 40.58102736784011 * i,
				-3.0435978905078382 + 44.10252317856987 * i,
				-45.28868462612712 - 19.357610910416774 * i,
				0.6119244754005919 + 12.029315132191982 * i,
				45.667928697568854 - 33.741916694005255 * i,
				-5.319170489857797 + 44.999887642295036 * i,
				-2.499749409194152 - 10.916740184609155 * i,
				-14.906124868709972 + 6.5580284485454605 * i,
				-36.41006902488422 + 15.277779586670043 * i,
				42.06145166917529 + 36.43943903686288 * i,
				-5.278497558550997 + 4.614313120775938 * i,
				-0.7814369150594511 + 0.2554244406303729 * i,
			],
			[
				-38.559415418246175 + 12.942428044479612 * i,
				9.142765199206991 + 36.553775652377794 * i,
				-19.85096478878062 - 4.318016595992248 * i,
				12.190749168705182 - 22.987857748099287 * i,
				21.840109291575317 + 102.78794891976389 * i,
				-13.504709728044475 + 2.142167431511652 * i,
				9.329612506839624 + 5.9791183694992185 * i,
				47.2040226762341 - 2.605698599147786 * i,
				15.56954035836074 + 49.40378998181629 * i,
				-21.005578507218218 - 5.138602909081655 * i,
				-30.7328703208926 - 48.07580062401752 * i,
				25.1642107052794 - 18.96334613309176 * i,
				24.619673970713073 - 75.16864353303332 * i,
				-21.47301367458084 + 30.123899598993397 * i,
				-31.929740929145616 + 7.539239865258523 * i,
				21.751878784754545 - 45.643454184115285 * i,
				-14.310567256383706 - 56.177723993010986 * i,
				-4.765464053981503 - 29.49385069913305 * i,
				-2.102968718688836 - 3.874706106767583 * i,
				2.7761122452965177 + 28.071372405888624 * i,
				19.603516172528003 + 15.961310621952464 * i,
				54.98958270557935 - 17.183005594293718 * i,
				11.473452392950831 + 22.91902137950231 * i,
				32.12636375610826 + 0.9896583067368567 * i,
				-30.660521884492894 - 19.382665446110735 * i,
				-46.81138014622253 + 4.778357108323689 * i,
				3.985877310529276 - 23.005596029742524 * i,
				-27.675886268755065 - 68.89153107353259 * i,
				0.04121876604281738 + 9.128698228551052 * i,
				25.676929551387616 + 35.06199647191609 * i,
				35.00948610031627 - 10.736934754580133 * i,
				-15.399135276845492 - 29.930459647805677 * i,
				-11.960176794253854 - 4.047928856890504 * i,
				5.468413302853591 - 87.36958177430682 * i,
				13.395299669989463 - 1.5507017166423083 * i,
				-33.708134856458514 - 7.56223202648269 * i,
				17.42584184797967 - 48.03781762197893 * i,
				-10.631864035714258 - 5.162567314521063 * i,
				27.642922199445273 + 65.48981418605682 * i,
				42.25552155293254 + 14.38359400373475 * i,
			],
			[
				-39.74533487284896 + 0.5445025071844896 * i,
				28.079032266441217 + 67.83719558565372 * i,
				-7.745842299621717 - 37.34131323266418 * i,
				20.684394815425094 - 40.16839701400792 * i,
				1.2521073165266472 + 16.878340365545085 * i,
				-21.955146307887063 + 33.54042447125204 * i,
				45.32793396552211 + 0.6448199565931816 * i,
				11.748782516278958 + 84.18256863809054 * i,
				5.270064106534818 - 16.728028134446276 * i,
				19.406918976160643 - 32.164294155406154 * i,
				10.983903372672785 - 31.546088847808463 * i,
				24.760937291416667 - 25.825895702702173 * i,
				49.26315628318396 - 13.32639343179136 * i,
				-72.70359489041586 - 2.636633619634921 * i,
				-26.13254889590239 - 0.872341050643783 * i,
				-17.83020924729837 - 25.13283965335107 * i,
				48.62355935395584 + 22.671410546387996 * i,
				0.16697213693643675 - 23.393331762826854 * i,
				-22.642687414212745 + 10.314617771066564 * i,
				1.1686406257963657 + 33.85049760797048 * i,
				-44.39233301827559 - 9.29769271427234 * i,
				-15.193516550214447 + 7.078856600978014 * i,
				16.4681905744229 - 23.869983958322162 * i,
				-17.93439725327398 + 19.134362760056252 * i,
				136.89688933650874 - 20.467521407869185 * i,
				-40.02404380255941 - 47.06372727910638 * i,
				17.703259063836896 + 6.662088795591025 * i,
				59.000970780083904 - 36.04648618802981 * i,
				20.869564664254753 + 40.99031356571227 * i,
				56.79353821392159 + 45.22335695615601 * i,
				-10.234669566495938 - 7.0703637012281035 * i,
				-11.557975076767814 + 37.75254658426536 * i,
				44.424949082348746 - 45.81242200521755 * i,
				26.546900593308628 - 16.156845201667977 * i,
				82.67814513583318 - 48.13434628739155 * i,
				-69.10680072907859 + 24.74127143297195 * i,
				26.360739346892938 - 33.01872647790613 * i,
				29.666989416691724 - 9.815565774496136 * i,
				10.320318543801546 + 61.397424055437554 * i,
				-6.521250642882733 + 78.74116218338078 * i,
			],
			[
				-23.048044587204572 + 13.326549450402183 * i,
				-1.1699234827022842 + 44.12482141577624 * i,
				-15.10516257001295 - 50.66351405555281 * i,
				57.937216114269546 + 10.791749475549928 * i,
				-4.469639633127102 + 44.05671119152047 * i,
				-6.376906804250588 + 23.81884970256685 * i,
				27.725709784698026 - 49.1620061239082 * i,
				4.176641060400726 + 0.6464002487200275 * i,
				13.460674888441417 - 4.480623237231628 * i,
				17.554227402528227 - 9.897501400574992 * i,
				-7.964378181673624 - 15.217288345624098 * i,
				6.479502818844253 - 13.814989175174908 * i,
				9.556368269583958 + 19.847230831525792 * i,
				10.879449994566329 - 25.202807298789338 * i,
				-39.39093830306417 + 23.841015899006862 * i,
				-16.345785873552813 + 1.2753077466940113 * i,
				16.055484478638277 - 13.74694620863329 * i,
				29.863143661460214 - 24.280133361320452 * i,
				-31.513676699500905 - 8.702226929316467 * i,
				-9.61343763093096 + 26.568582786749843 * i,
				-11.456120450732797 - 4.297967821472881 * i,
				26.578795953312586 + 17.3997017343943 * i,
				-1.4641745569479405 + 34.99654704059122 * i,
				-6.358253582428091 + 26.19837602791326 * i,
				-1.2016705057653922 - 53.32678615696298 * i,
				-4.693810507867163 + 12.82631301087425 * i,
				9.396442813388873 - 42.95623224345959 * i,
				18.302002065393815 - 44.63612499784425 * i,
				6.090663065732684 - 8.26834910914998 * i,
				21.21315983822064 - 5.50935088567384 * i,
				-20.473511881710515 + 11.910432138055604 * i,
				12.785537857884073 - 5.503030671231587 * i,
				-39.02553711229416 + 13.943670609251427 * i,
				38.568929016186814 - 13.14652499505437 * i,
				1.0506272463010262 + 5.0864741617094555 * i,
				-22.904134807017833 + 29.667996505059655 * i,
				-11.871998337822989 - 10.662004796498264 * i,
				-1.473550341460296 - 55.66331308160437 * i,
				32.23751976669902 - 1.8037521949535282 * i,
				60.66987105716622 + 2.247956055083236 * i,
			],
			[
				15.350183672341998 + 2.7201853965533758 * i,
				-83.3560218085059 - 1.2306126953306062 * i,
				28.34594400620116 - 3.5001061953892503 * i,
				46.72230561113287 + 51.07041324388919 * i,
				-25.52139352481271 + 20.99370718754467 * i,
				-14.376010148098942 + 5.728520613254915 * i,
				-34.26689150093324 + 45.496566681058084 * i,
				3.211348131750583 - 27.224189805825574 * i,
				-3.4486501105903073 + 29.166126802143353 * i,
				12.236608352752697 + 27.043983515978027 * i,
				17.184838642122145 - 28.784977246786838 * i,
				-14.30794894635764 + 6.796897922195278 * i,
				14.233855521871703 + 18.46968895830871 * i,
				35.160284044472796 - 27.868852426964494 * i,
				11.73279732500393 + 6.112908212323589 * i,
				19.98542506056427 + 19.422317496341435 * i,
				-6.927182941043853 - 19.066346054722256 * i,
				79.27644396299596 + 28.640632496474932 * i,
				4.081681904859238 + 15.16683507745256 * i,
				-12.290938007901019 - 47.72803559118711 * i,
				25.722632083494616 + 8.202532983791777 * i,
				-17.488068302703947 + 3.9432122509485623 * i,
				-6.073511421162721 + 19.886652048463418 * i,
				-0.6190591904074205 + 12.829610333257648 * i,
				-84.82576015172695 + 17.59172454493556 * i,
				48.979153985481375 - 16.948919939436934 * i,
				31.313781113398395 + 7.30445840561228 * i,
				22.018425323788392 + 28.730241961713823 * i,
				20.026380781897018 - 21.614972972953368 * i,
				-74.75176907692422 - 19.018269613101122 * i,
				-5.3819240698218636 - 25.223950278872024 * i,
				56.43115044031224 - 39.34062535061166 * i,
				-8.459310141238303 + 12.643324391849191 * i,
				43.50267413764142 + 43.61090410813755 * i,
				-41.742690367105254 + 64.72351745300965 * i,
				7.740630291303205 - 62.777596487973355 * i,
				43.77541448080712 + 1.2310675403849487 * i,
				-23.51143718595857 - 0.7458719676291139 * i,
				-55.7329837712228 + 32.316729046378455 * i,
				14.122152539214731 - 11.581426351539063 * i,
			],
			[
				7.137854601513979 + 12.140758830626591 * i,
				50.37367228721254 + 3.0286828199253364 * i,
				-19.895604808955227 - 5.582498989726567 * i,
				-23.997960235766698 - 54.68724692257891 * i,
				46.673582534028895 - 23.663080969831412 * i,
				62.04292495228238 - 0.3232875978346499 * i,
				-21.376731305208935 - 18.459634772812706 * i,
				-27.948442293029487 - 0.7227838836425278 * i,
				-13.885776109743224 - 18.279228619246524 * i,
				25.829343846121503 + 18.98997355212488 * i,
				-1.7197580589704415 + 35.040110167211964 * i,
				-8.689673198829325 + 6.544442931010394 * i,
				-34.93692246156019 + 17.30886761220996 * i,
				-12.948439907504053 + 17.148388881227874 * i,
				19.901178338239802 + 6.095744106949766 * i,
				-15.258506336684576 - 16.306102670659836 * i,
				-9.031735313935904 + 12.10983424967813 * i,
				-15.4227620881354 - 12.015947251869704 * i,
				-1.9439000092110348 + 14.48329690533172 * i,
				30.37940028720284 + 4.240602992028872 * i,
				13.788556659642211 - 0.536283183212106 * i,
				-7.556529291062674 + 33.4378624178742 * i,
				-7.908738789857551 - 56.666596888875944 * i,
				-13.715021863561958 - 9.797424875246776 * i,
				7.278140253618929 + 29.567367155533816 * i,
				26.092140808104958 + 23.35860327732442 * i,
				2.4437614781639 + 13.862969935523168 * i,
				-11.156017167013886 + 18.37275278630839 * i,
				23.901839071161973 - 12.902757340991087 * i,
				23.98176050569637 - 14.479322438694219 * i,
				13.583091895992574 + 18.150613583603814 * i,
				-0.8226338558714872 + 23.137024745600968 * i,
				14.855398269475902 - 22.291689039737484 * i,
				-29.816863522323914 - 16.098792984688565 * i,
				40.276925146262435 - 22.15784396539155 * i,
				16.043719239634104 + 34.653092092422106 * i,
				4.296934576622064 + 27.98258474595035 * i,
				-39.65728358303623 - 2.6873374122796037 * i,
				17.024864096128013 - 39.81719364700238 * i,
				-24.70264241822926 - 15.497539530276503 * i,
			],
			[
				-1.8702268875956634 + 46.1722190769777 * i,
				-6.200498495034537 + 4.102225033169834 * i,
				38.84977792025394 - 65.33122633513864 * i,
				47.71893834165549 - 6.10942038474775 * i,
				-41.49947015645955 + 5.858776613390916 * i,
				39.28064530025401 + 72.15691795504114 * i,
				28.280118351108314 - 40.74515000471365 * i,
				6.415607978937317 - 31.17124327223292 * i,
				23.694972482377437 - 7.058927293962476 * i,
				27.78404985739789 + 8.280643924230816 * i,
				-13.078108866928275 + 6.782596390679203 * i,
				3.745495730499158 + 20.94547200293055 * i,
				-28.006786759220333 + 12.53581617007849 * i,
				55.700457058767846 - 28.24615781120235 * i,
				-19.037730649241496 + 52.22138608355539 * i,
				-21.60190556664803 + 8.268751931020887 * i,
				-23.83838930993283 - 17.852226132108 * i,
				28.41034601076263 - 18.879993101438345 * i,
				5.843998338862386 + 17.117324115708968 * i,
				-20.26291094447616 - 19.589421202312835 * i,
				15.972742078615852 - 42.60592657737184 * i,
				-31.65673145300474 + 36.74507412165649 * i,
				-8.173691282134286 + 34.28900564813355 * i,
				-42.77808900588222 + 8.938606808980268 * i,
				15.941816292606717 - 19.58499266680142 * i,
				-8.889707470309682 + 6.983230716657705 * i,
				27.981480535219475 - 26.095366723935086 * i,
				29.132637782062705 - 13.638236809227479 * i,
				46.46141471154652 - 55.79574897835563 * i,
				11.484549158682437 - 33.67437544125386 * i,
				-14.152129134165278 - 27.625805072111714 * i,
				14.808960219930029 - 33.31488237981341 * i,
				-17.4544578446032 - 29.491865245390954 * i,
				29.12825386518096 + 13.208221216005427 * i,
				-15.482821082506753 - 6.395630403306326 * i,
				-56.12125578322483 + 23.07750355261873 * i,
				-1.3825024345136576 + 50.715177498168536 * i,
				20.835642352921 - 66.80657320046146 * i,
				14.170164833675964 + 2.2758334821018513 * i,
				27.230156415190695 - 47.89515865085828 * i,
			],
			[
				-12.500422032878419 + 7.56158595516186 * i,
				-20.705892443889404 + 39.315297828724795 * i,
				22.070978207936072 - 14.746718907391152 * i,
				54.109460091792954 - 40.372172652671054 * i,
				56.244952968557484 - 8.490737329322883 * i,
				-8.766740713367938 - 3.8874429731820044 * i,
				32.34348166507769 - 7.957470596690726 * i,
				-63.02848472066324 - 9.760959662244375 * i,
				-13.874496899818233 + 14.719030459716453 * i,
				10.21458898035172 + 33.523780705966544 * i,
				-69.01153009966714 - 5.213054632088067 * i,
				-0.2695286163505841 - 4.590417091796347 * i,
				10.757323016369995 - 5.177413552831695 * i,
				35.403737802137 - 49.36413893464271 * i,
				-17.817143473430917 - 15.284025742436398 * i,
				-5.8447602750289525 - 32.90812186904779 * i,
				12.487634501626047 - 22.80908426083286 * i,
				-52.0297430297333 + 3.459088297987762 * i,
				-25.888141886584748 - 5.812726937265381 * i,
				68.81170249358767 - 13.716011121893814 * i,
				10.507545744292486 + 12.251723267818821 * i,
				19.24903764689438 - 32.64940690119502 * i,
				11.780920360172601 + 14.508247155514937 * i,
				26.948131996285653 - 24.405753826866874 * i,
				-20.229869681711904 + 10.554435450241868 * i,
				2.3176462380464216 + 11.757175887155896 * i,
				20.668571350559723 - 38.51351028696999 * i,
				-22.968150405007382 - 18.085248585290174 * i,
				35.054581230033065 + 22.512614251286173 * i,
				-10.582534223807073 + 14.624746904916165 * i,
				-48.38997695801433 - 3.5595984849912625 * i,
				0.27707365733952827 - 11.335159765067727 * i,
				38.52158171842528 + 47.01078377585809 * i,
				23.44499848188687 + 9.873323135263075 * i,
				50.494129911856305 + 41.34557746199232 * i,
				-29.726019202985448 - 11.585306957029278 * i,
				5.838140789905715 + 7.7751880831559 * i,
				10.381640056712335 - 8.258734397624586 * i,
				33.14434869702984 - 82.96272828795406 * i,
				31.442513744288647 - 36.271911113817175 * i,
			],
			[
				33.557898059576836 - 16.08637884642054 * i,
				3.824372640993545 - 15.919339996854745 * i,
				-10.614067123290468 - 0.9588921982829106 * i,
				14.404925392752489 + 9.205465335887538 * i,
				17.144028654591864 - 89.28131134669451 * i,
				1.337576195520569 + 22.514233116741224 * i,
				30.81344304769874 - 25.117145039696368 * i,
				-61.9611853680309 + 43.44127684910211 * i,
				-46.66270288489612 - 56.677702969013495 * i,
				-34.56645628165424 - 15.485952803635172 * i,
				-8.84591638259836 + 29.32055728344138 * i,
				-39.083487361726625 + 50.14930201007829 * i,
				-68.11963324551999 + 13.298395497183463 * i,
				-49.40319093380771 + 5.635115635449694 * i,
				-15.673289864958463 - 53.53960897053018 * i,
				3.810088140484993 - 2.1975028391724862 * i,
				-0.10778231534740002 + 58.14998564543903 * i,
				-46.2570503421951 + 15.048757616299227 * i,
				10.189774783303989 - 40.74755953508638 * i,
				27.660985659685366 + 1.8033859466558866 * i,
				-5.454828174245762 - 8.65571998319939 * i,
				-20.738393466828768 - 11.7766623535488 * i,
				-13.048176368968399 - 22.486276615602236 * i,
				-13.903133154299653 - 50.72364297067493 * i,
				-1.9905770983328672 + 12.615478759820292 * i,
				62.28705419543856 + 50.73751754412473 * i,
				-11.720159984143756 + 25.039635976822893 * i,
				1.7557061827068026 + 60.21790783785003 * i,
				-7.160569498558186 + 10.60162011857281 * i,
				20.21250223496106 + 9.968879206244935 * i,
				-13.657663921004254 + 80.2297065855002 * i,
				-38.9887874233047 + 27.285216319845485 * i,
				-0.3634584445620117 + 73.32251558377382 * i,
				-11.920254658473203 + 25.911993884348846 * i,
				10.62803710276512 + 5.2514229511611745 * i,
				-57.832126991159036 + 78.81525023070225 * i,
				-67.42965599685277 - 15.433275288795649 * i,
				-2.689537907772646 - 7.164461552152549 * i,
				41.80282880284754 - 84.43875046178249 * i,
				2.4393221106114247 + 38.20368926447153 * i,
			],
			[
				-16.414652737949783 - 138.50318731912833 * i,
				-71.46147185205 - 2.018899511617647 * i,
				65.78385375877235 + 0.10718138919014564 * i,
				86.8964185922234 + 85.22633661199 * i,
				-68.70308801173853 - 67.56929584368721 * i,
				-32.40266258985616 - 25.2545311264814 * i,
				25.53373032279643 - 22.887279839167547 * i,
				-64.7982908123895 + 26.260330906848115 * i,
				-53.157041830686126 + 29.439473769759218 * i,
				-26.47948704177361 + 20.718120302220186 * i,
				19.784201706880964 - 31.38307134759046 * i,
				-23.72471826864387 + 56.1550197643778 * i,
				-6.075035034896047 + 40.80933574300536 * i,
				-16.303014312427457 + 57.67075278306396 * i,
				-61.32542954791059 - 132.04125398264904 * i,
				66.17145497837006 - 13.799528764460021 * i,
				61.41149576446757 + 22.836226607556014 * i,
				30.06889183956821 - 17.85022088900314 * i,
				0.281444804487748 - 20.96289434626103 * i,
				60.701348358526054 + 73.44646960513137 * i,
				45.65841415721198 + 54.245597464068766 * i,
				-24.172112978860753 - 16.62878098229183 * i,
				-48.81389480276735 + 1.0247490938810664 * i,
				-17.47635553265988 + 3.029434235736378 * i,
				-28.17948783327944 - 1.3975126440752632 * i,
				38.61803691717746 - 5.13928912624465 * i,
				37.09349308921347 + 74.39753624600314 * i,
				63.9909836965859 + 80.35426975689366 * i,
				24.644981818833223 - 15.375197028983706 * i,
				3.3349278109448406 + 16.60575254818285 * i,
				13.363815234142626 + 95.0012061443278 * i,
				14.322687189351159 + 33.330400593220766 * i,
				-21.16022942870942 + 124.23557126044611 * i,
				30.900992089384005 + 61.094717359219544 * i,
				-9.445299822266605 + 59.892951073953654 * i,
				-40.86416156800912 - 19.821353750971518 * i,
				9.57820247112624 - 48.79342034881343 * i,
				70.79446896136918 + 26.83112320832078 * i,
				-9.226397625450666 + 22.84876280788216 * i,
				11.972589886331804 + 117.75814026405848 * i,
			],
			[
				12.332880398312568 - 77.23329397307154 * i,
				-35.20048902362767 - 7.975468713155934 * i,
				28.982778119173688 - 0.4691492433821658 * i,
				-13.140202408574083 + 77.03999797006992 * i,
				-63.00848128126959 - 8.892050704649364 * i,
				-37.577991908878765 + 13.032756735758099 * i,
				12.818704895813848 + 30.510119639002895 * i,
				31.535171954010146 + 14.413853900441097 * i,
				-28.98279901501263 - 71.13171170528831 * i,
				-23.52101138705588 - 52.89467528803388 * i,
				61.76873767720794 - 7.007648514518422 * i,
				-2.3251056012321527 - 7.515782437838311 * i,
				14.140105837095504 + 34.72190962620493 * i,
				-52.88535373955108 - 16.1158387789105 * i,
				3.404488950840175 - 65.49158495799813 * i,
				26.704869560914574 + 43.8406348233317 * i,
				55.090628453778734 + 76.04243194944118 * i,
				42.4144976924026 + 16.561236850801095 * i,
				-10.849422264437464 + 6.210985239954461 * i,
				2.8846793961523503 + 51.33111806283149 * i,
				-63.11192061736465 + 16.93568189705359 * i,
				-26.5476299926189 + 0.93624815223464 * i,
				-5.862818884631508 - 13.080721202623963 * i,
				-19.668120322073438 - 12.274402628090371 * i,
				44.495992603631606 - 39.638366160807465 * i,
				6.361768629743594 - 35.64274311920397 * i,
				-35.31614603823407 + 40.085167559867294 * i,
				81.60790349374119 - 2.3092653857109795 * i,
				-55.75914459973389 + 22.178769402547317 * i,
				-8.307720033021496 + 18.543779538098136 * i,
				-47.653645807126864 + 49.03137307837942 * i,
				-1.5998542998764265 - 4.143784965826061 * i,
				-6.731947140639704 + 24.62150392211116 * i,
				41.41383612337724 + 31.238449953023704 * i,
				0.33773089192362704 + 12.277431035116408 * i,
				13.83822994170826 - 34.94750173922128 * i,
				23.200989869243585 - 22.627536608686118 * i,
				23.659572928625074 + 28.453207872851287 * i,
				-38.7114376237094 + 65.88425889560344 * i,
				-36.89748143613279 + 74.3085049273767 * i,
			],
			[
				16.38812772069653 + 21.641932229845054 * i,
				17.26464344944003 - 14.45242961179919 * i,
				19.35482279211385 - 2.5879720242048165 * i,
				-31.03943018723578 + 18.651901314598973 * i,
				72.89349391863395 - 78.46082086209236 * i,
				35.38983564848445 - 8.007847728443544 * i,
				-2.3264000217228897 - 33.461191701729476 * i,
				-54.5369647780133 + 44.6133969219938 * i,
				41.434622178996634 - 10.819765231029997 * i,
				45.79520276104623 - 4.114191182550712 * i,
				1.8209128620555504 + 38.89624984334574 * i,
				-32.99811363797412 + 5.561930844807662 * i,
				-30.500238686655166 + 4.444237099544097 * i,
				-29.132210648683582 + 11.371655828776838 * i,
				53.80861429297423 - 24.171266848795078 * i,
				19.117343370788376 - 16.921737377475523 * i,
				-79.94128963860697 - 14.668288674913953 * i,
				-26.083427383737018 + 57.84965881423189 * i,
				26.655330461847385 - 44.58586467254656 * i,
				19.305201000262798 - 40.398210216978924 * i,
				29.337757053577576 - 17.98269541869061 * i,
				4.806232177508946 - 41.60267175870774 * i,
				2.723502123874752 - 20.24606222854534 * i,
				-41.474384452617265 - 31.17684546795295 * i,
				-49.824385325271294 + 81.66690323014572 * i,
				34.13753515256142 + 63.73243615294102 * i,
				12.851215366978295 + 33.33269930126448 * i,
				-32.945064821254824 + 68.0437711316502 * i,
				37.64969463790469 + 3.207752502033202 * i,
				-7.1955935084893525 - 9.126414969414064 * i,
				-13.985389492222055 - 25.058883774119636 * i,
				-16.759065472427224 + 34.782738447246004 * i,
				13.97354175619129 - 23.920359557033187 * i,
				-46.76580743144058 + 12.757396020764702 * i,
				-0.589199478905357 - 22.445681808441748 * i,
				-78.11794586417572 + 45.623811610728964 * i,
				21.366343914266523 + 1.1783587331653465 * i,
				29.357056000927535 + 62.904952758368914 * i,
				-27.12205160911221 - 93.88485940763235 * i,
				-8.76803855260007 - 33.219961001206904 * i,
			],
			[
				-28.31328299166799 - 10.451281352668158 * i,
				-46.43189647669655 + 23.710857537841832 * i,
				-56.92164251718333 - 22.03167076412649 * i,
				25.707200458369698 - 10.842040376839151 * i,
				-62.87974636644472 + 101.14097442814389 * i,
				15.929951989217347 - 36.12012099695019 * i,
				-2.7232722248482446 - 55.049115410074165 * i,
				55.71509893360454 - 4.881114007602292 * i,
				-57.7289652581659 + 2.1453070827609935 * i,
				-59.0562955612244 + 1.891845243238464 * i,
				16.9199244616552 - 34.06993143602925 * i,
				39.236326903190104 - 18.635442210424436 * i,
				6.387131710927974 + 12.395216782073348 * i,
				-12.31327097223643 + 27.543985329250646 * i,
				-105.29881432055282 - 7.441312127163258 * i,
				22.219298199413664 - 15.436554707242337 * i,
				110.2549343665041 + 25.139738625790287 * i,
				33.107270192859836 - 27.978414445505386 * i,
				-31.710654741522006 + 13.896187159645773 * i,
				21.699354522494126 + 79.94954727508377 * i,
				-32.09461637699329 + 71.20377696246213 * i,
				17.9108785302557 + 24.36115778711742 * i,
				18.246509898714976 + 10.984528119123993 * i,
				-7.007513718819908 + 60.66465958083507 * i,
				74.36224016711363 - 152.98161672630602 * i,
				-40.025022034675416 - 16.137792768253263 * i,
				5.3999669679351 - 6.922501007783623 * i,
				60.53750690076238 - 99.81585564136489 * i,
				-47.70387077755859 - 25.38697176982151 * i,
				-3.5555913855108656 - 15.271887290774167 * i,
				4.279282931719081 + 39.671502626896 * i,
				5.606656563820079 + 15.787629022297963 * i,
				-27.795414988363945 + 47.09647938877382 * i,
				49.43923855582297 - 31.632614065739336 * i,
				-27.769641721125012 - 4.404067236704872 * i,
				54.92304900736685 + 59.411200957216174 * i,
				28.004667965040547 - 45.91429944273526 * i,
				-23.263631249619067 - 91.00750917130509 * i,
				57.004227177543044 + 82.70215995443832 * i,
				23.173451238476822 + 99.58044727077801 * i,
			],
			[
				-20.818493233100035 + 47.74951876746923 * i,
				84.91572052013882 - 7.357162083150399 * i,
				-95.06982869204035 - 15.467413519133324 * i,
				-44.215545221572654 - 27.67522203694756 * i,
				82.66162234705155 + 8.692012440934263 * i,
				46.578520073783665 + 24.080651768593786 * i,
				-5.445148850158034 - 36.94044022306639 * i,
				5.604792251422907 - 44.57310181148101 * i,
				15.922879170945208 + 10.624215103721035 * i,
				-12.78541747234317 + 16.706426627624737 * i,
				-53.94913094905454 + 39.7115071482894 * i,
				-42.45006432242212 - 53.81463132248712 * i,
				-20.371343049620094 - 49.39675734436645 * i,
				-2.1872464919428865 + 22.668639544270963 * i,
				28.266798095799402 + 74.38473712952546 * i,
				-31.236903648293847 + 34.34991809353957 * i,
				-0.30351258077897114 - 94.28545972461518 * i,
				-38.02341346737646 - 15.786897452086144 * i,
				50.02004525645174 + 47.89304573440441 * i,
				59.345724455608455 - 37.06543804803529 * i,
				22.310987441999536 + 38.083602946458356 * i,
				19.849520939094237 - 2.0097682939245374 * i,
				7.1522323448111536 - 10.507295753939108 * i,
				25.76998109001589 + 0.5362272175010148 * i,
				-95.53204121748084 - 101.8157842754285 * i,
				-32.81688365668617 + 70.06222214115043 * i,
				-28.558972521188895 - 26.81325686250768 * i,
				-98.82346892021235 - 39.75543337585483 * i,
				13.171190476708546 - 28.893764237255 * i,
				25.49967738001596 - 58.924205157403705 * i,
				8.331246502406149 + 19.46489277356153 * i,
				-8.121968378851072 + 15.26479463780797 * i,
				-38.80389268891574 - 39.514721582145846 * i,
				-40.73018055966737 - 49.003149374051276 * i,
				-32.57603208237699 - 76.4023186776447 * i,
				5.1994203744373735 + 51.551251018909525 * i,
				-63.2327133285304 - 37.46752130406753 * i,
				-80.13166935197161 - 22.67666843531579 * i,
				105.81834066212554 - 26.82472556844099 * i,
				69.1871545469955 - 6.508673056163799 * i,
			],
			[
				-9.461506394167182 + 56.94051746836616 * i,
				13.021807983508811 + 2.568384144057312 * i,
				-27.775224179532472 + 26.35656269768312 * i,
				-20.04315091265414 - 75.55608678294621 * i,
				51.42799764243682 - 9.241881866178687 * i,
				42.100002111353376 - 15.510611043084324 * i,
				-62.163167472786256 - 20.51120294879229 * i,
				-35.725886779404334 + 28.89033541400066 * i,
				-36.42263449646275 - 10.139476961578339 * i,
				14.524643023405531 + 42.384971925768184 * i,
				-13.342340643152554 + 20.002788471029767 * i,
				-1.0120065160111693 - 26.48480434565745 * i,
				43.70775189824921 + 32.47260310659765 * i,
				-7.37477284297276 + 4.232547566395773 * i,
				10.412861393773554 + 14.134458879998988 * i,
				36.44809760028653 + 14.327631518640231 * i,
				36.26623425072857 - 19.47598258085339 * i,
				-30.81855790178725 + 39.081442228132715 * i,
				-34.02584054190942 - 4.166035300078448 * i,
				42.25224229704031 - 6.170095701181623 * i,
				-21.30259365113136 + 26.716635723614417 * i,
				16.099023451476114 - 25.529712957258766 * i,
				52.66502617227164 - 38.77355041739767 * i,
				-4.712244928925163 - 1.6884987469416899 * i,
				28.270221918370837 + 45.016041543484874 * i,
				9.711523780699991 + 0.7914741974989044 * i,
				8.463523953674837 + 10.836732676140016 * i,
				12.022551015003643 - 13.325881439273957 * i,
				11.3494264745772 + 61.17055461003904 * i,
				-48.28711534769127 - 43.302481706725956 * i,
				-18.59078845829751 - 50.22779015506764 * i,
				19.116509410163772 + 65.10028005080821 * i,
				82.84843863834872 + 5.104968155945 * i,
				-49.101013461258134 + 3.5388481668593244 * i,
				30.88898780423007 - 30.17478740306833 * i,
				59.67471427066985 + 26.65616728519258 * i,
				4.463923606966169 + 51.61745232978415 * i,
				-51.17634580356744 - 18.64803105804926 * i,
				-0.1503651455193875 - 66.02774805420552 * i,
				-19.167177392555597 - 27.916757411460058 * i,
			],
			[
				-22.63061255853546 + 0.10720500211888861 * i,
				21.88943880240157 + 37.74242958633397 * i,
				1.2032843095363148 - 60.35613102327502 * i,
				-44.15804722469859 - 59.72020426252399 * i,
				56.97328860048714 + 24.661783730646476 * i,
				27.355369176303384 + 10.8668881138303 * i,
				13.114618656569952 - 9.880801274554603 * i,
				-56.16052973447667 - 27.812891332347178 * i,
				14.612820566658925 + 45.760129674412724 * i,
				7.81864507045902 + 5.579452620962044 * i,
				-40.689341572354664 + 6.641242372046165 * i,
				19.749593933804476 - 11.421459090973535 * i,
				39.03017429382567 - 10.522978764488975 * i,
				-29.779191492949366 - 24.655202102583303 * i,
				-27.40047329575792 + 14.212088525162272 * i,
				9.095068612795373 - 19.598852186070296 * i,
				7.448056545404867 - 29.62121811325575 * i,
				-32.18318867020736 + 12.631517257749854 * i,
				13.905833155882078 - 2.171344044015626 * i,
				31.388266975083905 - 32.49759791462229 * i,
				-3.236913745123239 + 8.62670196968675 * i,
				40.24414588756297 - 15.057910092675158 * i,
				44.47051246287276 + 10.592695489474863 * i,
				-1.8142529772531475 - 44.11127254563283 * i,
				-13.90303007726223 + 11.993867418748081 * i,
				-51.38945154001924 - 13.961881915618278 * i,
				-20.06468517260789 - 11.327358664374302 * i,
				-25.81846713082041 - 33.1907528656579 * i,
				-7.628689979641194 + 15.12047299610083 * i,
				0.8656454430023137 + 37.60986624000327 * i,
				-32.15450281337345 - 15.712272455308863 * i,
				-61.7413929969583 - 15.026862786684262 * i,
				62.358066151537855 - 14.652686590221538 * i,
				1.7684131407334611 - 25.630548031440146 * i,
				57.398658838574555 - 3.835385988681523 * i,
				-45.1752646381056 - 9.385306118052515 * i,
				24.510903841575406 - 38.49333522892131 * i,
				-8.937015135185675 + 12.688815239688743 * i,
				16.186412128446207 - 24.783961417680267 * i,
				-16.65425183606867 - 31.59552650097491 * i,
			],
			[
				-65.49679783222982 - 1.8568472793506032 * i,
				-19.031202731927728 - 4.757723366275968 * i,
				63.51876026770962 - 24.96413546925561 * i,
				46.78765190048882 + 32.18853389081437 * i,
				-77.48140639081706 - 27.69389567663682 * i,
				-33.313894533436255 + 1.3472251249176992 * i,
				4.251053843975157 + 4.359191593167168 * i,
				-16.026648524018626 + 52.8301574237686 * i,
				-8.198723536017539 + 38.61340862718583 * i,
				17.33161483203594 - 11.589652252905303 * i,
				21.65792978651413 - 21.461984409097212 * i,
				11.866880768200815 + 15.648055637976626 * i,
				4.223514919255123 + 30.163687662304262 * i,
				26.273395798938168 + 29.80096773039761 * i,
				-88.08255482817428 - 51.430335060047014 * i,
				5.876261968366814 - 22.432635699795497 * i,
				15.52890418224645 + 1.323735972292038 * i,
				0.5808244510900522 - 4.9570404193387105 * i,
				-10.957179638890807 - 8.601263376814515 * i,
				25.641213974225767 - 38.38007707993714 * i,
				36.18524029980506 - 17.195921848112775 * i,
				-19.021097567396865 + 46.00027867788357 * i,
				-76.53978886487377 - 7.071603926345745 * i,
				-7.884643866951947 + 11.03935522876931 * i,
				71.70299925569716 + 75.86448259880177 * i,
				48.366184478009345 - 23.279940583971296 * i,
				58.569476676274206 + 34.404937834382565 * i,
				75.41783006803455 + 30.79890901801963 * i,
				39.58320493193641 + 2.3263525712533024 * i,
				33.10164423638667 - 30.603046192539367 * i,
				73.51058636720066 + 11.56597038139559 * i,
				56.007530771719786 + 6.336557153566412 * i,
				-1.0036814220423054 - 2.817114286259706 * i,
				9.179717663832689 + 19.03919453573988 * i,
				23.66604506865717 - 15.763760874455322 * i,
				-43.882821277914914 - 20.562707693326274 * i,
				16.64047989698316 + 0.06874271719906488 * i,
				72.04879244409908 - 21.767534388794346 * i,
				-41.299404545311006 + 28.796496454211713 * i,
				-11.205049061302553 + 47.921774392714426 * i,
			],
			[
				1.7615021031892866 - 24.7237741275095 * i,
				24.346309065378108 - 37.15602815860399 * i,
				-13.563456869224837 + 39.74772382178125 * i,
				-24.735917451844898 + 3.9240513954506326 * i,
				-71.73640661770386 + 1.230861179235319 * i,
				20.583884928502613 - 20.838412633775455 * i,
				-30.793622495737097 - 57.84342282543567 * i,
				36.40978933979676 + 32.14959855805226 * i,
				-18.684006879357554 + 38.51911628174017 * i,
				9.83029543898482 + 13.681290838877509 * i,
				29.459509134129192 + 19.372036212175203 * i,
				37.34586571156941 + 9.947151764223733 * i,
				-15.171037091762301 + 38.75014078531257 * i,
				2.233808196593145 + 27.8914033773697 * i,
				48.150623333150406 + 7.4877313929322815 * i,
				-20.598155213412447 + 56.18500521739735 * i,
				-5.046589262238243 - 52.847754117358335 * i,
				1.3949545571264537 + 20.487182485449424 * i,
				21.344291113582727 - 24.116012009469394 * i,
				-38.46104180027537 - 1.3357824392702184 * i,
				5.055321394571802 - 3.614298326563471 * i,
				-1.465551215371221 + 41.90726516080039 * i,
				-10.123950764211926 + 10.804141091999853 * i,
				-12.9358228975548 + 43.01522777579308 * i,
				-94.56218992629162 - 27.057697142118542 * i,
				1.102104215656233 + 11.75448783953913 * i,
				-5.422875286593722 + 1.2121579275738696 * i,
				3.493710815798842 + 21.534228574343942 * i,
				-31.53389228815891 - 30.719681733785265 * i,
				41.63144097790006 - 53.69188867840383 * i,
				81.6366380610133 + 21.5968534963058 * i,
				33.11680676323306 + 41.11330820991645 * i,
				-43.196995695146654 - 19.833846341679106 * i,
				-8.636631020964414 - 47.8376687867875 * i,
				-51.69990763987627 - 33.67968172562482 * i,
				94.16593236076133 + 47.52054523010599 * i,
				-15.257517950543576 + 41.85246428126173 * i,
				-31.986191350862896 + 9.02101822926636 * i,
				-48.094299195154065 - 4.9627309424252175 * i,
				48.60088905761257 - 25.256557478778188 * i,
			],
			[
				40.29072488261545 + 33.92860240225368 * i,
				6.1863556099145 + 40.80297872436968 * i,
				5.16038554174466 + 5.762910322743437 * i,
				2.126527612378993 - 98.59300705073113 * i,
				74.44454189122632 + 43.91054025412538 * i,
				28.592325958610985 + 23.60796435535788 * i,
				-18.608223833060435 + 7.490304399971514 * i,
				-14.908113957881795 - 36.03140596336452 * i,
				-2.040982368869919 + 11.116160166929376 * i,
				6.507208773819286 + 6.935327375272017 * i,
				-31.27650122331218 - 20.045473108933933 * i,
				33.42771378589386 - 18.469669433978545 * i,
				70.99977959344504 + 26.00361496413273 * i,
				-19.125647773478505 - 67.69659704662561 * i,
				90.80139453407136 + 71.28907673363088 * i,
				29.96524003353747 + 26.19044437422844 * i,
				3.35787045090035 - 34.704140352274344 * i,
				-31.54932476611946 - 16.32916451560314 * i,
				-10.26511550887949 + 1.849779378953059 * i,
				-49.2667778950551 + 15.115831258736591 * i,
				-60.6765758386363 - 2.509538697295284 * i,
				28.487270551441668 - 27.22312588879754 * i,
				93.2325909864601 + 37.44809011638723 * i,
				6.0887431312373685 + 9.426394068784951 * i,
				-27.64943173212859 + 42.20623149569286 * i,
				-52.618124294424035 - 47.11071162419387 * i,
				-35.727008059586026 - 33.71808454737497 * i,
				-52.632902741170255 - 49.68907441462472 * i,
				-18.61881797518734 + 58.4464255214915 * i,
				-53.08381211363889 + 43.00204282082835 * i,
				-39.857973068297845 - 76.41554079581542 * i,
				-53.663042719481695 - 9.76484418798691 * i,
				84.26552992125795 - 29.774677995816383 * i,
				-14.556207837195167 - 41.02945401376737 * i,
				22.452041984021207 + 4.13536690160954 * i,
				11.835048664053346 - 11.442716223903275 * i,
				18.592469461368093 + 63.550930131221065 * i,
				-41.921486616577006 + 22.242064650423302 * i,
				-16.073918334481792 - 30.37255471226186 * i,
				-22.133400277821494 - 113.69464878085701 * i,
			],
			[
				-61.34229491359579 - 32.94069675585621 * i,
				21.731638139371363 + 39.41570364595369 * i,
				23.33392982361461 - 28.473470855421468 * i,
				54.58011168506856 - 22.16877541260736 * i,
				-25.6758884298346 - 66.35333143232201 * i,
				-19.96335878492861 + 25.155443970633144 * i,
				32.06547158024085 - 39.12369095720999 * i,
				-30.40186978847217 + 34.3778176346155 * i,
				3.95715363759534 + 30.583287157531117 * i,
				-9.846062497249221 + 40.505627786437145 * i,
				-37.749371296436806 + 6.105311731475041 * i,
				0.9788337464211736 + 31.400793926768664 * i,
				-2.937897928575012 + 28.521642099644765 * i,
				-6.69728844189561 + 43.086808693190335 * i,
				-93.43680395030827 - 15.343144372722463 * i,
				-35.32426886303202 - 42.98630717572277 * i,
				27.67928506606127 - 27.515119901007903 * i,
				-11.410782646610016 + 20.12172421361167 * i,
				-12.825050347523952 - 7.505734359525064 * i,
				62.528744046863395 - 55.854376857926724 * i,
				54.43344426194067 - 13.198789926589512 * i,
				-51.467385661757575 - 24.161340677411992 * i,
				-0.798839961300672 + 1.778392422646192 * i,
				-41.99942032797068 - 17.24436331698937 * i,
				-35.09493390448949 + 72.36632419155559 * i,
				31.413172642848455 - 21.41292942251953 * i,
				28.67506799882193 + 20.955279326982343 * i,
				43.094651851813374 + 44.62913488989466 * i,
				33.69099259259363 - 19.840664645243493 * i,
				73.14227629868375 + 3.9107537218389155 * i,
				27.42254492615038 + 50.136456612376946 * i,
				-6.961612521770725 + 34.95587172385608 * i,
				68.60353057720312 + 39.22681249543399 * i,
				8.046362349249737 + 5.876748458108061 * i,
				56.736742218102805 - 2.7861499309049123 * i,
				-35.2391297931044 + 49.61086009165956 * i,
				-60.03193309842085 - 34.17579053867348 * i,
				77.93073664615208 - 27.021479995354028 * i,
				-21.571792122625894 - 26.64749826087911 * i,
				59.35465627694965 + 7.463475901327525 * i,
			],
			[
				-11.50809060105854 - 74.12357128660885 * i,
				-25.296962220219996 - 2.3726157421070297 * i,
				47.61158045710991 + 21.835832577186807 * i,
				39.820266189675635 + 44.29409378808131 * i,
				-119.58036494026564 - 87.76710582487038 * i,
				-34.90688971539623 + 2.6401068402489614 * i,
				24.888105764309017 - 18.560149231756156 * i,
				8.826266590719115 + 73.49534580302989 * i,
				-9.827728268681284 + 2.0731676749570624 * i,
				34.29206760845268 - 7.985985942115434 * i,
				19.2979074838442 + 66.23893326241685 * i,
				9.822853735886383 + 51.86673472708438 * i,
				-44.59014205494799 + 21.527688685515898 * i,
				26.82682623910749 + 12.908684114856698 * i,
				-44.65345573916568 - 47.07766109805891 * i,
				6.490162521328813 + 13.84659288374337 * i,
				-18.164667338030057 + 51.46148201019167 * i,
				36.103508372868816 + 46.8902792705097 * i,
				-4.669682459559425 - 21.736283534250752 * i,
				-5.086914396994587 - 46.58558577347791 * i,
				-11.95569695015709 - 49.005208110384686 * i,
				-87.58602617073359 - 40.92372807661796 * i,
				-25.895060504122107 + 36.49000863364921 * i,
				-43.15850190844931 - 17.701553121786972 * i,
				50.65702908935986 + 58.94638419721778 * i,
				58.94685954198 - 59.37705601935524 * i,
				-7.422269193409804 + 4.824858218001676 * i,
				84.11096630894548 + 69.94739570921766 * i,
				-20.522976199742832 - 51.22686756729845 * i,
				-3.00392260272055 - 6.500905592227895 * i,
				9.48885150222603 + 55.61886322093645 * i,
				14.133515419235763 + 25.582080556078598 * i,
				38.35725530399099 + 40.806292165073735 * i,
				30.43828733269049 + 55.8597427985724 * i,
				3.5736337612659668 - 2.386953475504008 * i,
				-4.047487027433468 + 63.002325730098214 * i,
				-30.96370277405198 + 1.5383125768633263 * i,
				47.987821020381396 + 39.44107932107488 * i,
				-80.11129367572252 + 2.585079055852308 * i,
				26.01380538453138 + 23.408291265994116 * i,
			],
			[
				-57.550050102421665 - 4.869002012460442 * i,
				45.33679991719049 - 19.31293075656874 * i,
				-61.94316938919584 + 4.3506274846531525 * i,
				35.960982667151 - 38.18983864037145 * i,
				-1.9283202177613674 + 30.204119389063578 * i,
				14.517066142694716 - 4.6238019711800575 * i,
				10.276072903725478 - 44.58776088055066 * i,
				32.48713542372366 - 13.095611187673427 * i,
				-34.80920864156717 - 10.69107306405391 * i,
				-48.69452388576575 - 21.39227067865241 * i,
				-7.407667046930555 - 11.192664169439425 * i,
				7.563535409877744 - 13.943888286011555 * i,
				19.055604170156116 + 1.5018862639664121 * i,
				10.700705058249241 + 26.40219490226538 * i,
				-32.281511117325124 - 20.614213347317918 * i,
				-9.741232671254899 + 12.913759790644288 * i,
				58.56235775755262 + 7.552301662968688 * i,
				-27.11506722851849 - 29.1248811797019 * i,
				-28.578108644135284 + 0.7912104729229839 * i,
				35.489602536764636 + 30.811760722888224 * i,
				-49.14347138836839 + 55.93358479600633 * i,
				31.617843736827282 + 7.802807537404448 * i,
				14.091992689732944 + 22.4364784518897 * i,
				32.157088275155644 + 13.181980589024818 * i,
				28.73484945493007 - 70.27964680957551 * i,
				-24.889997256774006 - 35.98476247306483 * i,
				-14.072965689141384 - 21.117959317360288 * i,
				13.5791292354061 - 54.156488433521545 * i,
				-22.622704224192205 + 11.7845317826144 * i,
				29.247082890116616 - 31.038070774553578 * i,
				23.438154636802295 + 22.94881130697774 * i,
				18.118609940331005 + 23.69029048543273 * i,
				-27.479557334100296 + 39.35115251371793 * i,
				14.310438475729097 - 21.023120506329306 * i,
				-6.1290513136575875 - 22.944580798763425 * i,
				85.33068090613602 + 60.372768513243486 * i,
				-51.753871983167286 - 5.037291870166348 * i,
				-4.995493802670531 - 25.267970747246174 * i,
				58.08105129003811 + 5.358991397418642 * i,
				65.07188456423505 - 1.6130370795456976 * i,
			],
			[
				-40.99899632609345 - 23.952166622460823 * i,
				-3.8937554378175196 - 12.52956500101655 * i,
				-4.640865079858212 + 0.6291168356663661 * i,
				9.098682329554322 + 28.63048734411231 * i,
				-59.435390334018564 + 60.58502020021294 * i,
				-37.4820801312606 + 26.669459438067264 * i,
				16.371961911704602 + 2.901322796967179 * i,
				37.55671208462249 - 4.0170587482520395 * i,
				17.135816831746524 - 5.702724870606737 * i,
				-19.413243671525294 - 4.900065083970844 * i,
				-4.549613089146038 - 11.367298728928056 * i,
				23.479573949773453 - 19.390597878540383 * i,
				1.373089980752756 - 29.223947555784715 * i,
				7.884424733287773 + 4.213035920943689 * i,
				-20.899399122468125 + 7.313612298012288 * i,
				-34.705739404773794 + 31.10347171472381 * i,
				-11.768681446037817 + 24.308092578196806 * i,
				-25.440829345526843 + 25.420698707053305 * i,
				2.263263054412119 - 18.863447317655744 * i,
				-0.6245224607154682 - 12.933849721084044 * i,
				-44.33366219755102 - 26.195560242773468 * i,
				-5.294592820901222 + 41.223306070551985 * i,
				-14.772562992209425 + 11.66384300755557 * i,
				53.634332660028754 + 2.5866229198540935 * i,
				7.082960114477686 - 22.4621555082992 * i,
				-8.317087546330526 - 10.001805356511483 * i,
				-9.883932681273151 + 2.494607354244245 * i,
				18.829019785259106 - 2.9992349642114178 * i,
				-39.67302977410907 + 16.204845136954148 * i,
				52.356141337809575 - 44.90540090632044 * i,
				39.375159930393444 + 0.718610864862427 * i,
				13.769563641258891 - 0.362715410309022 * i,
				-18.651335399119834 + 13.001377829413508 * i,
				30.81235829591148 + 12.483957197715668 * i,
				-0.5795176189104438 + 30.273617166533924 * i,
				8.249143258284903 - 10.535310626335963 * i,
				-14.765907173102207 + 0.16389181285924437 * i,
				7.813371784502955 - 20.273886544616772 * i,
				14.890801633301027 + 26.776753963111823 * i,
				28.706483416806222 + 16.785133935275702 * i,
			],
			[
				-30.85966148554803 - 29.920836318623962 * i,
				-17.350580291743654 - 14.349009155319663 * i,
				39.887298340165216 - 3.295282288049947 * i,
				23.656954952456204 + 18.269105783477954 * i,
				-16.79494563013469 - 8.909125723409943 * i,
				-4.3143980008459835 + 30.282894132721268 * i,
				-7.193726098846909 + 22.062299931969932 * i,
				19.787523333676745 - 40.547775595862944 * i,
				-14.839709311099455 + 7.4578556930071045 * i,
				-5.568284189884835 + 20.467973144608074 * i,
				-23.57433915909879 - 7.173817514098644 * i,
				-21.59639564389604 - 5.708832196370324 * i,
				17.507293013979478 - 26.847742915095836 * i,
				51.53170069963929 + 19.307435620908038 * i,
				11.089984546068195 - 7.3925711282657005 * i,
				30.590916911097636 + 6.486224646836007 * i,
				11.138499745659939 - 45.831835535347125 * i,
				4.603092603090914 + 1.8882153085588413 * i,
				19.09021697933625 + 23.91759120405017 * i,
				80.74112332143179 - 27.883671535251153 * i,
				2.344095384983774 + 38.92607796217169 * i,
				-74.59371087197421 + 0.24914934796078114 * i,
				4.842651868553048 - 20.287106591425662 * i,
				21.506483933522276 + 13.459436826535406 * i,
				-45.98454901737034 - 34.984337804515775 * i,
				-24.575446621012347 + 3.3347282100393016 * i,
				25.990752417931333 + 16.92457084090057 * i,
				7.963567081260479 - 14.63591518417158 * i,
				18.681459033268652 + 1.3954769821164135 * i,
				-28.82821696717008 - 14.639268772158424 * i,
				-24.715288733399383 + 14.172286173010392 * i,
				56.28695048309481 + 5.316612557710393 * i,
				39.73828657470548 - 15.752077648989243 * i,
				26.69131590440496 + 63.55237723792064 * i,
				-44.85295367901015 + 18.251218503038807 * i,
				14.79823043153804 - 29.972735521807856 * i,
				-27.189242589448906 + 2.992160958540591 * i,
				13.689906660791463 + 13.502166801911159 * i,
				47.95082452657738 + 14.509572104410772 * i,
				63.64939975241785 - 6.308925452503188 * i,
			],
			[
				-13.636832655092547 + 66.96661548414401 * i,
				15.710878408855727 + 0.719108812189063 * i,
				10.221382837253694 - 26.329395470391887 * i,
				-6.720892988040475 - 52.8720492517555 * i,
				-29.36739174534695 + 75.23565394473363 * i,
				-6.956328983510115 + 8.940551448979704 * i,
				-34.91776943489802 - 7.218969087162122 * i,
				27.473099319602568 - 42.12174874759958 * i,
				20.814772082982927 + 16.07593804261772 * i,
				-5.226804321614971 - 8.558753792480395 * i,
				20.73607776807893 - 40.42646904155305 * i,
				49.0675597324163 + 0.39138635008731626 * i,
				20.460266384253167 + 7.5633613415770995 * i,
				41.26622412057178 - 31.951009056252374 * i,
				-37.56088515845397 + 100.51131949771474 * i,
				-49.36454902532041 - 13.725383573662953 * i,
				13.193709635913269 - 14.674154535740108 * i,
				-7.612596674622949 - 49.89214425946085 * i,
				-3.4743329067482023 + 27.108185008032816 * i,
				-49.662882122157825 - 9.26867597700889 * i,
				15.512059312702284 - 40.505518962201144 * i,
				6.549034758360165 + 41.3143307726952 * i,
				12.808556623578589 + 25.557925661842233 * i,
				-1.0598692251339514 + 29.192495094327967 * i,
				43.00276923886056 - 50.03213551402932 * i,
				-65.5012246825863 - 27.938762354070576 * i,
				5.516419609426272 - 22.9780724160305 * i,
				-18.939131975615823 - 60.0527050785889 * i,
				-14.469245538176693 - 5.78175804469393 * i,
				24.509028416307736 + 13.733403282245149 * i,
				39.56349737671232 - 56.66311656875321 * i,
				-15.215918144565258 - 36.133198411991955 * i,
				11.3269040971646 - 50.82442043789421 * i,
				0.3381219540496403 - 50.23043486533197 * i,
				11.278870361258617 - 17.83703880329092 * i,
				26.927159276798346 - 43.62251481900854 * i,
				15.12048418425836 + 23.43675315777416 * i,
				9.858277696026871 - 72.11599681895993 * i,
				-10.92672903763888 + 53.00150764387699 * i,
				-32.185285217371124 - 40.429036666838364 * i,
			],
			[
				4.158782062488797 - 64.37854064199206 * i,
				-48.50542390135236 + 15.73182613367213 * i,
				72.22086316318163 - 12.734145498393445 * i,
				15.817725368175324 + 104.1078331197341 * i,
				-115.06196395821841 - 74.52794612252214 * i,
				-72.18123952634483 + 11.468304840511507 * i,
				45.959377892317576 + 95.72510423378517 * i,
				29.194053505458783 + 49.40810267271604 * i,
				19.383827624146576 + 9.002707677736417 * i,
				50.16683393451059 - 29.437608671418843 * i,
				56.64919149491635 - 13.471395684299239 * i,
				-7.744096615263544 + 46.012388320191434 * i,
				24.18286624688844 + 3.850365873524808 * i,
				20.177469629547545 + 13.577673454776814 * i,
				-25.04725242182705 - 49.199689061519656 * i,
				-13.768676901595917 - 6.912453185623335 * i,
				-7.922488009656464 + 64.55877521071126 * i,
				96.05752757269602 + 29.362928799714012 * i,
				6.779018449863351 - 2.463166353980057 * i,
				-63.97575572062498 - 60.14547978884191 * i,
				-6.98605212402768 - 58.833081212916596 * i,
				-68.30229336523823 + 17.001295921246687 * i,
				-72.19049511967421 + 10.668887729884744 * i,
				-19.08334951611179 + 1.8270546774787313 * i,
				114.16072573865213 + 66.48844797037329 * i,
				56.99074022681114 - 98.6387956200331 * i,
				46.20260434292837 + 15.278344208625919 * i,
				96.85769784793642 + 80.89876634573253 * i,
				34.746244631719776 - 22.965114227116672 * i,
				-21.328510372777252 + 57.4987865122778 * i,
				-15.7837947486992 - 4.3119404159498735 * i,
				40.638494455211124 - 17.644427595152962 * i,
				5.8592045201312715 - 62.63249058831346 * i,
				67.02222120422873 + 118.10522980760922 * i,
				7.882659604092991 + 24.991512134943722 * i,
				-56.521821945467295 - 49.28910358899675 * i,
				33.37726701047976 - 35.79426820081556 * i,
				74.56517093833698 + 45.586107809132855 * i,
				-118.80258925281726 + 81.01946225102868 * i,
				-30.278339726933375 + 79.23489485087187 * i,
			],
			[
				29.020264278714876 - 14.199742927175102 * i,
				-2.430156324883967 + 23.117421150400077 * i,
				-2.6232689256916615 - 5.580129370248914 * i,
				7.6360946961896055 - 17.07712326206807 * i,
				-6.326121195754411 - 114.46920822083251 * i,
				-9.619597668400154 - 48.68865087342945 * i,
				-22.460543340523557 + 5.5136385425726075 * i,
				-78.16564920409829 + 84.07709500425858 * i,
				49.34903989754312 + 12.016385991786251 * i,
				78.63146712639696 - 18.721332677745423 * i,
				10.652568753289984 + 56.87985119530748 * i,
				-41.9252327699932 + 45.96996568741376 * i,
				24.75785596963197 + 23.632731374222317 * i,
				-6.098211032192193 - 30.009490918381005 * i,
				28.750554807277325 - 43.28823944874304 * i,
				32.763478884421346 - 19.289875330166694 * i,
				-79.8323355899991 - 4.691599426672866 * i,
				28.34091553330328 - 0.8020886309732731 * i,
				19.35745660586204 - 21.1798590738013 * i,
				-40.80465137871802 - 31.20413613069012 * i,
				46.49124887962297 - 46.228290948545485 * i,
				-19.174959869330777 - 53.56762833016502 * i,
				17.157311102207245 + 28.04460209459124 * i,
				-17.262881869748732 + 11.307108368978026 * i,
				-8.575660041923456 + 126.17788204733422 * i,
				50.14383730075204 - 4.084777301450956 * i,
				-4.168951527705104 - 16.906766911642322 * i,
				10.325575633104549 + 90.71572124262512 * i,
				-3.4594332435024526 - 61.83525077447947 * i,
				-66.74434926108822 + 39.44189535495606 * i,
				-38.070220046259706 + 26.43246640789051 * i,
				6.338839751387681 + 3.19971005519956 * i,
				6.922972512095463 + 32.80910049481056 * i,
				-22.01037644984361 + 50.509238396198924 * i,
				-18.958833732673924 + 10.276005903325881 * i,
				-30.891643978002413 + 35.31956533955065 * i,
				9.212918917981337 - 22.269422801901676 * i,
				2.5566393877229254 + 37.595103958950354 * i,
				-67.0685311723968 - 44.019006431020074 * i,
				-24.00785268016788 - 25.36034547703837 * i,
			],
			[
				-13.211416135469996 + 41.951520937157966 * i,
				29.287101673636634 - 9.162628305468985 * i,
				5.108397345052012 - 43.57043028843542 * i,
				-5.719379757654053 - 32.25876860224872 * i,
				35.31355844261981 + 34.85727439321789 * i,
				53.5644965212112 + 54.72812003923711 * i,
				13.892272937320357 - 38.5086132357946 * i,
				22.41596128098773 - 65.01076715828856 * i,
				27.578998281957013 - 55.312355600985825 * i,
				-29.143233640630974 - 2.412811321612992 * i,
				-56.55638117023184 + 15.792954396317537 * i,
				2.927518517416928 - 35.92936463906324 * i,
				-61.745766864030664 - 52.096549246455375 * i,
				6.4154322191591575 + 11.554631665796371 * i,
				-2.209010025447924 - 19.327946979940176 * i,
				-6.926202276578692 + 6.671588917549684 * i,
				-42.953265865881974 + 23.0145825406142 * i,
				-32.23184179691662 - 29.161043334160297 * i,
				-20.35122194222305 - 2.6819436620743815 * i,
				9.591727193768577 + 37.04740251256568 * i,
				-30.72489779095602 + 3.295685212498114 * i,
				-28.46644402098077 + 33.15580231087127 * i,
				23.617282931071145 - 58.10847503252987 * i,
				14.929738152078983 - 2.0126692695147455 * i,
				27.00027265084705 + 37.28939256287889 * i,
				-32.75480210032184 + 78.97106904413369 * i,
				-19.909512246706193 - 14.286797688983107 * i,
				-35.19470430698671 - 44.239228263599614 * i,
				22.729726961380003 + 9.209352310110368 * i,
				54.227754957839736 - 36.836906506258174 * i,
				-41.968744203095994 + 11.906641097530922 * i,
				-21.054350649290292 - 5.35627500744134 * i,
				-21.53831614444494 - 34.79788428767433 * i,
				-29.148945175947887 - 46.29095305917917 * i,
				20.951194059556713 - 17.191483440705753 * i,
				-25.40359822630483 + 33.3207279701501 * i,
				-20.508410861296316 + 35.123345845623 * i,
				31.908848786822723 - 86.29171790195822 * i,
				64.04990903476033 + 7.139766550292478 * i,
				-53.73175092624056 - 68.0266773244694 * i,
			],
			[
				-31.07693246918978 + 22.298947857931793 * i,
				43.503435093929035 - 11.88430785516509 * i,
				13.236065490387233 - 16.162001182145644 * i,
				40.389914091923444 - 24.95630156826025 * i,
				-44.55665999561507 - 1.2890855508178163 * i,
				-5.401783260055595 + 31.247395370004767 * i,
				-7.794962284389607 - 18.939505013934422 * i,
				25.289982351683847 + 40.07997434159475 * i,
				15.731584801680143 + 20.18082952891806 * i,
				36.95965399701333 - 32.71103389245309 * i,
				2.28086742369058 - 12.86079231167895 * i,
				7.331971090941579 + 10.999894779383382 * i,
				22.958205509017617 + 12.948033720947919 * i,
				24.575766746955857 - 11.648277658567586 * i,
				18.12031366910233 + 61.23988669347962 * i,
				-46.929661022049316 + 20.099911012294214 * i,
				-17.96720343708418 - 42.53291307203543 * i,
				-31.996327482300103 + 9.373133016637919 * i,
				27.023860159767242 + 5.113869040537111 * i,
				17.029537291856464 - 82.33720769110722 * i,
				-11.420545522824675 - 67.06582542470075 * i,
				-10.180292317265497 + 20.540783000138703 * i,
				-30.345348986029425 + 56.727488500538804 * i,
				0.8035832364545286 + 20.502414144808128 * i,
				-42.48919428020813 - 23.968097440814873 * i,
				11.253595495328298 - 32.740405358950575 * i,
				26.45500638783109 - 21.9576494910674 * i,
				21.598377134333457 - 0.6405425291244597 * i,
				14.35627243011723 - 39.39628030376109 * i,
				30.920588023011756 - 52.34737650914744 * i,
				34.92881813258915 - 28.572001308435105 * i,
				40.50406759029518 - 6.594263316072144 * i,
				13.064047387769975 - 17.06519931493132 * i,
				21.9944466549845 + 7.573033513537439 * i,
				-0.5986376091863406 - 42.279821798624525 * i,
				-28.346857638440106 + 36.68691508529607 * i,
				-7.398799500691627 + 11.718026533673715 * i,
				24.760291448568754 - 0.7918836640538416 * i,
				-29.95001739590883 - 30.378495238579994 * i,
				90.98472995390726 - 29.10886695441996 * i,
			],
			[
				-40.25963601878951 - 62.64777175089944 * i,
				6.214037286628837 + 25.628326182386644 * i,
				47.23307971549795 - 23.69660799763591 * i,
				88.42929655937778 + 28.782035522328954 * i,
				-17.47282730214239 - 70.97686111921425 * i,
				-24.458949148286678 + 18.503439196107195 * i,
				30.98671950870576 - 4.554370597875746 * i,
				21.71647131852896 + 39.3763636569602 * i,
				0.08430663951475736 + 1.2224286918491316 * i,
				15.778746128625627 - 10.047068987823607 * i,
				-16.86868864966445 - 23.6569545924675 * i,
				1.798134997187674 + 42.55574017812603 * i,
				-24.06535717974631 + 14.493063095263253 * i,
				13.696492214723108 + 64.5146569530664 * i,
				-32.43654496456537 - 67.36607144674804 * i,
				-11.84860501948333 - 48.740846758974385 * i,
				-9.381429296027989 + 4.42157191149488 * i,
				30.73163858464374 - 17.346352565950582 * i,
				-24.72122121765969 - 22.06556757272523 * i,
				51.76602173320938 - 11.176510974756653 * i,
				36.867309752718306 - 37.541800605160816 * i,
				-38.956154647563025 - 29.733316627038207 * i,
				-53.185575536718474 + 6.6771986367810285 * i,
				-41.860363462219894 + 2.9865746808951883 * i,
				-18.710855165599916 + 87.06731802293524 * i,
				71.77694667728251 - 21.126775466922524 * i,
				63.827447837065115 - 15.02670877943666 * i,
				62.58444401756677 + 36.09211161804461 * i,
				49.99727272529665 - 38.86683532936699 * i,
				43.67240875925811 + 37.90266489693023 * i,
				3.491307357293053 + 23.461636436328618 * i,
				15.836838080040557 + 16.04882223161198 * i,
				20.521286972956517 + 8.384317900191432 * i,
				20.820757325911075 + 7.016190024984741 * i,
				42.896549861459526 - 4.974308517481049 * i,
				-65.64027519170588 + 59.888425961409084 * i,
				-46.94690438439747 - 37.13962589387811 * i,
				98.91952407930265 + 12.519509819791068 * i,
				-31.879155388724172 + 9.191463000504049 * i,
				71.62026804691611 + 22.27809178319523 * i,
			],
			[
				-12.04353224674183 - 8.31187930570298 * i,
				-19.358018763263487 - 21.893440110915847 * i,
				8.572414295855168 - 17.050826090022326 * i,
				32.22420645303372 + 2.713588640408304 * i,
				-6.887459524672408 + 1.7126217221695939 * i,
				39.51684899959989 - 10.571546818946272 * i,
				-6.634061729901589 - 18.45511681538524 * i,
				18.27461612242951 - 38.16432925320339 * i,
				-2.161565943400346 - 9.22608177796939 * i,
				-10.756414063240456 - 0.9584084409136722 * i,
				-0.9218215824084837 - 2.4105755073021875 * i,
				19.293858468371525 + 18.189504497999835 * i,
				-33.26498375161637 + 17.18843344694838 * i,
				48.28068231900177 + 2.5251362844498573 * i,
				15.876688933154085 - 16.563617783574337 * i,
				10.071645982925958 - 11.458875522558927 * i,
				-4.373597621973289 - 4.931293504945643 * i,
				30.18989546916572 - 22.66579070674601 * i,
				-11.682224193123801 - 10.207725389472301 * i,
				-1.6463836307005089 - 3.3216391457421537 * i,
				7.8306004389795305 + 19.108676304972462 * i,
				-20.677758981818073 + 9.278389513156107 * i,
				3.124101268135206 + 4.149036275712234 * i,
				-36.335459695662614 + 17.248575047880703 * i,
				-51.937241874384455 + 10.923633349041431 * i,
				-4.628861429076764 + 8.680817703812977 * i,
				21.834641980861676 - 22.835221490431692 * i,
				12.744243220084002 - 26.231317796529027 * i,
				-2.38897215556689 - 38.88707207701086 * i,
				-15.278555991400822 + 9.945012703950283 * i,
				-14.700678875786242 + 11.059241025999093 * i,
				12.340864525292261 - 14.004822334333454 * i,
				-8.044891094671073 - 19.108026457751237 * i,
				34.716113879257115 - 19.730020611071946 * i,
				-21.62478136892574 + 7.993531737025805 * i,
				35.28765006264224 + 27.03943710952273 * i,
				-23.713276920119192 + 26.02563895029239 * i,
				38.70701054111705 - 10.504807061663824 * i,
				-1.466945281101271 + 0.3779290617959692 * i,
				24.139474430708862 - 60.817958570889864 * i,
			],
			[
				51.100228727799376 + 37.22654371536027 * i,
				32.46268087256175 - 32.82422603130783 * i,
				-29.44413268485738 + 15.765319502259104 * i,
				-39.395273433537724 - 25.818110853275662 * i,
				91.1918363550727 - 24.09689836725063 * i,
				83.4549967540554 - 20.34134879703427 * i,
				14.845553062520082 - 17.329688701627873 * i,
				-57.79299873582244 - 61.6190659268681 * i,
				-11.341895639832885 - 29.389275980068188 * i,
				-6.103606129159111 + 2.6732605138042516 * i,
				-30.639674305109082 + 38.30047965862747 * i,
				-32.52405690925754 - 1.147975907681655 * i,
				-60.91464903420922 + 3.3569835263393486 * i,
				0.9465373318364527 - 12.319332957187953 * i,
				74.55664621734081 - 18.56283082736708 * i,
				17.803562763968827 - 23.82546060059542 * i,
				-51.806669208246944 - 30.915651046889167 * i,
				-62.162718932301345 - 32.517877652384755 * i,
				36.98331909299938 - 8.453357560368715 * i,
				25.670062202629836 + 14.042422507005398 * i,
				21.722934057138772 + 12.849989979431058 * i,
				40.15558451876862 + 12.329157892817022 * i,
				-9.139517533292345 - 25.743989625176226 * i,
				-4.468120791789221 - 13.755867478499477 * i,
				-69.01712528118367 + 8.639846913771777 * i,
				10.839156913536948 + 75.26051799617554 * i,
				-37.746419555778104 - 15.850150517077978 * i,
				-68.64741450018217 - 20.527021740042183 * i,
				-15.434262576161444 - 28.137900034870583 * i,
				13.210389026964918 + 4.156278279710207 * i,
				-45.62063377733667 + 31.0877877497326 * i,
				-26.1631431584863 - 12.079306214684529 * i,
				-78.11257536327938 - 2.0945118991163714 * i,
				-48.602415683300734 - 50.252561412350786 * i,
				-36.63139136867553 - 10.753264854693908 * i,
				-4.413610245696162 + 48.86992836326893 * i,
				-9.688690027055344 + 55.27630347193527 * i,
				-49.44780929458748 + 8.589562480583625 * i,
				81.94654000279007 - 97.13615598710763 * i,
				1.9811020394914598 - 53.536238319414494 * i,
			],
			[
				-52.26002756952291 - 12.420745406989852 * i,
				-48.649220026936035 + 7.357476578955215 * i,
				38.787613829805125 + 10.060178851506798 * i,
				66.73096163304089 - 29.035248735396436 * i,
				-61.64298521757553 - 18.69502418563181 * i,
				-29.453498465992812 + 17.273712641503778 * i,
				-23.978289374625334 + 5.426227955442368 * i,
				-22.187325339436626 + 7.616246279667706 * i,
				0.23983491995083472 + 29.039311850170208 * i,
				-23.98539518469839 + 22.208046420756617 * i,
				-21.949181055868166 - 6.377838450298583 * i,
				-3.7265685887609905 + 37.45221650597609 * i,
				-4.143690915425717 + 24.28814333995993 * i,
				31.38920960676393 + 12.194486215746437 * i,
				-79.51417920753015 - 16.50518165970584 * i,
				32.71507304910372 - 11.776944279561476 * i,
				17.17306580323782 - 14.977922134779789 * i,
				6.885898385994769 - 10.98778009512454 * i,
				-8.275369291448726 - 1.9605494472583729 * i,
				19.48762176513239 - 54.32936286437996 * i,
				44.357911062016996 - 3.9032778083657718 * i,
				-44.97315666672529 - 6.184203147927221 * i,
				-6.058516855245614 + 27.83425935490075 * i,
				-2.050191134621514 - 16.210824046071107 * i,
				-60.30984541807075 + 73.2259570501726 * i,
				17.61949960977492 - 35.141160821726814 * i,
				5.115244950064259 + 10.874296478175332 * i,
				55.83189663550353 + 10.363015579614597 * i,
				-24.990503094223847 - 21.874177365821854 * i,
				-10.097267728879046 - 8.658355090659887 * i,
				17.767114705732443 + 39.570479820025625 * i,
				29.577401265203108 - 25.912819592353227 * i,
				57.09539968262975 + 82.31124802224056 * i,
				30.92321422368017 - 3.8563763701881157 * i,
				8.845784787148993 + 37.3323860605405 * i,
				5.002676757558589 - 7.182096583667285 * i,
				-51.886884166861805 + 16.09726666941668 * i,
				47.074640241096716 - 32.6760384906439 * i,
				-16.303505294511297 - 13.412914880672735 * i,
				43.70182473837136 - 32.15973260778437 * i,
			],
		];
		let (m, n) = A.shape();
		let mut QR = A.clone();
		let mut H = Mat::zeros(20, A.ncols());
		qr_in_place(
			QR.rb_mut(),
			H.rb_mut(),
			Par::Seq,
			MemStack::new(&mut MemBuffer::new(qr_in_place_scratch::<c64>(
				A.nrows(),
				A.ncols(),
				20,
				Par::Seq,
				default(),
			))),
			default(),
		);
		let approx_eq = CwiseMat(ApproxEq {
			abs_tol: 1e-10,
			rel_tol: 1e-10,
		});
		let mut Q = Mat::<c64>::zeros(m, m);
		let mut R = QR.as_ref().cloned();
		for j in 0..m {
			Q[(j, j)] = c64::ONE;
		}
		householder::apply_block_householder_sequence_on_the_left_in_place_with_conj(
			QR.rb(),
			H.rb(),
			Conj::No,
			Q.as_mut(),
			Par::Seq,
			MemStack::new(&mut MemBuffer::new(
				householder::apply_block_householder_sequence_transpose_on_the_left_in_place_scratch::<c64>(m, 20, m),
			)),
		);
		for j in 0..n {
			for i in j + 1..m {
				R[(i, j)] = c64::ZERO;
			}
		}
		assert!(Q * R ~ A);
		let QR = A.qr().reconstruct();
		assert!(QR ~ A);
	}
}
