% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find-peaks.R
\name{find_peaks}
\alias{find_peaks}
\alias{find_valleys}
\title{Find local or global maxima (peaks) or minima (valleys)}
\usage{
find_peaks(
  x,
  global.threshold = NULL,
  local.threshold = NULL,
  local.reference = "median",
  threshold.range = NULL,
  span = 3,
  strict = FALSE,
  na.rm = FALSE
)

find_valleys(
  x,
  global.threshold = NULL,
  local.threshold = NULL,
  local.reference = "median",
  threshold.range = NULL,
  span = 3,
  strict = FALSE,
  na.rm = FALSE
)
}
\arguments{
\item{x}{numeric vector.}

\item{global.threshold}{numeric A value belonging to class \code{"AsIs"} is
interpreted as an absolute minimum height or depth expressed in data units.
A bare \code{numeric} value (normally between 0.0 and 1.0), is interpreted
as relative to \code{threshold.range}. In both cases it sets a
\emph{global} height (depth) threshold below which peaks (valleys) are
ignored. A bare negative \code{numeric} value indicates the \emph{global}
height (depth) threshold below which peaks (valleys) are be ignored. If
\code{global.threshold = NULL}, no threshold is applied and all peaks
returned.}

\item{local.threshold}{numeric A value belonging to class \code{"AsIs"} is
interpreted as an absolute minimum height (depth) expressed in data units
relative to a within-window computed reference value. A bare \code{numeric}
value (normally between 0.0 and 1.0), is interpreted as expressed in units
relative to \code{threshold.range}. In both cases \code{local.threshold}
sets a \emph{local} height (depth) threshold below which peaks (valleys)
are ignored. If \code{local.threshold = NULL} or if \code{span} spans the
whole of \code{x}, no threshold is applied.}

\item{local.reference}{character One of \code{"median"}, \code{"median.log"},
\code{"median.sqrt"}, \code{"farthest"}, \code{"farthest.log"} or
\code{"farthest.sqrt"}. The reference used to assess the height of the
peak, is either the minimum/maximum value within the window or the median
of all values in the window.}

\item{threshold.range}{numeric vector If of length 2 or a longer vector
\code{range(threshold.range)} is used to scale both thresholds. With
\code{NULL}, the default, \code{range(x)} is used, and with a vector of
length one \code{range(threshold.range, x)} is used, i.e., the range is
expanded.}

\item{span}{odd positive integer A peak is defined as an element in a
sequence which is greater than all other elements within a moving window of
width \code{span} centred at that element. The default value is 5, meaning
that a peak is taller than its four nearest neighbours. \code{span = NULL}
extends the span to the whole length of \code{x}.}

\item{strict}{logical flag: if \code{TRUE}, an element must be strictly
greater than all other values in its window to be considered a peak.
Default: \code{FALSE} (since version 0.13.1).}

\item{na.rm}{logical indicating whether \code{NA} values should be stripped
before searching for peaks.}
}
\value{
A vector of logical values of the same length as \code{x}. Values
  that are TRUE correspond to local peaks in vector \code{x} and can be used
  to extract the rows corresponding to peaks from a data frame.
}
\description{
These functions find peaks (maxima) and valleys (minima) in a numeric vector,
using a user selectable span and global and local size thresholds, returning
a \code{logical} vector.
}
\details{
As \code{find_valleys}, \code{stat_peaks} and \code{stat_valleys}
  call \code{find_peaks} to search for peaks or valleys, this description
  applies to all four functions.

  Function \code{find_peaks} is a wrapper built onto function
  \code{\link[splus2R]{peaks}} from \pkg{splus2R}, adds support for peak
  height thresholds and handles \code{span = NULL} and non-finite (including
  NA) values differently than \code{splus2R::peaks}. Instead of giving an
  error when \code{na.rm = FALSE} and \code{x} contains \code{NA} values,
  \code{NA} values are replaced with the smallest finite value in \code{x}.
  \code{span = NULL} is treated as a special case and selects \code{max(x)}.
  Passing `strict = TRUE` ensures that multiple global and within window
  maxima are ignored, and can result in no peaks being returned.#'

  Two tests make it possible to ignore irrelevant peaks. One test
  (\code{global.threshold}) is based on the absolute height of the peaks and
  can be used in all cases to ignore globally low peaks. A second test
  (\code{local.threshold}) is available when the window defined by `span`
  does not include all observations and can be used to ignore peaks that are
  not locally prominent. In this second approach the height of each peak is
  compared to a summary computed from other values within the window of width
  equal to \code{span} where it was found. In this second case, the reference
  value used within each window containing a peak is given by
  \code{local.reference}. Parameter \code{threshold.range} determines how the
  bare \code{numeric} values passed as argument to \code{global.threshold}
  and \code{local.threshold} are scaled. The default, \code{NULL} uses the
  range of \code{x}. Thresholds for ignoring too small peaks are applied
  after peaks are searched for, and threshold values can in some cases result
  in no peaks being found. If either threshold is not available (\code{NA})
  the returned value is a \code{NA} vector of the same length as \code{x}.

  The \code{local.threshold} argument is used \emph{as is} when
  \code{local.reference} is \code{"median"} or \code{"farthest"}, i.e., the
  same distance between peak and reference is used as cut-off irrespective of
  the value of the reference. In cases when the prominence of peaks is
  positively correlated with the baseline, a \code{local.threshold} that
  increases together with increasing computed within window median or
  farthest value applies apply a less stringent height requirement in regions
  with overall low height. In this case, natural logarithm or square root
  weighting can be requested with `local.reference` arguments `"median.log"`,
  `"farthest.log"`, `"median.sqrt"`, and `"farthest.sqrt"` as arguments for
  \code{local.reference}.
}
\note{
The default for parameter \code{strict} is \code{FALSE} in functions
  \code{find_peaks()} and \code{find_valleys()}, while it is
  \code{strict = TRUE} in \code{\link[splus2R]{peaks}}.
}
\examples{
# lynx is a time.series object
lynx_num.df <-
  try_tibble(lynx,
             col.names = c("year", "lynx"),
             as.numeric = TRUE) # years -> as numeric

which(find_peaks(lynx_num.df$lynx, span = 5))
which(find_valleys(lynx_num.df$lynx, span = 5))
lynx_num.df[find_peaks(lynx_num.df$lynx, span = 5), ]
lynx_num.df[find_peaks(lynx_num.df$lynx, span = 51), ]
lynx_num.df[find_peaks(lynx_num.df$lynx, span = NULL), ]
lynx_num.df[find_peaks(lynx_num.df$lynx,
                       span = 15,
                       global.threshold = 2/3), ]
lynx_num.df[find_peaks(lynx_num.df$lynx,
                       span = 15,
                       global.threshold = I(4000)), ]
lynx_num.df[find_peaks(lynx_num.df$lynx,
                       span = 15,
                       local.threshold = 0.5), ]

}
\seealso{
\code{\link[splus2R]{peaks}}.

Other peaks and valleys functions: 
\code{\link{find_spikes}()}
}
\concept{peaks and valleys functions}
