/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5O_FRIEND      
#include "H5SMmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      
#include "H5SMpkg.h"     

static void  *H5SM__bt2_crt_context(void *udata);
static herr_t H5SM__bt2_dst_context(void *ctx);
static herr_t H5SM__bt2_store(void *native, const void *udata);
static herr_t H5SM__bt2_debug(FILE *stream, int indent, int fwidth, const void *record, const void *_udata);

const H5B2_class_t H5SM_INDEX[1] = {{
    
    H5B2_SOHM_INDEX_ID,    
    "H5B2_SOHM_INDEX_ID",  
    sizeof(H5SM_sohm_t),   
    H5SM__bt2_crt_context, 
    H5SM__bt2_dst_context, 
    H5SM__bt2_store,       
    H5SM__message_compare, 
    H5SM__message_encode,  
    H5SM__message_decode,  
    H5SM__bt2_debug        
}};

H5FL_DEFINE_STATIC(H5SM_bt2_ctx_t);

static void *
H5SM__bt2_crt_context(void *_f)
{
    H5F_t          *f = (H5F_t *)_f;  
    H5SM_bt2_ctx_t *ctx;              
    void           *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (ctx = H5FL_MALLOC(H5SM_bt2_ctx_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "can't allocate callback context");

    
    ctx->sizeof_addr = H5F_SIZEOF_ADDR(f);

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5SM__bt2_dst_context(void *_ctx)
{
    H5SM_bt2_ctx_t *ctx = (H5SM_bt2_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    ctx = H5FL_FREE(H5SM_bt2_ctx_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__bt2_store(void *native, const void *udata)
{
    const H5SM_mesg_key_t *key = (const H5SM_mesg_key_t *)udata;

    FUNC_ENTER_PACKAGE_NOERR

    
    *(H5SM_sohm_t *)native = key->message;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__bt2_debug(FILE *stream, int indent, int fwidth, const void *record, const void H5_ATTR_UNUSED *_udata)
{
    const H5SM_sohm_t *sohm = (const H5SM_sohm_t *)record;

    FUNC_ENTER_PACKAGE_NOERR

    if (sohm->location == H5SM_IN_HEAP)
        Rfprintf(stream, "%*s%-*s {%" PRIu64 ", %" PRIo32 ", %" PRIxHSIZE "}\n", indent, "", fwidth,
                "Shared Message in heap:", sohm->u.heap_loc.fheap_id.val, sohm->hash,
                sohm->u.heap_loc.ref_count);
    else {
        assert(sohm->location == H5SM_IN_OH);
        Rfprintf(stream, "%*s%-*s {%" PRIuHADDR ", %" PRIo32 ", %x, %" PRIx32 "}\n", indent, "", fwidth,
                "Shared Message in OH:", sohm->u.mesg_loc.oh_addr, sohm->hash, sohm->msg_type_id,
                sohm->u.mesg_loc.index);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
