#' Tests the equality of dependent ICCs
#'
#' @export
#' @param data A data frame with (at least) three columns: the outcome, the subject and the setting identifiers.
#' @param ry Character string. The outcome variable.
#' @param rind Character string. The subject identifier.
#' @param rtype Character string. The setting identifier.
#' @param alpha Numeric. Significance level. Default to 0.05.
#' @param Wald Logical. Should the Wald test be run? Default is FALSE
#' @param WL Vector of length two. Which pair of settings should be compared using the Wald test? The first two settings are the default.
#' @param Boot Logical. Should bootstrap be run? Default is FALSE.
#' @param nboot Numeric. Number of bootstrap resamples. Default is 500.
#' @param Perm Logical. Should permutations test be run? Default value is FALSE.
#' @param nperm Numeric. Number of permutations. Default value is 100.
#' @param parallel Logical. Use parallel computation? Default value is TRUE.
#' @param workers Numeric. Number of cores used in parallelization. Default value is 15.
#' @param future_seed Logical/Integer. The seed to be used for parallellization. Further details in \code{\link[furrr]{furrr_options}}.
#' @param progress Logical. If TRUE a progress bar is created while computing bootstrap and permutations. Default value is TRUE
#' @param future_seed Logical/Integer. The seed to be used for parallellization. Further details in \code{\link[furrr]{furrr_options}}.
#' @return The output is a list with the following components:
#' \itemize{
#'   \item \code{Model}. Object of class lme with the fitted model estimates.
#'   \item \code{Estimates}. List containing three objects: 1) a matrix with the ICCs etimates; 2) the variance-covariance matrix of the ICC estimates; 3) Confidence intervals of the ICC estimates.
#'   \item \code{p_values}. List with the computed p-values testing the equality of the ICCs. 
#'   \item \code{boot_res}. List with three objects: 1) bootstrap estimates of the ICCs; 2) mean of the bootstrap ICC estimates; 3) variance-covariance matrix of the ICC estimates obtained by bootstrapping.
#'   \item \code{perm_res}. List with three elements: 1) P-value obtained by permutations to test the equality of the ICCs; 2) Vector containing the sum of the squared differences between the ICCs obtained in each permutation; 3) the sum of the squared differences between the ICCs obtained in the original sample. 
#' }
#' @details
#' The variance components required for ICC estimation are obtained using a linear mixed-effects model that accounts for correlations across settings.
#' The null hypothesis of equality between dependent ICCs is evaluated through the following methods:
#'  
#'  - Wald test based on Fisher’s Z and Konishi–Gupta transformations, using either asymptotic or bootstrap standard errors;
#'  
#'  - Chi-square test with asymptotic or bootstrap standard errors;
#'  
#'  - Permutation test.
#' @examples
#' sin_res_b<-icc_dep_test(sin_data,ry="Sinuosity",rind="id",rtype="Section",alpha=0.05,Wald=TRUE,
#'WL=1:2,)
#'
#'\donttest{
#' sin_res<-icc_dep_test(sin_data,ry="Sinuosity",rind="id",rtype="Section",alpha=0.05,Wald=TRUE,
#'                      WL=1:2,Boot=TRUE,nboot=500,Perm=TRUE,nperm=100,
#'                      parallel=TRUE,workers=15,future_seed = NULL,progress=TRUE)
#'
#'
#'
#' dia_res<-icc_dep_test(cccrm::bpres,ry="DIA",rind="ID",rtype="METODE",alpha=0.05,Wald=TRUE,
#' WL=1:2,Boot=TRUE,nboot=500,Perm=TRUE,nperm=100,
#' parallel=TRUE,workers=15,future_seed = NULL,progress=TRUE)
#'
#'
#'}
icc_dep_test<-function(data,ry,rind,rtype,alpha=0.05,Wald=FALSE,
                       WL=1:2,Boot=FALSE,nboot=500,Perm=FALSE,nperm=100,
                       parallel=TRUE,workers=15,future_seed = NULL,progress=TRUE){
  
  data<-as.data.frame(data)  
  dades <- data.frame(y=data[,ry],ind=data[,rind],type=data[,rtype])
  dades$type <- as.factor(dades$type)
  dades$type <- factor(dades$type, levels = sort(levels(dades$type)))
  dades$ind <- as.factor(dades$ind)
  
  # Fit the model
  
  model_dif<-fit_model_dep_icc(dades,"y","ind","type")
  
  # Estimates
  out_icc<-icc_est_dep(model_dif)
  
  # Replicates by group
  nr<-nrep(dades)
  
  # Number of subjects
  ns<-nsub(dades)
  
  # Number of types
  nt<-length(levels(dades$type))
  
  
  labs <- as.character(unique(model_dif$data$type))
  
  # Confidence intervals
  CI<-list()
  for (i in (1:nt)){
    CI[[labs[i]]]<-ic_icc(out_icc$ICC[i,i],out_icc$S[i,i],alpha=alpha, 
                          m = nr[i], N=ns)
    
  }
  
  
  out_icc$CI$Asymp <- CI
  
  
  
  # Bootstrap
  CI_boot<-NULL
  boot_icc <- NULL
  if (Boot){
    message("Bootstrapping...")
    boot_icc<-np_boot(dades,"y", "ind","type",nboot=nboot,parallel=parallel,
                      workers=workers,future_seed = future_seed)
    
    
    
    CI_boot<-list()
    
    for (i in (1:nt)){
      CI_boot[[labs[i]]]<-bca(boot_icc$theta[,i],out_icc$ICC[i,i])
      
    }
  }

  
  out_icc$CI$BCA <- CI_boot
  
  
  
  
  # Tests
  
  # Wald test
  
  
  
  if (Wald){
    
    p_WAs <- W_test(out_icc$ICC[WL,WL],out_icc$S[WL,WL])
    p_WZ <- WZ_test(out_icc$ICC[WL,WL],out_icc$S[WL,WL],nr[WL[1]],nr[WL[2]])
    p_WK <- WK_test(out_icc$ICC[WL,WL],out_icc$S[WL,WL],nr[WL[1]],nr[WL[2]])
    
    p_Wald<-matrix(c(p_WAs,p_WZ,p_WK),nrow=1)
    colnames(p_Wald)<-c("Asym.","Z-trans.","KG-trans.")
    
    
    p_Wald_boot<-NULL
    if (!is.null(boot_icc)){
      
      # Wald test amb S obtinguda amb bootstrap
      
      p_WAs_boot <- W_test(out_icc$ICC[WL,WL],boot_icc$S[WL,WL])
      p_WZ_boot <- WZ_test(out_icc$ICC[WL,WL],boot_icc$S[WL,WL],nr[WL[1]],nr[WL[2]])
      p_WK_boot <- WK_test(out_icc$ICC[WL,WL],boot_icc$S[WL,WL],nr[WL[1]],nr[WL[2]])
      
      p_Wald_boot<-matrix(c(p_WAs_boot,p_WZ_boot,p_WK_boot),nrow=1)
      colnames(p_Wald_boot)<-c("Asym.","Z-trans.","KG-trans.")
    }
    
  }
  
  p_values<-list()
  
  p_values$Wald$Asymp<-p_Wald
  p_values$Wald$Boot<-p_Wald_boot
  
  
  # Chi-square
  
  # Original differences
  
  icc<-matrix(diag(out_icc$ICC),ncol=1)
  
  
  k <- length(icc) # vector length
  comb <- combn(k, 2)  # all pairs (i < j)
  
  L <- matrix(0, nrow = ncol(comb), ncol = k)
  for (i in 1:ncol(comb)) {
    L[i, comb[2, i]] <- 1   # positive
    L[i, comb[1, i]] <- -1  # negative
  }
  
  dif<-L %*% icc
  
  # Asymptotic
  
  TT<-t(dif)%*%ginv(L%*%out_icc$S%*%t(L))%*%dif
  
  p_chisq_As<-1-pchisq(TT,nt-1)
  colnames(p_chisq_As)<-"Asym. Chi-Square"
  
  p_chisq_Boot<-NULL
  if (!is.null(boot_icc)){
    # Bootstrap
    
    TT<-t(dif)%*%ginv(L%*%boot_icc$S%*%t(L))%*%dif
    p_chisq_Boot<-1-pchisq(TT,nt-1)
    colnames(p_chisq_Boot)<-"Bootstrap Chi-Square"
    
  } else {
    p_chisq_Boot <- NULL
  }
  
  p_values$chisq$Asymp<-p_chisq_As
  p_values$chisq$Boot<-p_chisq_Boot
  
  if (Perm){
    # Permutations
    message("Running permutations...")
    perm_icc<-perm_test(dades,ry="y", rind="ind",rtype="type",nperm=nperm,
                        parallel=parallel,workers=workers,future_seed=future_seed,
                        progress=progress)
    
    
    
    p_values$Perm<-perm_icc$pval
  } else {
    perm_icc<-NULL
  }
  
  out<-list(Model=model_dif,Estimates=out_icc,p_values=p_values,boot_res=boot_icc,
            perm_res=perm_icc)
  return(out)
  
}

