# general condition handling (any package) -----------

# Try/catch for executing functions that should not disrupt overall flow
# (conceptually similar to purrr::safely but has more detailed conditions summary)
# @param error_value which value to return if an error is caught
# @param catch_errors whether to catch errors (vs. throwing them)
# @param catch_warnigns whether to catch warnings (vs. throwing them)
# @param truncate_call_stack whether to omit the try_catch_cnds calls from the resulting call stack in errors
# @return list with result and conditions, use show_cnds(out$conditions) to show conditions if any were caught
try_catch_cnds <- function(
  expr,
  error_value = NULL,
  catch_errors = TRUE,
  catch_warnings = TRUE,
  truncate_call_stack = TRUE,
  call = caller_call()
) {
  conds <- tibble::tibble(type = character(0), condition = list())

  handle_warning <- function(cnd) {
    # add warning
    conds <<- conds |>
      dplyr::bind_rows(tibble::tibble(type = "warning", condition = list(cnd)))
    cnd_muffle(cnd)
  }

  handle_error <- function(cnd) {
    # truncate call stack to omit the helper function?
    # note: this does more than just rebase the call stack (i.e. trace(base = env) is not the same)
    if (truncate_call_stack && is(cnd$trace, "rlang_trace")) {
      is_helper_start <- cnd$trace$call |>
        sapply(function(x) as.character(x)[1] == "try_catch_cnds")
      is_helper_end <- cnd$trace$call |>
        sapply(function(x) as.character(x)[1] == "withCallingHandlers")
      # found start and end? --> remove helpers from call stack
      if (
        any(is_helper_start) && any(is_helper_end[cumsum(is_helper_start) > 0])
      ) {
        # for each helper (i.e. also works in nested try_catch_cnds calls)
        total_shift <- 0
        for (first_call in which(is_helper_start)) {
          last_call <- first_call +
            which(is_helper_end[seq_along(is_helper_end) > first_call])[1]
          if (last_call > first_call) {
            shift <- last_call - first_call + 1L
            cnd$trace <-
              cnd$trace |>
              # correct parent references
              dplyr::mutate(
                parent = ifelse(
                  .data$parent > last_call - total_shift,
                  .data$parent - shift,
                  .data$parent
                )
              ) |>
              # omit the helper callstack
              dplyr::filter(
                dplyr::row_number() < first_call - total_shift |
                  dplyr::row_number() > last_call - total_shift
              )
            total_shift <- total_shift + shift
          }
        }
      }
    }

    # re-throw error or catch it?
    if (!catch_errors) {
      abort(
        message = NULL,
        parent = cnd,
        class = class(cnd),
        call = call,
        trace = cnd$trace
      )
    } else {
      conds <<- conds |>
        dplyr::bind_rows(tibble::tibble(type = "error", condition = list(cnd)))
    }
    return(error_value)
  }

  # don't catch warnings if not wanted to make sure they get handled as originally intended
  # still keeps the withCallingHandlers call to deal with callstack truncation properly
  if (catch_warnings) {
    result <- tryCatch(
      error = handle_error,
      withCallingHandlers(
        warning = handle_warning,
        expr
      )
    )
  } else {
    result <- tryCatch(error = handle_error, withCallingHandlers(expr))
  }

  # pull out call and message
  conds <- conds |>
    dplyr::mutate(
      call = .data$condition |>
        purrr::map_chr(~ as.character(conditionCall(.x))[1]),
      message = .data$condition |> purrr::map_chr(conditionMessage),
      .before = "condition"
    )

  # return
  return(list(result = result, conditions = conds))
}

# summarize cnds, i.e. how many issues/errors in cli format
# @param conditions conditions data frame generated by try_catch_cnds
# @param message the message to include (if included in summary_format)
# @param include_symbol whether to include the checkmark/warning/error symbol at ther beginning
# @param include_call whether to include the call from which the summarize_cnds was called
# @param call_format how to format the call
# @param summary_format how to format the overall summary
# @param indent whether to wrap with a specific indent in mind
# @return safely cli-formatted single line of text, best printed with cli_bullets
summarize_cnds <- function(
  conditions,
  message = NULL,
  include_symbol = TRUE,
  include_call = TRUE,
  call_format = "in {.strong {call}()}: ",
  summary_format = "{issues} {message}",
  indent = 0,
  .call = caller_call()
) {
  # call
  call <- as.character(.call[1])
  if (is_empty(call)) {
    include_call <- FALSE
  }

  # issues
  issues <- c()
  if (nrow(conditions) == 0) {
    issues <- format_inline("{cli::col_green('no issues')}")
  }
  if ((n <- sum(conditions$type == 'warning')) > 0) {
    issues <- format_inline(
      "{cli::col_yellow(format_inline('{n} warning{?s}'))}"
    )
  }
  if ((n <- sum(conditions$type == 'error')) > 0) {
    issues <- c(
      issues,
      format_inline("{cli::col_red(format_inline('{n} error{?s}'))}")
    )
  }

  # assemble
  summary <- format_inline(summary_format)
  if (include_call) {
    summary <- paste0(format_inline(call_format), summary)
  }
  if (include_symbol) {
    symbol <-
      if (nrow(conditions) == 0L) {
        "v"
      } else if (any(conditions$type == "error")) {
        "x"
      } else {
        "!"
      }
    # use name to support both bullets and abort
    summary <- set_names(summary, symbol)
  }

  # takes care of line breaks and escapes {}
  # it seems cli_escape in format_bullets_raw does NOT properly escape < and >
  # will be fixed in https://github.com/r-lib/cli/issues/789
  # take care of proper line breaks (also covers escaping {})
  summary <-
    withr::with_options(
      list(cli.width = console_width() - indent),
      summary |>
        gsub(pattern = "<", replacement = "<<", fixed = TRUE) |>
        gsub(pattern = ">", replacement = ">>", fixed = TRUE) |>
        format_bullets_raw()
    )
  return(summary)
}

# helper to cli format conditions into a bullet list
# @param conditions conditions data frame generated by try_catch_cnds
# @param include_symbol whether to include the warning/error symbol at ther beggining
# @param include_call whether to include the issuing call (if available)
# @param prefix any text to prefix the each condition line with
# @param call_format how to format the call (only relevant if include_call = TRUE)
# @param indent how many tabs (=2 spaces) to indent each line by?
# @return safely cli-formatted text, best printed with cli_bullets
format_cnds <- function(
  conditions,
  include_symbol = TRUE,
  include_call = TRUE,
  prefix = "",
  call_format = "in {.strong {call}()}: ",
  indent = 0
) {
  if (nrow(conditions) == 0L) {
    return(c())
  }
  out <- conditions |>
    mutate(
      symbol = ifelse(
        .data$type == "error",
        format_inline("{col_red(cli::symbol$cross)} "),
        format_inline("{col_yellow('!')} ")
      ),
      call_label = .data$call |>
        purrr::map_chr(
          ~ {
            if (!is.na(.x)) {
              call <- .x
              format_inline(call_format)
            } else {
              ""
            }
          }
        ),
      # account for multi-line messages
      message_w_type = strsplit(.data$message, "\n", fixed = TRUE) |>
        list(.data$symbol, .data$call_label) |>
        purrr::pmap(
          function(msg, symbol, call) {
            c(
              # add prefix/symbol/call to first line of the message
              paste0(
                !!prefix,
                if (!!include_symbol) symbol,
                if (!!include_call) call,
                utils::head(msg, 1)
              ),
              utils::tail(msg, -1)
            )
          }
        )
    ) |>
    pull(.data$message_w_type) |>
    unlist()

  # take care of proper line breaks (also covers escaping {})
  out <-
    withr::with_options(
      list(cli.width = console_width() - indent * 2),
      out |>
        # it seems cli_escape in format_bullets_raw does NOT properly escape < and >
        # might be fixed if https://github.com/r-lib/cli/issues/789 is addressed
        gsub(pattern = "<", replacement = "<<", fixed = TRUE) |>
        gsub(pattern = ">", replacement = ">>", fixed = TRUE) |>
        format_bullets_raw()
    )

  # indendation
  if (indent > 0) {
    out <-
      # additional indents after first
      paste0(rep("\u00a0", (indent - 1) * 2) |> paste(collapse = ""), out) |>
      # first space via names to support both bullets and aborts
      set_names(" ")
  }

  return(out)
}

# Summarizes and formats cnds
# this is usually not called directly but with show_cnds() or abort_cnds()
# @inheritParams summarize_cnds
# @inheritParams format_cnds
# @param include_cnds whether to show the cnds
# @param collapse_single_line_cnd - collapse cnd onto the summary line if there's only one
# @return empty vector unless there's either a summary message OR at least 1 condition
summarize_and_format_cnds <- function(
  conditions,
  # for summarize_cnds
  include_symbol = TRUE,
  include_summary = TRUE,
  include_call = include_summary,
  summary_format = "{message} encountered {issues}",
  summary_indent = 0,
  message = NULL,
  # for format_cnds
  include_cnds = TRUE,
  include_cnd_calls = TRUE,
  indent_cnds = include_summary,
  collapse_single_line_cnd = TRUE,
  .call = caller_call()
) {
  # safety
  force(.call)
  if (missing(conditions) || !is.data.frame(conditions)) {
    cli_abort("{.var conditions} must be provided as a data frame")
  }

  # summary
  summary_line <- NULL
  if (include_summary) {
    summary_line <- summarize_cnds(
      conditions,
      message = message,
      include_symbol = include_symbol,
      include_call = include_call,
      summary_format = summary_format,
      indent = summary_indent,
      .call = .call,
    )
  }

  # format the cnds
  formatted_cnds <- c()
  if (include_cnds) {
    formatted_cnds <- conditions |>
      format_cnds(
        include_call = include_cnd_calls,
        indent = indent_cnds,
        prefix = if (indent_cnds) {
          format_inline("{cli::symbol$arrow_right} ")
        } else {
          ""
        }
      )

    # if the cnds are a single line long --> combine with summary
    if (
      collapse_single_line_cnd && include_summary && length(formatted_cnds) == 1
    ) {
      formatted_cnds <- c()
      summary_line <-
        c(
          utils::head(summary_line, -1),
          paste(
            utils::tail(summary_line, 1),
            format_inline("{cli::symbol$arrow_right}"),
            # attach to last summary line
            format_cnds(
              conditions,
              include_call = include_cnd_calls,
              include_symbol = FALSE,
              call_format = "{.strong {call}()}: ",
            ) |>
              paste(collapse = " ")
          )
        )
    }
  }
  # output
  return(c(summary_line, formatted_cnds))
}

# Prints out the caught conditions in a cli_bullets list
# @inheritParams summarize_and_format_cnds
# only prints if there are any conditions at all
show_cnds <- function(
  conditions,
  # for summarize_cnds
  include_symbol = TRUE,
  include_summary = TRUE,
  include_call = include_summary,
  summary_format = "{message} encountered {issues}",
  message = NULL,
  # for format_cnds
  include_cnds = TRUE,
  include_cnd_calls = TRUE,
  indent_cnds = include_summary,
  collapse_single_line_cnd = FALSE,
  # call info
  .call = caller_call()
) {
  # output as cli_bullets
  if (nrow(conditions) > 0) {
    output <-
      summarize_and_format_cnds(
        conditions,
        include_symbol = include_symbol,
        include_summary = include_summary,
        include_call = include_call,
        summary_format = summary_format,
        message = message,
        include_cnds = include_cnds,
        include_cnd_calls = include_cnd_calls,
        collapse_single_line_cnd = collapse_single_line_cnd,
        .call = .call
      )
    if (is_interactive()) {
      # use bullets in interactive to work with progress bars
      cli_bullets(output)
    } else {
      # add cli in notebooks to make multiline text more compact (instead of individual paragraphs)
      cli(cli_bullets(output))
    }
  }
}

# Aborts if there are any conditions (for both warnings and errors)
# @inheritParams summarize_and_format_cnds
# only aborts if there are any conditions at all
abort_cnds <- function(
  conditions,
  # for summarize_cnds
  include_symbol = FALSE,
  include_summary = TRUE,
  include_call = FALSE,
  summary_format = "{message} encountered {issues}",
  summary_indent = 5, # seems to work best
  message = NULL,
  # for format_cnds
  include_cnds = TRUE,
  include_cnd_calls = TRUE,
  indent_cnds = include_summary,
  collapse_single_line_cnd = FALSE,
  # call and envinfo
  .call = caller_call(),
  .env = caller_env()
) {
  # throw an error
  if (nrow(conditions) > 0) {
    summarize_and_format_cnds(
      conditions,
      include_symbol = include_symbol,
      include_summary = include_summary,
      include_call = include_call,
      summary_format = summary_format,
      summary_indent = summary_indent,
      message = message,
      include_cnds = include_cnds,
      include_cnd_calls = include_cnd_calls,
      collapse_single_line_cnd = collapse_single_line_cnd,
      .call = .call
    ) |>
      cli_abort(
        call = .call,
        trace = trace_back(bottom = .env)
      )
  }
}
