% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mhp.R
\name{mhp_filter}
\alias{mhp_filter}
\title{Modified Hodrick-Prescott Filter}
\usage{
mhp_filter(x, max_lambda = 100000L, as_dt = TRUE)
}
\arguments{
\item{x}{Numeric vector. The time series to decompose. Must have at least
5 observations and no missing values.}

\item{max_lambda}{Integer. Maximum lambda value to search over. Default is
100000, which covers most macroeconomic applications. The search ranges
from 1 to `max_lambda`.}

\item{as_dt}{Logical. If TRUE (default), returns a data.table. If FALSE,
returns a list with class "mhp".}
}
\value{
If \code{as_dt = TRUE}: A \code{data.table} with columns:
\describe{
  \item{original}{The input series}
  \item{trend}{The estimated trend component}
  \item{cycle}{The cyclical component (original - trend)}
}
With attributes \code{lambda} (optimal lambda) and \code{gcv} (GCV value).

If \code{as_dt = FALSE}: A list with class "mhp" containing elements:
\describe{
  \item{original}{The input series}
  \item{trend}{The estimated trend component}
  \item{cycle}{The cyclical component}
  \item{lambda}{Optimal smoothing parameter}
  \item{gcv}{Generalized cross-validation value}
}
}
\description{
Decomposes a time series into trend and cyclical components using the
Modified HP Filter, which automatically selects the optimal smoothing
parameter lambda via generalized cross-validation (GCV).
}
\details{
The function performs a grid search over lambda values from 1 to `max_lambda`
and selects the lambda that minimizes the GCV criterion. For each lambda,
it solves the system:

\deqn{(I + \lambda K'K)g = y}

where \eqn{K} is the second-difference matrix, \eqn{g} is the trend,
and \eqn{y} is the original series.
}
\examples{
# Simulate a trend + cycle series
set.seed(42)
n <- 100
trend <- cumsum(c(0, rnorm(n - 1, mean = 0.5, sd = 0.2)))
cycle <- 2 * sin(2 * pi * (1:n) / 20) + rnorm(n, sd = 0.5)
y <- trend + cycle

# Apply Modified HP filter
result <- mhp_filter(y, max_lambda = 10000)

# Extract optimal lambda
get_lambda(result)

# Extract GCV value
get_gcv(result)

# Print summary
print(result)

# Plot with ggplot2
if (require(ggplot2)) {
  autoplot(mhp_filter(y, max_lambda = 10000, as_dt = FALSE))
}

}
\references{
Choudhary, M.A., Hanif, M.N., & Iqbal, J. (2014). On smoothing macroeconomic
time series using the modified HP filter. \emph{Applied Economics}, 46(19),
2205-2214.
}
\seealso{
\code{\link{hp_filter}}, \code{\link{autoplot.mhp}}, \code{\link{get_lambda}}, \code{\link{get_gcv}}
}
