#' Run population pharmacokinetic model with pre-defined search space
#'
#' Fits a population PK model using nlmixr2 with configurable search spaces.
#' Supports pre-defined model structures (IV, oral) and custom configurations
#' for advanced modeling scenarios.
#'
#' @param string Numeric vector of parameter values. The length and interpretation
#'   depends on the search.space configuration:
#'   \itemize{
#'     \item "ivbase": 10 values in order: (no.cmpt, eta.km, eta.vc,
#'       eta.vp, eta.vp2, eta.q, eta.q2, mm, mcorr, rv)
#'     \item "oralbase": 11 values in order: (no.cmpt, eta.km, eta.vc,
#'       eta.vp, eta.vp2, eta.q, eta.q2,eta.ka, mm, mcorr, rv)
#'     \item "custom": Length determined by custom_config$params in the order specified
#'   }
#'   The meaning of each element name is defined in \code{ppkmodGen()}.
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param search.space Character string specifying which search space to use.
#'   Options are "ivbase", "oralbase", or "custom". Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param nlmixr2autoinits Logical; if TRUE, use automatic initial estimates
#'   from nlmixr2. Default is TRUE.
#' @param reuse_cache Integer; if 1, attempt to reuse cached results from
#'   previous runs. Default is 1.
#' @param precomputed_results_file Character string; path to a CSV file containing
#'   pre-computed model results for caching
#' @param foldername Character string specifying the folder name for storing
#'   model files and results. If \code{NULL} (default), \code{tempdir()}
#'   is used for temporary storage. If specified, a cache directory
#'   is created in the current working directory.
#' @param filename Character string; base name for output files (without extension).
#'   Required parameter with no default.
#' @param save_fit_rds Logical; if TRUE, save the fitted model object as an
#'   RDS file. Default is FALSE.
#' @param save_csv Logical; if TRUE, save results to a CSV file. Default is
#'   TRUE.
#' @param .modEnv Environment for storing state across multiple model runs. If
#'   NULL, a new environment will be created.
#' @param verbose Logical; if TRUE, print progress messages. Default is
#'   TRUE.
#' @param custom_config List; custom search space configuration for use with
#'   search.space = "custom". Must contain four elements: route,
#'   params, param_dependencies, and fixed_params. See Details
#'   and Examples.
#' @param ... Additional arguments passed to nlmixr2 control functions (e.g.,
#'   saem.control, table.control, max_wall_time)
#'
#' @details
#' This function implements a flexible framework for fitting population PK models
#' with different structural configurations. It uses a configuration-driven approach
#' where the search.space parameter determines how the string vector
#' is interpreted and which model structure is generated.
#'
#' \strong{Search Space Configurations:}
#' The function supports three types of search spaces:
#' \describe{
#'   \item{ivbase}{Pre-defined IV bolus model with 11 parameters. Supports 1-3
#'     compartment models with linear or Michaelis-Menten elimination.}
#'   \item{oralbase}{Pre-defined oral model with 12 parameters (adds eta.ka for
#'     first-order absorption). Same features as ivbase plus absorption kinetics.}
#'   \item{custom}{User-defined model structure requiring custom_config
#'     argument. Allows any combination of parameters supported by \code{ppkmodGen()}.
#'     Supported parameters include: no.cmpt, abs.bio, abs.type, abs.delay,
#'     eta.ka, eta.vc, eta.vp, eta.vp2, eta.q, eta.q2, mm, eta.km, eta.tlag, eta.n,
#'     eta.mtt, eta.bio, eta.D2, eta.F1, eta.Fr, mcorr, rv, and allometric_scaling.
#'     Note: eta.cl and eta.vmax are mutually exclusive and cannot be placed in the
#'     search space simultaneously; NLME models must include either eta.cl (when mm = 0)
#'     or eta.vmax (when mm = 1) to ensure at least one random effect on elimination.
#'     For advanced model parameters not covered by \code{nlmixr2autoinit()},
#'     initial estimates default to 1 before any transformation.
#'     Users can provide custom initial estimates through the param_table argument.}
#' }
#'
#' \strong{Custom Configuration Structure:}
#' When using search.space = "custom", the custom_config argument
#' must be provided as a list with four required elements:
#'
#' \describe{
#'   \item{route}{Character string: "bolus", "oral", or "mixed_iv_oral"}
#'   \item{params}{Character vector of parameter names expected in string,
#'     in the exact order they appear. Length of this vector must match length of
#'     string.}
#'   \item{param_dependencies}{Named list of functions where each function computes
#'     a parameter value based on other parameters. Example: eta.vmax = function(mm) if (mm == 0) 0 else 1.
#'     Use empty list if no dependencies exist.}
#'   \item{fixed_params}{Named list of parameters with fixed values that are NOT
#'     in string. These parameters are automatically passed to the model
#'     generator. Use empty list if no fixed parameters exist.}
#' }
#'
#' \strong{Using fixed_params:}
#' The fixed_params element specifies parameter values that remain constant
#' and do not appear in the string vector. This mechanism serves to:
#' \itemize{
#'   \item Define model structure (e.g., compartment count, absorption type)
#'   \item Fix certain parameters at specific values across all model runs
#'   \item Keep the string vector shorter and focused on variable parameters
#' }
#'
#' Caching System:
#' A two-level caching system avoids re-fitting identical models:
#' \itemize{
#'   \item In-memory cache: Results stored in .modEnv during current session
#'   \item File-based cache: Results loaded from CSV file specified by filename
#' }
#'
#' To enable caching, set reuse_cache = 1 (default) and use consistent
#' filename across runs. Pass the same .modEnv object to subsequent
#' calls to maintain in-memory cache between model evaluations.
#'
#' @return Numeric value representing the fitness score of the fitted model
#'
#' @seealso
#' \code{\link{spaceConfig}} for search space configuration details.
#' \code{\link{parseParams}} for parameter parsing.
#' \code{\link{ppkmodGen}} for model generation.
#' \code{\link{penaltyControl}} for penalty control settings.
#'
#' @import rxode2
#' @import nlmixr2
#' @import nlmixr2data
#' @importFrom rxode2 rxControl
#' @importFrom nlmixr2est nlmixr2 foceiControl nlsControl nlmControl nlminbControl
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#' # Example 1: IV model with pre-defined search space
#' param_table <- initialize_param_table()
#' param_table$init[param_table$Name == "lcl"] <- log(0.008)
#' param_table$init[param_table$Name == "lvc1cmpt"] <- log(0.6)
#' result <- mod.run(
#'   string = c(1, 0, 0, 0, 0, 0, 0, 0, 0, 1),
#'   dat = pheno_sd,
#'   search.space = "ivbase",
#'   param_table = param_table,
#'   saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
#' )
#'
#' # Example 2: Oral model with pre-defined search space
#' param_table <- initialize_param_table()
#' param_table$init[param_table$Name == "lcl"] <- log(2.72)
#' param_table$init[param_table$Name == "lvc1cmpt"] <- log(31.5)
#' result <- mod.run(
#'   string = c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1),
#'   dat = theo_sd,
#'   search.space = "oralbase",
#'   param_table = param_table,
#'   saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
#' )
#'
#' # Example 3: Simplified 1-compartment model with allometric scalling on
#' # Fix no.cmpt=1 and mcorr=0, vary only CL, Vc, and residual error
#' param_table <- initialize_param_table()
#' param_table$init[param_table$Name == "lcl"] <- log(0.008 *((70/3.5)^0.75))
#' param_table$init[param_table$Name == "lvc1cmpt"] <- log(0.6 *((70/3.5)))
#' simple_config <- list(
#'   route = "bolus",
#'   params = c("eta.vc", "mcorr", "rv"),
#'   param_dependencies = list(),
#'   fixed_params = list(
#'     no.cmpt = 1,
#'     eta.cl = 1,
#'     allometric_scaling = 1
#'   )
#' )
#' dat<-pheno_sd
#' dat$LOGWT<-log(dat$WT/70)
#' result <- mod.run(
#'   string = c(1, 1, 1),  # Only 3 values needed
#'   dat = dat,
#'   search.space = "custom",
#'   custom_config = simple_config,
#'   param_table = param_table,
#'   saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
#' )
#' }
#'
#' @export
mod.run <- function(string = NULL,
                    dat = NULL,
                    search.space = "ivbase",
                    no.cores = NULL,
                    penalty.control = penaltyControl(),
                    param_table = NULL,
                    nlmixr2autoinits = TRUE,
                    reuse_cache = 1,
                    precomputed_results_file = NULL,
                    foldername = NULL,
                    filename = "test",
                    save_fit_rds = FALSE,
                    save_csv = TRUE,
                    .modEnv = NULL,
                    verbose = TRUE,
                    custom_config,
                    ...) {

  if (missing(custom_config)){
    custom_config <-NULL
  }
  # Determine which .modEnv to use
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
    # .modEnv <- get(".modEnv", inherits = TRUE)
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  if (is.null(filename) || !nzchar(filename)) {
    stop("`filename` must be a non-empty character string.", call. = FALSE)
  }

  if (search.space == "custom") {
    if (is.null(custom_config)) {
      stop("custom_config must be provided when search.space = 'custom'",
           call. = FALSE)
    }

    required_elements <-
      c("route", "params", "param_dependencies", "fixed_params")
    missing_elements <-
      setdiff(required_elements, names(custom_config))

    if (length(missing_elements) > 0) {
      stop(sprintf(
        "custom_config is missing required elements: %s",
        paste(missing_elements, collapse = ", ")
      ),
      call. = FALSE)
    }

    if (!is.null(custom_config$route) &&
        !custom_config$route %in% c("bolus", "oral", "mixed_iv_oral")) {
      stop(
        sprintf(
          "custom_config$route must be 'bolus', 'oral', or 'mixed_iv_oral', got: '%s'",
          custom_config$route
        ),
        call. = FALSE
      )
    }

    if (is.null(custom_config$params) ||
        length(custom_config$params) == 0) {
      stop("custom_config$params must be a non-empty character vector",
           call. = FALSE)
    }

    if (!is.character(custom_config$params)) {
      stop("custom_config$params must be a character vector",
           call. = FALSE)
    }

    if (!is.list(custom_config$param_dependencies)) {
      stop("custom_config$param_dependencies must be a list",
           call. = FALSE)
    }

    if (!is.list(custom_config$fixed_params)) {
      stop("custom_config$fixed_params must be a list",
           call. = FALSE)
    }
  }

  if (is.null(foldername) || !nzchar(foldername)) {
    # foldername <-
    #   paste0("modRunCache_", filename, "_", digest::digest(dat))
    foldername <- tempdir()
  }
  if (!dir.exists(foldername)) {
    dir.create(foldername,
               showWarnings = FALSE,
               recursive = TRUE)
  }
  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL
  if (is.null(.modEnv$saem.control))
    .modEnv$saem.control <- NULL
  modi <- .modEnv$modi
  r <- .modEnv$r

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }
  # Load precomputed results once
  if (isTRUE(reuse_cache == 1) &&
      !isTRUE(.modEnv$precomputed_cache_loaded)) {
    if (!is.null(precomputed_results_file) &&
        is.character(precomputed_results_file) &&
        nzchar(precomputed_results_file)) {
      if (!file.exists(precomputed_results_file)) {
        stop(
          sprintf(
            "The specified precomputed_results_file does not exist: %s",
            precomputed_results_file
          ),
          call. = FALSE
        )
      }
      csv_path <- precomputed_results_file
    } else if (!is.null(filename) && nzchar(filename)) {
      csv_path <- file.path(foldername, paste0(filename, ".csv"))
    } else {
      csv_path <- NULL
    }

    if (!is.null(csv_path) && file.exists(csv_path)) {
      tmp <- utils::read.csv(csv_path, stringsAsFactors = FALSE)
      if ("current.time" %in% names(tmp)) {
        suppressWarnings({
          tmp$current.time <- as.POSIXct(
            tmp$current.time,
            tryFormats = c(
              "%Y-%m-%d %H:%M:%S",
              "%Y-%m-%d %H:%M",
              "%Y/%m/%d %H:%M:%S",
              "%d/%m/%Y %H:%M",
              "%Y-%m-%d"
            )
          )
        })
      }
      .modEnv$precomputed_results <- tmp
      message(sprintf("Loaded cached model results from '%s'.", csv_path))
    }

    .modEnv$precomputed_cache_loaded <- TRUE
  }

  # Initial estimates
  if (!is.null(param_table)) {
    param_table_use <- param_table
  } else if (!is.null(.modEnv$param_table)) {
    param_table_use <- .modEnv$param_table
  } else {
    param_table_use <- auto_param_table(
      dat = dat,
      nlmixr2autoinits = TRUE,
      foldername = foldername,
      filename = filename,
      out.inits = TRUE
    )
    .modEnv$param_table <- param_table_use
  }

  param_table <- param_table_use

  # Parse search space configuration and decode string to model parameters
  if (search.space == "custom" && !is.null(custom_config)) {
    config <- custom_config
  } else {
    config <- spaceConfig(search.space)
  }

  mod_params <- parseParams(string, config)

  mod_args <-
    c(list(modi = modi, param_table = param_table), mod_params)

  # Generate model file
  mod_file <- file.path(foldername, paste0("mod", modi, ".txt"))
  do.call(ppkmodGen, c(mod_args, list(
    out.dir = foldername, verbose = verbose
  )))
  source(mod_file)

  # Try to reuse in-memory results
  have_cached_fit <- FALSE
  Store. <- NULL

  if (!is.null(.modEnv$Store.all) && nrow(.modEnv$Store.all) > 0) {
    Store.all <- .modEnv$Store.all
    model.code.columns <-
      setdiff(names(mod_args), c("modi", "route", "param_table"))
    cols_present <- intersect(model.code.columns, names(Store.all))

    if (length(cols_present) == length(model.code.columns)) {
      stored_keys <-
        do.call(paste, c(Store.all[model.code.columns], sep = "_"))
      current_key <-
        paste(mod_args[model.code.columns], collapse = "_")
      match.idx <- which(stored_keys == current_key)
      if (length(match.idx) > 0) {
        fit_row <- Store.all[match.idx[1], , drop = FALSE]
        drop_cols <- unique(c(
          model.code.columns,
          "fitness",
          grep("^flag\\.", names(fit_row), value = TRUE),
          grep("^count\\.constraint\\.", names(fit_row), value = TRUE)
        ))
        Store. <-
          fit_row[, setdiff(names(fit_row), drop_cols), drop = FALSE]
        have_cached_fit <- TRUE
      }
    }
  }

  # Otherwise reuse from precomputed CSV
  if (!have_cached_fit &&
      !is.null(.modEnv$precomputed_results) &&
      nrow(.modEnv$precomputed_results) > 0) {
    precomputed_results <- .modEnv$precomputed_results
    model.code.columns <-
      setdiff(names(mod_args), c("modi", "route", "param_table"))
    cols_present <-
      intersect(model.code.columns, names(precomputed_results))
    if (length(cols_present) == length(model.code.columns)) {
      stored_keys <-
        do.call(paste, c(precomputed_results[model.code.columns], sep = "_"))
      current_key <-
        paste(mod_args[model.code.columns], collapse = "_")
      match.idx <- which(stored_keys == current_key)
      if (length(match.idx) > 0) {
        Store. <- precomputed_results[match.idx[1], , drop = FALSE]
        have_cached_fit <- TRUE
      }
    }
  }

  # Fit the model if needed
  if (!have_cached_fit) {
    dot_args <- list(...)
    saem.control <- if (!is.null(dot_args$saem.control)) {
      dot_args$saem.control
    } else {
      nlmixr2est::saemControl(
        seed = 1234,
        print = 1,nBurn = 200,nEm = 300,
        rxControl = nlmixr2est::rxControl(cores = no.cores),
        logLik = TRUE
      )
    }
    table.control <- if (!is.null(dot_args$table.control)) {
      dot_args$table.control
    } else {
      nlmixr2est::tableControl(cwres = TRUE)
    }

    if (is.null(.modEnv$saem.control)){
      niter <- saem.control$mcmc$niter
      nBurn <- niter[1]
      nEm   <- niter[2]
      seed  <- saem.control[["seed"]]
      prt   <- saem.control[["print"]]

      if (verbose) {
        message(sprintf(
          "SAEM control (core) = niter=%s|%s; nBurn=%s; nEm=%s; seed=%s; print=%s",
          niter[1], niter[2], nBurn, nEm, seed, prt
        ))
      }
      .modEnv$saem.control<-saem.control
    }
    max_wall_time <-
      if (!is.null(dot_args$max_wall_time))
        dot_args$max_wall_time
    else
      86400

    result <- run_model_in_subprocess(
      modi = modi,
      dat = dat,
      f = f,
      saem.control = saem.control,
      table.control = table.control,
      max_wall_time = max_wall_time
    )
    fit.s <- result$fit.s
    Store. <- get.mod.lst(fit.s, modi = modi)
    # save rds
    if (isTRUE(save_fit_rds)) {
      rds_file <- file.path(foldername,
                            sprintf("%s_mod%d.rds", filename, modi))
      saveRDS(fit.s, rds_file)
    }
  }

  # Store model parameters from search space configuration (excluding route)
  for (var in setdiff(names(mod_params), "route")) {
    Store.[[var]] <- mod_params[[var]]
  }

  Store2 <- fitness(search.space, Store., dat, penalty.control)
  Store2$model.num <- modi
  Store2$round.num <- r
  Store2$current.time <- Sys.time()

  # Append to Store.all in internal environment
  if (is.null(.modEnv$Store.all)) {
    .modEnv$Store.all <- Store2
  } else {
    .modEnv$Store.all <- rbind(.modEnv$Store.all, Store2)
  }
  rownames(.modEnv$Store.all) <- seq_len(nrow(.modEnv$Store.all))

  if (!is.null(filename) && nzchar(filename)) {
    utils::write.csv(.modEnv$Store.all,
                     file.path(foldername, paste0(filename, ".csv")),
                     row.names = FALSE)
  }

  # Increment modi
  .modEnv$modi <- .modEnv$modi + 1L

  return(Store2$fitness)
}
