#' Perform 1-bit local search
#'
#' Runs a 1-bit neighbourhood local search on a binary-coded model string and
#' returns a data frame of candidate models with their computed fitness (and ranks).
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param start.string Optional numeric/integer vector of 0 or 1 values giving the
#'   starting binary code.
#' @param diff_tol A numeric value specifying the significance difference threshold.
#'   Values within this threshold are considered equal and receive the same rank.
#'   Default is 1.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the folder name for storing
#'   intermediate results. If \code{NULL} (default), \code{tempdir()}
#'   is used for temporary storage. If specified, a cache directory
#'   is created in the current working directory.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional environment used to persist state across calls
#'   (e.g., cached parameter tables and precomputed results). When `NULL`,
#'   a new environment is created.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to \code{mod.run()}.
#'
#' @details
#' For each position in the starting binary code, \code{runlocal()} constructs a
#' candidate by flipping that single bit (a 1-bit flip proposal). Some model
#' components are encoded by linked two-bit schemes (e.g., "no.cmpt1"/"no.cmpt2"
#' and "rv1"/"rv2"); when a proposal targets the second bit of a linked pair,
#' a feasibility rule is applied to maintain a valid encoding.
#'
#' Each candidate is then canonicalised/validated using
#' \code{validStringbinary} before evaluation. Fitness is obtained by
#' calling \code{mod.run} for each candidate and results are ranked using
#' \code{rank_new}.
#'
#' If ".modEnv" is supplied and contains the GA iteration counter ".modEnv$r",
#' local search does not advance this counter; implementations may decrement
#' ".modEnv$r" (with a lower bound of 1) so that local search does not consume
#' a GA "round".
#'
#' @return
#' A data frame where each row corresponds to a unique candidate model. Columns
#' include the binary encoding (one column per bit), the computed "fitness",
#' and the resulting "rank".
#'
#' @seealso
#' \code{\link{mod.run}}, \code{\link{auto_param_table}}, \code{\link{validStringbinary}},
#' \code{\link{penaltyControl}}, \code{\link{rank_new}}
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#' # Example best model binary code
#'   current_code <- c(1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0)
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#' # Run local search
#'   result_local <- runlocal(
#'   dat                      = dat,
#'   search.space             = "ivbase",
#'   start.string             = current_code,
#'   filename                 = "local_search_test",
#'   saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#' )
#' print(result_local)
#' }
#' @export

runlocal <-  function(dat,
                      param_table = NULL,
                      search.space = c("ivbase", "oralbase"),
                      no.cores = NULL,
                      start.string = NULL,
                      diff_tol = 1,
                      penalty.control = penaltyControl(),
                      precomputed_results_file = NULL,
                      foldername = NULL,
                      filename = "test",
                      .modEnv = NULL,
                      verbose = TRUE,
                      ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
    # .modEnv <- get(".modEnv", inherits = TRUE)
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL
  if (is.null(.modEnv$saem.control))
    .modEnv$saem.control <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  if (is.null(foldername) || !nzchar(foldername)) {
    # foldername <-
    #   paste0("gaCache_", filename, "_", digest::digest(dat))
    foldername <- tempdir()
  }
  if (!dir.exists(foldername)) {
    dir.create(foldername,
               showWarnings = FALSE,
               recursive = TRUE)
  }

  # Initial estimates
  if (!is.null(param_table)) {
    param_table_use <- param_table
  } else if (!is.null(.modEnv$param_table)) {
    param_table_use <- .modEnv$param_table
  } else {
    param_table_use <- auto_param_table(
      dat = dat,
      nlmixr2autoinits = TRUE,
      foldername = foldername,
      filename = filename,
      out.inits = TRUE
    )
    .modEnv$param_table <- param_table_use
  }

  param_table <- param_table_use

  search.space <-
    match.arg(search.space, choices = c("ivbase", "oralbase"))
  # GA does not support a custom search space.
  custom_config <- NULL
  if (identical(search.space, "custom")) {
    stop(
      "GA currently does not support search.space = 'custom'. Use 'ivbase' or 'oralbase'.",
      call. = FALSE
    )
  }

  cfg <- spaceConfig(search.space)
  bit.names <- if (identical(search.space, "custom")) {
    custom_config$params
  } else {
    cfg$params
  }
  if (identical(search.space, "custom")) {
    bit.names <- custom_config$params
    params_use <- custom_config$params
    twobit_params <-
      c("no.cmpt",
        "abs.type",
        "abs.delay",
        "rv",
        "allometric_scaling")
  } else {
    params_use <- cfg$params
    twobit_params <- c("no.cmpt", "rv")
  }

  bit.names <- unlist(lapply(params_use, function(p) {
    if (p %in% twobit_params)
      c(paste0(p, "1"), paste0(p, "2"))
    else
      p
  }), use.names = FALSE)

  nbits <- length(bit.names)

  if (!is.null(start.string)) {
    current_code <- start.string
  } else {
    current_code <-
      encodeBinary(base_model(search.space = search.space))
  }

  ls.population <-
    matrix(rep(current_code, nbits), nrow = nbits, byrow = TRUE)

  # Flip bits on diagonal
  diag_indices <- cbind(1:nbits, 1:nbits)
  ls.population[diag_indices] <- 1 - ls.population[diag_indices]

  idx_no_cmpt2 <- which(bit.names == "no.cmpt2")
  idx_no_cmpt1 <- which(bit.names == "no.cmpt1")
  if (length(idx_no_cmpt2) > 0 && current_code[idx_no_cmpt1] == 0) {
    ls.population[idx_no_cmpt2,] <- current_code
    ls.population[idx_no_cmpt2, idx_no_cmpt1] <- 1
    ls.population[idx_no_cmpt2, idx_no_cmpt2] <- 0
  }

  idx_rv2 <- which(bit.names == "rv2")
  idx_rv1 <- which(bit.names == "rv1")
  if (length(idx_rv2) > 0 && current_code[idx_rv1] == 0) {
    ls.population[idx_rv2,] <- current_code
    ls.population[idx_rv2, idx_rv1] <- 1
    ls.population[idx_rv2, idx_rv2] <- 0
  }

  ls.population <- t(apply(ls.population, 1, validStringbinary,
                           search.space = search.space))

  ls.population <- as.data.frame(ls.population)
  ls.population <- unique(ls.population)
  ls.population$fitness <- vapply(seq_len(nrow(ls.population)),
                                  function(k) {
                                    string_vec <- as.vector(ls.population[k, 1:nbits])
                                    result <- try(mod.run(
                                      dat                  = dat,
                                      string               = decodeBinary(string_vec),
                                      search.space         = search.space,
                                      no.cores             = no.cores,
                                      penalty.control =  penalty.control,
                                      param_table          = param_table,
                                      precomputed_results_file = precomputed_results_file,
                                      filename             = filename,
                                      foldername           = foldername,
                                      .modEnv            = .modEnv,
                                      verbose           = verbose,
                                      ...
                                    ),
                                    silent = TRUE)
                                    if (is.numeric(result) &&
                                        length(result) == 1)
                                      result
                                    else
                                      NA_real_
                                  },
                                  numeric(1))
  # Local search does not advance the GA iteration counter
  .modEnv$r <- max(1L, .modEnv$r - 1L)
  # Rank models based on fitness
  ls.population$rank <- rank_new(ls.population$fitness, diff_tol)
  colnames(ls.population)[seq_len(length(bit.names))] <- bit.names
  rownames(ls.population) <- NULL

  return(ls.population)
}
