# version 1.5.5 February 12, 2026 by alexis.dinno@pdx.edu
# perform Horn's parallel analysis for a principal component or common factor 
# analysis, so as to adjust for finite sample bias in the retention of 
# components.

`paran` <-
function(x=NA, 
    iterations=0,
    centile=0,
    quietly=FALSE,
    status=TRUE,
    all=FALSE,
    cfa=FALSE,
    graph=FALSE,
    color=TRUE,
    col=c("black", "red", "blue"),
    lty=c(1,2,3),
    lwd=1,
    legend=TRUE,
    file="",
    width=640,
    height=640,
    grdevice="png",
    seed=0,
    mat=NA,
    n=NA) {

# Confirm that x and mat were NOT both provided
  if ( !is.na(mat[[1]][1]) & !is.na(x[[1]][1]) ) {
    rlang::abort(message="You must supply either x or mat but not both.")
    }

# Set the RNG seed if necessary
  if (seed != 0) {
    set.seed(seed*k)
    }

# Set number of variables
  if ( is.na(mat[[1]][1]) & !is.na(x[[1]][1]) ) {
    P <- length(as.matrix(x)[1,])    
    }
  if ( !is.na(mat[[1]][1]) & is.na(x[[1]][1]) ) {
    P <- length(mat[1,])    
    }

# Confirm correlation matrix, and not covariance matrix
  if (!is.na(mat[[1]][1])) {
    if (length(mat[1,]) != length(mat[,1])) {
      rlang::abort(message="The matrix provided with the mat argument is not a correlation matrix.")
      }
    if ( is.element(FALSE , (diag(mat) == rep(1,P))) ) {
      rlang::abort(message="The matrix provided with the mat argument is not a correlation matrix.\nParallel analysis is not compatible with the eigendecomposition of a covariance matrix.")
      }
    }

# Is message suppression requested?
  muffled <- TRUE %in% grepl("muffleMessage", deparse(sys.call(-1)), fixed = TRUE)

# Is global option rlib_message_verbosity quiet?
  if (is.null(getOption("rlib_message_verbosity")) == FALSE) {
    if (getOption("rlib_message_verbosity") == "quiet") quietly <- TRUE
    }
  if (muffled == TRUE) quietly <- TRUE
  
# Make the correlation matrix R from the data, or take
# from the supplied matrix
  if ( !is.na(x[[1]][1]) ) {
    N <- length(as.matrix(x)[,1])
    if ( !is.na(n) ) {
      rlang::warn(message="The n argument is only for use with the mat argument. Ignoring n argument.")
      }
    R <- cor(x)
    if (quietly == FALSE) rlang::inform(message="Using eigendecomposition of correlation matrix.")
    }
  if ( !is.na(mat[[1]][1]) ) {
    if (!is.na(mat[[1]][1]) & is.na(n)) {
      rlang::abort(message="You must also provide the sample size when using the matrix argument.")
      }
    N <- n
    R <- mat
    if (quietly == FALSE) rlang::inform(message="Using eigendecomposition of provided correlation matrix.")
    }

# quick validation of centile as an integer value
  centile <- round(centile)
  if (centile > 99 | centile < 0) {
    rlang::abort(message="You must specify a centile value between 1 and 99.\n(Specifying centile 0 will use the mean.)")
    }

# Perform pca or cfa
  if (cfa == FALSE) {
    eigenvalues <- eigen(R, only.values = TRUE, EISPACK = FALSE)[[1]]
    }
  if (cfa == TRUE) {
    C <- R - ginv(diag(diag(ginv(R))))
    eigenvalues <- eigen(C, only.values = TRUE, EISPACK = FALSE)[[1]]
    }
  
# Get the eigenvalues .  .  .
  Ev <- eigenvalues

# note which model
  model <- "component"
  models <- "components"
  Model <- "Component"
  Models <- "Components"
  if (cfa == TRUE) {
    model <- "factor"
    models <- "factors   "
    Model <- "Factor   "
    Models <- "Factors"
    }

# clean up iteration and determine value
   if (iterations<1) {
    iterations <- 30*P
    }
   if (iterations<0) {
    rlang::inform(message=paste0("Invalid number of iterations! Using default value of ", iterations, sep=""))
    }

# prepare to save the results of each pca
#    N <- length(as.matrix(x[1]))
    if ( cfa == FALSE ) {
      SimEvs <- matrix(NA,iterations,P)
      }
    if ( cfa == TRUE ) {
      SimEvs <- matrix(NA,iterations,P)
      }

  for (k in 1:iterations) {
  
# Let the user know the program is working if neccesary
    if (status == TRUE & quietly == FALSE) {
      if (k == 1) cat("\nComputing: ")
      progress <- ""
      if (k %% (iterations/10) == 1 & iterations >= 10 & k > 1) {
        pct <- (k%/%(iterations/10))*10
        progress <- paste0(progress, pct,"%  ",sep="")
        cat(progress)
        }
      if (k == iterations) {
        progress <- paste0(progress, "100%\n", sep="")
        cat(progress)
        }
      if (k == iterations) cat("\n")
      }

# initialize previously created random dataset.
    Sim <- matrix(NA,N,P)
      
# Create the random dataset.
    # for normally distributed simulations
    Sim <- matrix(rnorm(N*P),N,P)

# Extract principal components or factors from the random dataset
# (which is the same size and dimension as the user dataset.)

    if (cfa == FALSE) {
      eigenvalues <- eigen(cor(Sim), only.values = TRUE, EISPACK = FALSE)[[1]]        
      }
    if (cfa == TRUE) {
      R <- cor(Sim)
      C <- R - ginv(diag(diag(ginv(R))))
      eigenvalues <- eigen(C, only.values = TRUE, EISPACK = FALSE)[[1]]
      }

# Get the eigenvalues .  .  .
    Evs <- eigenvalues

# Save eigenvalues
    SimEvs[k,] <- Evs

# end the for k loop
  }

# Define the table horizontal border
  table_header_footer <- paste0(paste0(rep("\U2500", times=50), collapse=""), sep="")

# display if neccesary
  if (quietly == FALSE) {
    rlang::inform(message=paste0("\nResults of Horn's Parallel Analysis for ", model, " retention", sep=""))
    if (iterations == 1) {
      mean_message <- "1 iteration, using the mean estimate\n"
      centile_message <- paste0("1 iteration, using the ", centile, " centile estimate\n", sep="")
      if (centile == 0) {
        rlang::inform(message=mean_message)
        }
      if (centile != 0) {
        rlang::inform(message=centile_message)
        }
      }

    if (iterations > 1) {
      mean_message <- paste0(iterations, " iterations, using the mean estimate\n", sep="")
      centile_message <- paste0(iterations, " iterations, using the ", centile, " centile estimate\n", sep="")
      centile_median_message <- paste0(iterations, " iterations, using the ", centile, " centile (median) estimate", "\n", sep="")
      if (centile == 0) {
        rlang::inform(message=mean_message)
        }
      if (centile != 0 & centile != 50) {
        rlang::inform(message=centile_message)
        }
      if (centile == 50) {
        rlang::inform(message=centile_median_message)
        }    
      }
    table_header_footer <- paste0(paste0(rep("\U2500", times=50), collapse=""), sep="")
    rlang::inform(message=table_header_footer)
    rlang::inform(message=paste0(Model, pad.spaces(3), "Adjusted", pad.spaces(4), "Unadjusted", pad.spaces(4), "Estimated", sep=""))
    rlang::inform(message=paste0(pad.spaces(12), "Eigenvalue", pad.spaces(2), "Eigenvalue", pad.spaces(4), "Bias", sep=""))
    rlang::inform(message=table_header_footer)
    }

  RndEv = c(1:P)*NA 

  if (centile > 0) {
    for (p in 1:P) {
      RndEv[[p]] <- quantile(SimEvs[,p],probs=centile/100)[[1]]
      }
    }
  if (centile==0) {
    for (p in 1:P) {
      RndEv[[p]] <- mean(SimEvs[,p])      }
    }

  if (Ev[[1]] < 1 | RndEv[[1]] < 1) { 
    if (quietly == FALSE) {
      rlang::inform(message="No components passed.","\n")
      rlang::inform(message=table_header_footer)
      }
    }

  Bias <- rep(0,P)
  AdjEv <- rep(1,P)
  retained <- P
  for (p in 1:P) {
    if (cfa == TRUE) {
      Bias[p] <- RndEv[p]
      }
    if (cfa == FALSE) {
      Bias[p] <- RndEv[p] - 1
      }
    AdjEv[p] <- Ev[[p]] - Bias[p]
    }

  # calculate how many components or factors to return by counting those 
  # components or factors with adjusted eigenvalues greater than one for
  # PCA, or greater than zero for CFA.
  y <- NA
  for (x in 1:P) {
    y <- x
    if (cfa == TRUE) {
      if (AdjEv[x] <= 0) {
        y <- x - 1
        retained <- y
        break
        }
      }
    if (cfa == FALSE) {
      if (AdjEv[x] <= 1) {
        y <- x - 1
        retained <- y
        break
        }
      }
    }

  if ( all == TRUE ) {
    y <- P
    }

  for (x in 1:y) {
    if ( AdjEv[x] >=0 ) {
      AdjSpace = " "
      }
    if ( AdjEv[x] < 0 ) {
      AdjSpace = ""
      }
    if ( Ev[[x]] >= 0 ) {
      EvSpace = " "
      }
    if ( Ev[[x]] < 0 ) {
      EvSpace = ""
      }
    if ( Bias[x] >= 0 ) {
      BiasSpace = " "
      }
    if ( Bias[x] < 0 ) {
      BiasSpace = ""
      }

# Pad the rear of x in case of single-digits
    if ( x > 9 ) {
      xPad = ""
      }
    if ( x <= 9 ) {
      xPad = " "
      }

# Pad the front of AdjEv in case of eigenvalues > 10, 100, etc.
    AdjFPad = "   "
    if ( round(AdjEv[x]) >= 10 ) {
      AdjFPad = "  "
      }
    if ( round(AdjEv[x]) >= 100 ) {
      AdjFPad <- " "
      }

# Set the strtrim number SN
    SN <- 8
    if ( abs(AdjEv[x]) >= 10 ) {
      SN <- 9
      }
    if ( abs(AdjEv[x]) >= 100 ) {
      SN >= 10
      }
    if ( AdjEv[x] < 0 ) {
      SN <- SN + 1
      }

# Pad the front of Ev in case of eigenvalues > 10, 100, etc.
    EvFPad = "   "
    if ( round(Ev[[x]]) >= 10 ) {
      EvFPad = "  "
      }
    if ( round(Ev[[x]]) >= 100 ) {
      EvFPad = " "
      }

# Set the strtrim number SN
    EvSN <- 8
    if ( abs(Ev[[x]]) >= 10 ) {
      EvSN <- 9
      }
    if ( abs(Ev[[x]]) >= 100 ) {
      EvSN <- 10
      }
    if (abs(Ev[[x]]) >= .0000005) {
      EvZPad <- ""
      }
    if (abs(Ev[[x]]) < .0000005) {
      Ev[[x]] <- 0
      EvZPad <- ".000000"
      }

# Set the strtrim number SN
    BiasSN <- 8
    if ( Bias[x] >= 10 ) {
      BiasSN <- 9
      }
    if ( Bias[x] >= 100 ) {
      BiasSN >= 10
      }

    if (quietly == FALSE) {
      eigenvalues_output_line = paste0(x, xPad, pad.spaces(6), AdjFPad, AdjSpace, strtrim(AdjEv[x], SN), EvFPad, EvSpace, strtrim(Ev[[x]], EvSN), EvZPad, pad.spaces(5), BiasSpace, strtrim(Bias[x], BiasSN), sep="")
      rlang::inform(message=eigenvalues_output_line)
      }
    }
  if (quietly == FALSE) {
    rlang::inform(message=table_header_footer)
    if (cfa == TRUE) {
      retention_message <- paste0("Criterion: retain adjusted eigenvalues > 0.\n(", retained, " ", trimws(if (retained>1) models else model), " retained)\n", sep="")
      }
    if (cfa == FALSE) {
      retention_message <- paste0("Criterion: retain adjusted eigenvalues > 1.\n(", retained, " ", trimws(if (retained>1) models else model), " retained)\n", sep="")
      }
    rlang::inform(message=retention_message)
    }

# Graph it if needed
  if (graph == TRUE) {
    AdjEvCol = col[1]
    EvCol = col[2]
    RndEvCol = col[3]
    AdjEvLty = 1
    EvLty = 1
    RndEvLty = 1
    if (color == FALSE) {
      EvCol = "black"
      RndEvCol = "black"
      EvLty = lty[2]
      RndEvLty = lty[3]
      }
    if (cfa==FALSE) {
      par(yaxs='i', xaxs='i', lab=c(P,ceiling(max(AdjEv[1],Ev[1],RndEv[1])),2))
      plot.default(c(1:P), RndEv, type='o', main='Parallel Analysis', xlab='Components', ylab='Eigenvalues', pch=20, col=RndEvCol, lty=RndEvLty, lwd=lwd, xlim=c(.5,P+.5), ylim=c(min(AdjEv, Ev,RndEv)-.5,ceiling(max(AdjEv[[1]],Ev[[1]],RndEv[[1]]))))
      }
    if (cfa==TRUE) {
      par(xaxp=c(1,P,1))
      plot.default(c(1:P), RndEv, type='o', main='Parallel Analysis', xlab='Factors', ylab='Eigenvalues', pch=20, col=RndEvCol, lty=RndEvLty, lwd=lwd, xlim=c(.5,P+.5), ylim=c(min(AdjEv, Ev,RndEv)-.5,ceiling(max(AdjEv[[1]],Ev[[1]],RndEv[[1]]))))
      }
    if (cfa == TRUE) {
      abline(h=0, col='grey', lwd=.5)
      }
    if (cfa == FALSE) {
      abline(h=1, col='grey', lwd=.5)
      }
    points(c(1:P),AdjEv, type='o', col=AdjEvCol, lty=AdjEvLty, pch=21, bg='white', lwd=lwd)
    points(c(1:P),Ev, type='o', col=EvCol, lty=EvLty, pch=20, lwd=lwd)
    if (retained >= 1) {
      points(c(1:retained), AdjEv[1:retained], type='p', pch=19, col=AdjEvCol, lty=AdjEvLty, lwd=lwd)
      }

# Add a legend to help with interpretation (thanks to Ulrich Keller)
    if (legend==TRUE) {
      legend("topright", legend=c("Adjusted Ev (retained)", "Adjusted Ev (unretained)", "Unadjusted Ev", "Random Ev"), col=c(AdjEvCol, AdjEvCol, EvCol, RndEvCol), pch = c(19, 21, 20, 20), lty = c(AdjEvLty, AdjEvLty, EvLty, RndEvLty))
      }

# Save the graph it if they have requested it be saved using the graphic 
# device they have specified.
    if (file != "" & typeof(file) == "character") {
      dev.copy(device=grdevice, height=height, width=width, file=file)
      dev.off()
      }
    }

  invisible(list(Retained = retained, AdjEv = AdjEv, Ev = Ev, RndEv = RndEv, Bias = Bias, SimEvs = SimEvs))
}

